# API設計書

## 概要

本ドキュメントは、Northwind Traders APIの設計書です。本APIはASP.NET Core WebAPIをベースに構築されており、Categories（カテゴリ）、Customers（顧客）、Employees（従業員）、Products（商品）の4つのリソースに対するCRUD操作を提供します。CQRSパターン（MediatR）を採用し、コマンドとクエリを分離したアーキテクチャで実装されています。

## 共通仕様

### ベースURL

```
https://{host}/api
```

### 認証方式

IdentityServer4を使用したOAuth 2.0 / OpenID Connect認証

- 認証が必要なエンドポイントは `[Authorize]` 属性で保護
- 一部のエンドポイントは `[AllowAnonymous]` 属性により匿名アクセス可能

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Authorization | 条件付き | Bearer Token形式の認証トークン（認証必須エンドポイントのみ） |
| Content-Type | ○ | application/json |
| Accept | - | application/json |

### 共通エラーレスポンス

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - バリデーションエラー / 不正なリクエスト |
| 401 | Unauthorized - 認証エラー |
| 403 | Forbidden - 権限エラー |
| 404 | Not Found - リソース未検出 |
| 500 | Internal Server Error - サーバーエラー |

### エラーレスポンス形式

**バリデーションエラー時**（400）
```json
{
  "fieldName": ["エラーメッセージ1", "エラーメッセージ2"]
}
```

**Not Found時**（404）
```json
{
  "error": "Entity \"EntityName\" (id) was not found."
}
```

**ProblemDetails形式**
```json
{
  "type": "string",
  "title": "string",
  "status": 0,
  "detail": "string",
  "instance": "string"
}
```

## API一覧

| カテゴリ | エンドポイント | メソッド | 認証 | 説明 |
| --- | --- | --- | --- | --- |
| Categories | /api/Categories/GetAll | GET | 不要 | カテゴリ一覧取得 |
| Categories | /api/Categories/Upsert | POST | 必要 | カテゴリ作成/更新 |
| Categories | /api/Categories/Delete/{id} | DELETE | 必要 | カテゴリ削除 |
| Customers | /api/Customers/GetAll | GET | 必要 | 顧客一覧取得 |
| Customers | /api/Customers/Get/{id} | GET | 必要 | 顧客詳細取得 |
| Customers | /api/Customers/Create | POST | 必要 | 顧客作成 |
| Customers | /api/Customers/Update/{id} | PUT | 必要 | 顧客更新 |
| Customers | /api/Customers/Delete/{id} | DELETE | 必要 | 顧客削除 |
| Employees | /api/Employees/GetAll | GET | 不要 | 従業員一覧取得 |
| Employees | /api/Employees/Get/{id} | GET | 不要 | 従業員詳細取得 |
| Employees | /api/Employees/Upsert | POST | 不要 | 従業員作成/更新 |
| Employees | /api/Employees/Delete/{id} | DELETE | 不要 | 従業員削除 |
| Products | /api/Products/GetAll | GET | 不要 | 商品一覧取得 |
| Products | /api/Products/Get/{id} | GET | 不要 | 商品詳細取得 |
| Products | /api/Products/Create | POST | 必要 | 商品作成 |
| Products | /api/Products/Update | PUT | 必要 | 商品更新 |
| Products | /api/Products/Delete/{id} | DELETE | 必要 | 商品削除 |
| Products | /api/Products/Download | GET | 不要 | 商品一覧ファイルダウンロード |

## 各APIエンドポイント定義

### Categories（カテゴリ）

#### 1. カテゴリ一覧取得

カテゴリの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Categories/GetAll` |
| 認証 | 不要（AllowAnonymous） |
| 権限 | なし |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "categories": [
    {
      "id": 1,
      "name": "Beverages",
      "description": "Soft drinks, coffees, teas, beers, and ales",
      "picture": "base64encodedstring"
    },
    {
      "id": 2,
      "name": "Condiments",
      "description": "Sweet and savory sauces, relishes, spreads, and seasonings",
      "picture": null
    }
  ],
  "count": 2
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| categories | array | カテゴリオブジェクトの配列 |
| categories[].id | integer | カテゴリID |
| categories[].name | string | カテゴリ名 |
| categories[].description | string | カテゴリ説明 |
| categories[].picture | string (byte) | カテゴリ画像（Base64） |
| count | integer | カテゴリ総数 |

---

#### 2. カテゴリ作成/更新（Upsert）

カテゴリを新規作成または更新します。IDが指定されている場合は更新、指定されていない場合は新規作成となります。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/Categories/Upsert` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "id": 1,
  "name": "Beverages",
  "description": "Soft drinks, coffees, teas, beers, and ales",
  "picture": "base64encodedstring"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| id | integer | - | - | カテゴリID（更新時のみ指定） |
| name | string | - | - | カテゴリ名 |
| description | string | - | - | カテゴリ説明 |
| picture | string (byte) | - | - | カテゴリ画像（Base64） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
1
```

作成または更新されたカテゴリのIDを返します。

**レスポンス（エラー時）**

ステータスコード: `default`（ProblemDetails形式）

```json
{
  "type": "string",
  "title": "string",
  "status": 0,
  "detail": "string",
  "instance": "string"
}
```

---

#### 3. カテゴリ削除

指定されたIDのカテゴリを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/Categories/Delete/{id}` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 削除対象のカテゴリID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```json
{
  "type": "string",
  "title": "string",
  "status": 404,
  "detail": "string",
  "instance": "string"
}
```

---

### Customers（顧客）

#### 4. 顧客一覧取得

顧客の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Customers/GetAll` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "customers": [
    {
      "id": "ALFKI",
      "name": "Alfreds Futterkiste"
    },
    {
      "id": "ANATR",
      "name": "Ana Trujillo Emparedados y helados"
    }
  ]
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| customers | array | 顧客オブジェクトの配列 |
| customers[].id | string | 顧客ID |
| customers[].name | string | 会社名 |

---

#### 5. 顧客詳細取得

指定されたIDの顧客詳細を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Customers/Get/{id}` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | string | ○ | 顧客ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": "ALFKI",
  "address": "Obere Str. 57",
  "city": "Berlin",
  "companyName": "Alfreds Futterkiste",
  "contactName": "Maria Anders",
  "contactTitle": "Sales Representative",
  "country": "Germany",
  "fax": "030-0076545",
  "phone": "030-0074321",
  "postalCode": "12209",
  "region": null
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| id | string | 顧客ID |
| address | string | 住所 |
| city | string | 市区町村 |
| companyName | string | 会社名 |
| contactName | string | 連絡先担当者名 |
| contactTitle | string | 連絡先担当者役職 |
| country | string | 国 |
| fax | string | FAX番号 |
| phone | string | 電話番号 |
| postalCode | string | 郵便番号 |
| region | string | 地域 |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```json
{
  "type": "string",
  "title": "string",
  "status": 404,
  "detail": "string",
  "instance": "string"
}
```

---

#### 6. 顧客作成

新規顧客を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/Customers/Create` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "id": "NEWID",
  "address": "123 Main Street",
  "city": "Tokyo",
  "companyName": "New Company Ltd.",
  "contactName": "John Doe",
  "contactTitle": "Manager",
  "country": "Japan",
  "fax": "03-1234-5678",
  "phone": "03-1234-5679",
  "postalCode": "1000001",
  "region": "Kanto"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| id | string | ○ | Length(5), NotEmpty | 顧客ID（5文字固定） |
| address | string | - | MaximumLength(60) | 住所 |
| city | string | - | MaximumLength(15) | 市区町村 |
| companyName | string | ○ | MaximumLength(40), NotEmpty | 会社名 |
| contactName | string | - | MaximumLength(30) | 連絡先担当者名 |
| contactTitle | string | - | MaximumLength(30) | 連絡先担当者役職 |
| country | string | - | MaximumLength(15) | 国 |
| fax | string | - | MaximumLength(24) | FAX番号 |
| phone | string | - | MaximumLength(24) | 電話番号 |
| postalCode | string | ○ | MaximumLength(10), NotEmpty | 郵便番号 |
| region | string | - | MaximumLength(15) | 地域 |

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `default`（ProblemDetails形式）

---

#### 7. 顧客更新

既存顧客の情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/Customers/Update/{id}` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | string | ○ | 更新対象の顧客ID |

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "id": "ALFKI",
  "address": "Updated Address",
  "city": "Berlin",
  "companyName": "Alfreds Futterkiste Updated",
  "contactName": "Maria Anders",
  "contactTitle": "Sales Manager",
  "country": "Germany",
  "fax": "030-0076545",
  "phone": "030-0074321",
  "postalCode": "12209",
  "region": null
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| id | string | ○ | MaximumLength(5), NotEmpty | 顧客ID |
| address | string | - | MaximumLength(60) | 住所 |
| city | string | - | MaximumLength(15) | 市区町村 |
| companyName | string | ○ | MaximumLength(40), NotEmpty | 会社名 |
| contactName | string | - | MaximumLength(30) | 連絡先担当者名 |
| contactTitle | string | - | MaximumLength(30) | 連絡先担当者役職 |
| country | string | - | MaximumLength(15) | 国 |
| fax | string | ○ | MaximumLength(24), NotEmpty | FAX番号 |
| phone | string | ○ | MaximumLength(24), NotEmpty | 電話番号 |
| postalCode | string | - | MaximumLength(10) | 郵便番号 |
| region | string | - | MaximumLength(15) | 地域 |

**追加バリデーションルール**

- オーストラリア国の場合：郵便番号は4桁の数字
- オーストラリア・QLD州（郵便番号が4で始まる）の場合：電話番号またはFAX番号が07で始まること

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

---

#### 8. 顧客削除

指定されたIDの顧客を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/Customers/Delete/{id}` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | string | ○ | 削除対象の顧客ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

---

### Employees（従業員）

#### 9. 従業員一覧取得

従業員の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Employees/GetAll` |
| 認証 | 不要 |
| 権限 | なし |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "name": "Davolio, Nancy",
    "position": "Sales Representative",
    "extension": "5467"
  },
  {
    "id": 2,
    "name": "Fuller, Andrew",
    "position": "Vice President, Sales",
    "extension": "3457"
  }
]
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| [].id | integer | 従業員ID |
| [].name | string | 従業員名（姓, 名形式） |
| [].position | string | 役職 |
| [].extension | string | 内線番号 |

---

#### 10. 従業員詳細取得

指定されたIDの従業員詳細を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Employees/Get/{id}` |
| 認証 | 不要 |
| 権限 | なし |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 従業員ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "title": "Ms.",
  "firstName": "Nancy",
  "lastName": "Davolio",
  "birthDate": "1968-12-08T00:00:00",
  "address": "507 - 20th Ave. E. Apt. 2A",
  "city": "Seattle",
  "region": "WA",
  "postalCode": "98122",
  "country": "USA",
  "homePhone": "(206) 555-9857",
  "position": "Sales Representative",
  "extension": "5467",
  "hireDate": "1992-05-01T00:00:00",
  "notes": "Education includes a BA in psychology...",
  "photo": "base64encodedstring",
  "managerId": 2,
  "territories": [
    {
      "territoryId": "06897",
      "territory": "Wilton",
      "region": "Eastern"
    }
  ]
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| id | integer | 従業員ID |
| title | string | 敬称 |
| firstName | string | 名 |
| lastName | string | 姓 |
| birthDate | string (date-time) | 生年月日 |
| address | string | 住所 |
| city | string | 市区町村 |
| region | string | 地域 |
| postalCode | string | 郵便番号 |
| country | string | 国 |
| homePhone | string | 自宅電話番号 |
| position | string | 役職 |
| extension | string | 内線番号 |
| hireDate | string (date-time) | 入社日 |
| notes | string | 備考 |
| photo | string (byte) | 写真（Base64） |
| managerId | integer | 上司ID |
| territories | array | 担当地域の配列 |
| territories[].territoryId | string | 地域ID |
| territories[].territory | string | 地域名 |
| territories[].region | string | リージョン名 |

---

#### 11. 従業員作成/更新（Upsert）

従業員を新規作成または更新します。IDが指定されている場合は更新、指定されていない場合は新規作成となります。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/Employees/Upsert` |
| 認証 | 不要 |
| 権限 | なし |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "id": null,
  "title": "Mr.",
  "firstName": "John",
  "lastName": "Smith",
  "birthDate": "1990-01-15T00:00:00",
  "address": "123 Main Street",
  "city": "New York",
  "region": "NY",
  "postalCode": "10001",
  "country": "USA",
  "homePhone": "(212) 555-1234",
  "position": "Sales Representative",
  "extension": "1234",
  "hireDate": "2023-01-01T00:00:00",
  "notes": "New employee",
  "photo": "base64encodedstring",
  "managerId": 2
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| id | integer | - | - | 従業員ID（更新時のみ指定） |
| title | string | - | - | 敬称 |
| firstName | string | - | - | 名 |
| lastName | string | - | - | 姓 |
| birthDate | string (date-time) | - | - | 生年月日 |
| address | string | - | - | 住所 |
| city | string | - | - | 市区町村 |
| region | string | - | - | 地域 |
| postalCode | string | - | - | 郵便番号 |
| country | string | - | - | 国 |
| homePhone | string | - | - | 自宅電話番号 |
| position | string | - | - | 役職 |
| extension | string | - | - | 内線番号 |
| hireDate | string (date-time) | - | - | 入社日 |
| notes | string | - | - | 備考 |
| photo | string (byte) | - | - | 写真（Base64） |
| managerId | integer | - | - | 上司ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
1
```

作成または更新された従業員のIDを返します。

**レスポンス（エラー時）**

ステータスコード: `default`（ProblemDetails形式）

---

#### 12. 従業員削除

指定されたIDの従業員を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/Employees/Delete/{id}` |
| 認証 | 不要 |
| 権限 | なし |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 削除対象の従業員ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

**注意事項**

- 従業員は自分自身のアカウントを削除できません（400 Bad Request）
- 関連するユーザーアカウントも同時に削除されます
- 関連するテリトリーや注文がある場合は削除に失敗する可能性があります

---

### Products（商品）

#### 13. 商品一覧取得

商品の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Products/GetAll` |
| 認証 | 不要（AllowAnonymous） |
| 権限 | なし |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "products": [
    {
      "productId": 1,
      "productName": "Chai",
      "unitPrice": 18.00,
      "supplierId": 1,
      "supplierCompanyName": "Exotic Liquids",
      "categoryId": 1,
      "categoryName": "Beverages",
      "discontinued": false
    },
    {
      "productId": 2,
      "productName": "Chang",
      "unitPrice": 19.00,
      "supplierId": 1,
      "supplierCompanyName": "Exotic Liquids",
      "categoryId": 1,
      "categoryName": "Beverages",
      "discontinued": false
    }
  ],
  "createEnabled": true
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| products | array | 商品オブジェクトの配列 |
| products[].productId | integer | 商品ID |
| products[].productName | string | 商品名 |
| products[].unitPrice | decimal | 単価 |
| products[].supplierId | integer | サプライヤーID |
| products[].supplierCompanyName | string | サプライヤー会社名 |
| products[].categoryId | integer | カテゴリID |
| products[].categoryName | string | カテゴリ名 |
| products[].discontinued | boolean | 販売終了フラグ |
| createEnabled | boolean | 作成権限フラグ |

---

#### 14. 商品詳細取得

指定されたIDの商品詳細を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Products/Get/{id}` |
| 認証 | 不要（AllowAnonymous） |
| 権限 | なし |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 商品ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "productId": 1,
  "productName": "Chai",
  "unitPrice": 18.00,
  "supplierId": 1,
  "supplierCompanyName": "Exotic Liquids",
  "categoryId": 1,
  "categoryName": "Beverages",
  "discontinued": false,
  "editEnabled": true,
  "deleteEnabled": true
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| productId | integer | 商品ID |
| productName | string | 商品名 |
| unitPrice | decimal | 単価 |
| supplierId | integer | サプライヤーID |
| supplierCompanyName | string | サプライヤー会社名 |
| categoryId | integer | カテゴリID |
| categoryName | string | カテゴリ名 |
| discontinued | boolean | 販売終了フラグ |
| editEnabled | boolean | 編集権限フラグ |
| deleteEnabled | boolean | 削除権限フラグ |

---

#### 15. 商品作成

新規商品を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/Products/Create` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "productName": "New Product",
  "unitPrice": 25.00,
  "supplierId": 1,
  "categoryId": 1,
  "discontinued": false
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| productName | string | - | - | 商品名 |
| unitPrice | decimal | - | - | 単価 |
| supplierId | integer | - | - | サプライヤーID |
| categoryId | integer | - | - | カテゴリID |
| discontinued | boolean | - | - | 販売終了フラグ |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
77
```

作成された商品のIDを返します。

---

#### 16. 商品更新

既存商品の情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/Products/Update` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

```json
{
  "productId": 1,
  "productName": "Updated Product Name",
  "unitPrice": 30.00,
  "supplierId": 2,
  "categoryId": 2,
  "discontinued": true
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| productId | integer | ○ | - | 商品ID |
| productName | string | - | - | 商品名 |
| unitPrice | decimal | - | - | 単価 |
| supplierId | integer | - | - | サプライヤーID |
| categoryId | integer | - | - | カテゴリID |
| discontinued | boolean | - | - | 販売終了フラグ |

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `default`（ProblemDetails形式）

---

#### 17. 商品削除

指定されたIDの商品を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/Products/Delete/{id}` |
| 認証 | 必要 |
| 権限 | 認証済みユーザー |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 削除対象の商品ID |

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `default`（ProblemDetails形式）

---

#### 18. 商品一覧ファイルダウンロード

商品一覧をファイル形式でダウンロードします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/Products/Download` |
| 認証 | 不要（AllowAnonymous） |
| 権限 | なし |

**パスパラメータ**

なし

**クエリパラメータ**

なし

**リクエストボディ**

なし

**レスポンス（成功時）**

ステータスコード: `200 OK`

Content-Type: `application/octet-stream`

バイナリファイル形式で商品一覧データを返します。

## 備考

### アーキテクチャ

本APIはClean ArchitectureとCQRS（Command Query Responsibility Segregation）パターンを採用しています。

- **MediatR**: コマンドとクエリの処理を仲介
- **FluentValidation**: リクエストのバリデーション処理
- **AutoMapper**: エンティティとDTOのマッピング

### OpenAPI/Swagger

Swagger UIは `/api` エンドポイントで利用可能です。OpenAPI仕様書は `/api/specification.json` で取得できます。

### ルーティング規則

ベースコントローラーで定義されたルーティングパターン: `api/[controller]/[action]`

### ヘルスチェック

`/health` エンドポイントでアプリケーションとデータベースの健全性を確認できます。
