# インフラ設計書

## 1. 概要

### 1.1 文書の目的

本文書は、Northwind Tradersアプリケーションのインフラストラクチャ設計について記述する。本システムは、.NET Core 3.0をベースとしたClean Architectureで構築されたWebアプリケーションである。

### 1.2 システム概要

| 項目 | 内容 |
|------|------|
| システム名 | Northwind Traders |
| アプリケーションタイプ | Webアプリケーション（SPA + API） |
| バックエンドフレームワーク | ASP.NET Core 3.0 |
| フロントエンドフレームワーク | Angular 8.2.5 |
| データベース | SQL Server（LocalDB / Azure SQL） |
| 認証方式 | IdentityServer4 + ASP.NET Core Identity |

### 1.3 対象読者

- インフラエンジニア
- DevOpsエンジニア
- システム管理者
- 開発チームリーダー

---

## 2. システムアーキテクチャ

### 2.1 論理アーキテクチャ

本システムはClean Architectureに基づいた4層構造を採用している。

| レイヤー | プロジェクト | 役割 |
|----------|-------------|------|
| Presentation | WebUI | Webインターフェース、APIエンドポイント |
| Infrastructure | Infrastructure, Persistence | 外部サービス連携、データベースアクセス |
| Application | Application | ビジネスロジック、CQRS実装 |
| Domain | Domain | エンティティ、ドメインロジック |
| Common | Common | 共通インターフェース |

### 2.2 プロジェクト依存関係

```
WebUI
  |-- Infrastructure
  |-- Persistence
  |-- Common

Persistence
  |-- Application
  |-- Infrastructure
  |-- Common

Infrastructure
  |-- Application
  |-- Common

Application
  |-- Domain
  |-- Common

Domain
  (依存なし)

Common
  (依存なし)
```

### 2.3 技術スタック

#### バックエンド

| カテゴリ | 技術/ライブラリ | バージョン |
|----------|-----------------|-----------|
| フレームワーク | ASP.NET Core | 3.0 |
| ORM | Entity Framework Core | 3.0.0 |
| CQRS/Mediator | MediatR | 7.0.0 |
| オブジェクトマッピング | AutoMapper | 9.0.0 |
| バリデーション | FluentValidation | 8.5.0 |
| API文書 | NSwag | 13.1.2 |
| 認証 | IdentityServer4 | 3.0.0 |
| CSV処理 | CsvHelper | 12.1.2 |

#### フロントエンド

| カテゴリ | 技術/ライブラリ | バージョン |
|----------|-----------------|-----------|
| フレームワーク | Angular | 8.2.5 |
| UIライブラリ | ngx-bootstrap | 5.1.1 |
| 認証クライアント | oidc-client | 1.9.1 |
| アイコン | angular-feather | 6.0.2 |
| テスティング | Jasmine/Karma | 3.3.0/4.0.0 |
| E2Eテスト | Protractor | 5.4.0 |

---

## 3. インフラストラクチャ構成

### 3.1 推奨デプロイ環境

本システムは以下の環境へのデプロイを想定している。

#### 3.1.1 Azure App Service（推奨）

本プロジェクトはAzure Pipelinesを使用したCI/CDが設定されており、Azure App Serviceへのデプロイを主な対象としている。

| コンポーネント | Azureサービス | SKU推奨 |
|---------------|--------------|---------|
| Webアプリケーション | Azure App Service | S1以上 |
| データベース | Azure SQL Database | S0以上 |
| 認証 | Azure AD B2C / IdentityServer4 | - |
| ストレージ | Azure Blob Storage | Standard LRS |
| 監視 | Azure Application Insights | - |

#### 3.1.2 オンプレミス環境

| コンポーネント | 推奨構成 |
|---------------|---------|
| Webサーバー | IIS 10.0+ / Kestrel |
| OSサーバー | Windows Server 2019+ |
| データベース | SQL Server 2017+ |
| ランタイム | .NET Core 3.0 Runtime |

### 3.2 ネットワーク構成

#### 3.2.1 ポート構成

| サービス | ポート | プロトコル | 用途 |
|----------|-------|-----------|------|
| WebUI（開発） | 44376 | HTTPS | フロントエンド/API |
| WebUI（開発） | 54608 | HTTP | フロントエンド/API |
| Angular Dev Server | 4200 | HTTP | 開発時SPA |
| SQL Server | 1433 | TCP | データベース接続 |

#### 3.2.2 SSL/TLS設定

- 開発環境: ASP.NET Core開発証明書を使用
- 本番環境: `CN=northwindtraders.azurewebsites.net` の証明書をストアから参照

```json
{
  "IdentityServer": {
    "Key": {
      "Type": "Store",
      "StoreName": "My",
      "StoreLocation": "CurrentUser",
      "Name": "CN=northwindtraders.azurewebsites.net"
    }
  }
}
```

---

## 4. データベース設計

### 4.1 データベース構成

| 項目 | 内容 |
|------|------|
| DBMS | SQL Server |
| データベース名 | NorthwindTraders |
| 接続方式 | Entity Framework Core（Code First） |
| マイグレーション | EF Core Migrations |

### 4.2 接続文字列

#### 開発環境（LocalDB）
```
Server=(localdb)\mssqllocaldb;Database=NorthwindTraders;Trusted_Connection=True;MultipleActiveResultSets=true;Application Name=NorthwindTraders;
```

### 4.3 テーブル一覧

| テーブル名 | 説明 |
|-----------|------|
| Categories | 商品カテゴリ |
| Customers | 顧客情報 |
| Employees | 従業員情報 |
| EmployeeTerritories | 従業員担当地域（中間テーブル） |
| Orders | 注文情報 |
| OrderDetails | 注文明細 |
| Products | 商品情報 |
| Region | 地域マスタ |
| Shippers | 配送業者 |
| Suppliers | 仕入先 |
| Territories | 担当地域 |

### 4.4 認証データベース

IdentityServer4およびASP.NET Core Identityのための別途ApplicationDbContextが存在する。

| テーブル | 説明 |
|---------|------|
| AspNetUsers | ユーザー情報 |
| AspNetRoles | ロール情報 |
| AspNetUserRoles | ユーザーロール関連 |
| その他Identity関連テーブル | トークン、クレーム等 |

---

## 5. CI/CD パイプライン

### 5.1 Azure Pipelines設定

本プロジェクトは`azure-pipelines.yml`によりCI/CDが構成されている。

#### 5.1.1 パイプライン構成

| 項目 | 設定値 |
|------|--------|
| トリガーブランチ | master |
| ビルドエージェント | ubuntu-latest |
| ビルド構成 | Release |
| .NET SDK | global.jsonで指定（3.0.100） |

#### 5.1.2 パイプラインステップ

1. **SDK設定**: .NET Core SDK 3.0.100のインストール
2. **ビルド**: `dotnet build --configuration Release`
3. **パブリッシュ**: WebUIプロジェクトの自己完結型パブリッシュ
   - ランタイム: win-x86
   - 出力形式: ZIP
4. **アーティファクト公開**: ビルド成果物の保存

#### 5.1.3 パイプライン設定ファイル

```yaml
trigger:
- master

pool:
  vmImage: 'ubuntu-latest'

variables:
  buildConfiguration: 'Release'

steps:
- task: UseDotNet@2
  inputs:
    packageType: 'sdk'
    useGlobalJson: true

- script: dotnet build --configuration $(buildConfiguration)
  displayName: 'dotnet build $(buildConfiguration)'

- task: DotNetCoreCLI@2
  displayName: 'dotnet publish $(buildConfiguration)'
  inputs:
    command: publish
    publishWebProjects: False
    projects: 'Src/WebUI/WebUI.csproj'
    arguments: '--self-contained --runtime win-x86 --configuration $(BuildConfiguration) --output $(Build.ArtifactStagingDirectory)'
    zipAfterPublish: True

- task: PublishBuildArtifacts@1
  inputs:
    pathtoPublish: '$(Build.ArtifactStagingDirectory)'
    artifactName: 'Drop'
```

---

## 6. セキュリティ設計

### 6.1 認証・認可

#### 6.1.1 認証方式

| 項目 | 内容 |
|------|------|
| 認証プロトコル | OpenID Connect / OAuth 2.0 |
| 認証サーバー | IdentityServer4（内蔵） |
| ユーザー管理 | ASP.NET Core Identity |
| トークン形式 | JWT |

#### 6.1.2 クライアント設定

```json
{
  "IdentityServer": {
    "Clients": {
      "Northwind.WebUI": {
        "Profile": "IdentityServerSPA"
      }
    }
  }
}
```

#### 6.1.3 テスト用クライアント

統合テスト用に以下のクライアントが設定されている：

| 項目 | 設定値 |
|------|--------|
| ClientId | Northwind.IntegrationTests |
| GrantType | ResourceOwnerPassword |
| AllowedScopes | Northwind.WebUIAPI, openid, profile |

### 6.2 HTTPS設定

- 本番環境ではHTTPSを強制（HSTS有効）
- HSTSデフォルト有効期間: 30日

### 6.3 CORS設定

SPAアプリケーションとの連携のため、適切なCORS設定が必要。

---

## 7. アプリケーション設定

### 7.1 設定ファイル構成

| ファイル | 用途 |
|---------|------|
| appsettings.json | 基本設定 |
| appsettings.Development.json | 開発環境用設定 |
| appsettings.Production.json | 本番環境用設定 |
| appsettings.Test.json | テスト環境用設定 |
| appsettings.Local.json | ローカル上書き設定（gitignore対象） |

### 7.2 設定読み込み順序

1. appsettings.json
2. appsettings.{Environment}.json
3. appsettings.Local.json
4. User Secrets（開発環境のみ）
5. 環境変数
6. コマンドライン引数

### 7.3 ログ設定

#### 開発環境
```json
{
  "Logging": {
    "LogLevel": {
      "Default": "Debug",
      "System": "Information",
      "Microsoft": "Information"
    }
  }
}
```

#### 本番環境
```json
{
  "Logging": {
    "LogLevel": {
      "Default": "Warning"
    }
  }
}
```

---

## 8. 監視・運用

### 8.1 ヘルスチェック

本システムは`/health`エンドポイントでヘルスチェックを提供する。

| チェック項目 | 内容 |
|-------------|------|
| DbContext | NorthwindDbContextの接続確認 |

```csharp
services.AddHealthChecks()
    .AddDbContextCheck<NorthwindDbContext>();
```

### 8.2 API仕様

- Swagger UI: `/api`
- OpenAPI仕様: NSwagにより自動生成

### 8.3 推奨監視項目

| 監視項目 | 説明 | 閾値（参考） |
|---------|------|-------------|
| CPU使用率 | Webサーバーの負荷 | 80%以上でアラート |
| メモリ使用率 | アプリケーションメモリ | 85%以上でアラート |
| レスポンスタイム | API応答時間 | 2秒以上でアラート |
| エラー率 | 5xxエラーの発生率 | 1%以上でアラート |
| DB接続 | データベース接続状況 | 失敗時アラート |

---

## 9. スケーリング考慮事項

### 9.1 水平スケーリング

本アプリケーションは以下の点を考慮することでスケールアウト可能：

| 項目 | 考慮事項 |
|------|---------|
| セッション | デフォルトでインメモリ。分散環境ではRedis等の外部ストアを使用 |
| 認証トークン | JWTベースのため、ステートレス。スケールアウト可能 |
| データベース | 単一SQL Server。読み取り専用レプリカの追加を検討 |

### 9.2 垂直スケーリング

| リソース | 推奨スケーリング方針 |
|---------|---------------------|
| CPU | コア数の増加によりリクエスト処理能力向上 |
| メモリ | Entity Frameworkキャッシュ向上のためメモリ増加 |
| ストレージ | データベースサイズに応じて拡張 |

---

## 10. 障害復旧

### 10.1 バックアップ戦略

| 対象 | バックアップ方法 | 推奨頻度 |
|------|-----------------|---------|
| データベース | SQL Server バックアップ | 日次フルバックアップ、差分バックアップ |
| アプリケーション | ソースコードリポジトリ | CI/CDによる継続的デプロイ |
| 設定ファイル | 構成管理ツールによる管理 | 変更時 |

### 10.2 復旧手順（概要）

1. 新しいApp Serviceインスタンスの作成
2. データベースの復元
3. アプリケーションのデプロイ（CI/CDパイプライン実行）
4. 接続文字列・設定の更新
5. 動作確認

### 10.3 RTO/RPO目標（参考）

| 指標 | 目標値 |
|------|--------|
| RTO（目標復旧時間） | 4時間以内 |
| RPO（目標復旧時点） | 24時間以内 |

---

## 11. 付録

### 11.1 必要ソフトウェア（開発環境）

| ソフトウェア | バージョン | 用途 |
|-------------|-----------|------|
| .NET Core SDK | 3.0.100 | バックエンドビルド |
| Node.js | 10.x以上 | フロントエンドビルド |
| npm | 6.x以上 | パッケージ管理 |
| SQL Server LocalDB | 2016以上 | ローカルデータベース |
| Visual Studio | 2019以上 | IDE（推奨） |
| Visual Studio Code | 最新版 | IDE（代替） |

### 11.2 参考リンク

- [ASP.NET Core ドキュメント](https://docs.microsoft.com/aspnet/core/)
- [Entity Framework Core ドキュメント](https://docs.microsoft.com/ef/core/)
- [IdentityServer4 ドキュメント](https://identityserver4.readthedocs.io/)
- [Angular ドキュメント](https://angular.io/docs)
- [Azure App Service ドキュメント](https://docs.microsoft.com/azure/app-service/)

---

## 改訂履歴

| 版 | 日付 | 変更内容 | 作成者 |
|----|------|---------|--------|
| 1.0 | 2026-01-07 | 初版作成 | Claude Code |
