# Northwind Traders コードリーディングガイドライン

## はじめに

このガイドラインは、Northwind Tradersのコードベースを効率的に理解するための手引きです。
C#/.NET Coreに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、C#の基本構文と概念を解説します。

### 1.1 プログラム構造

C#プログラムは名前空間（namespace）でコードを組織化し、クラスを基本単位として構成されます。

```csharp
// ファイル: Src/Domain/Entities/Customer.cs:1-6
using System.Collections.Generic;

namespace Northwind.Domain.Entities
{
    public class Customer
    {
```

**ポイント:**
- `using` ディレクティブで他の名前空間のクラスをインポート
- `namespace` でコードを論理的にグループ化
- `public class` でクラスを定義

### 1.2 データ型と変数

C#は強い型付け言語です。プロパティ（Property）はフィールドへのアクセサを提供します。

```csharp
// ファイル: Src/Domain/Entities/Customer.cs:12-23
public string CustomerId { get; set; }
public string CompanyName { get; set; }
public string ContactName { get; set; }
public string ContactTitle { get; set; }
public string Address { get; set; }
public string City { get; set; }
public string Region { get; set; }
public string PostalCode { get; set; }
public string Country { get; set; }
public string Phone { get; set; }
public string Fax { get; set; }

public ICollection<Order> Orders { get; private set; }
```

**主なデータ型:**
| 型 | 説明 | 例 |
|---|---|---|
| `string` | 文字列 | `"ALFKI"` |
| `int` | 整数 | `42` |
| `decimal` | 高精度数値（金額等） | `19.99m` |
| `DateTime` | 日時 | `DateTime.Now` |
| `bool` | 真偽値 | `true` / `false` |

### 1.3 制御構造

条件分岐やループは他の言語と同様のパターンを使用します。

```csharp
// ファイル: Src/Persistence/NorthwindDbContext.cs:53-68
foreach (var entry in ChangeTracker.Entries<AuditableEntity>())
{
    switch (entry.State)
    {
        case EntityState.Added:
            entry.Entity.CreatedBy = _currentUserService.UserId;
            entry.Entity.Created = _dateTime.Now;
            break;
        case EntityState.Modified:
            entry.Entity.LastModifiedBy = _currentUserService.UserId;
            entry.Entity.LastModified = _dateTime.Now;
            break;
    }
}
```

### 1.4 関数/メソッド定義

メソッドはクラス内で定義され、`async`/`await`による非同期処理をサポートします。

```csharp
// ファイル: Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQueryHandler.cs:22-34
public async Task<CustomersListVm> Handle(GetCustomersListQuery request, CancellationToken cancellationToken)
{
    var customers = await _context.Customers
        .ProjectTo<CustomerLookupDto>(_mapper.ConfigurationProvider)
        .ToListAsync(cancellationToken);

    var vm = new CustomersListVm
    {
        Customers = customers
    };

    return vm;
}
```

**ポイント:**
- `async` キーワードで非同期メソッドを宣言
- `await` で非同期操作の完了を待機
- `Task<T>` は非同期操作の結果を表す型

### 1.5 モジュール/インポート

C#では `using` ディレクティブで名前空間をインポートします。

```csharp
// ファイル: Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQueryHandler.cs:1-7
using AutoMapper;
using AutoMapper.QueryableExtensions;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Northwind.Application.Common.Interfaces;
using System.Threading;
using System.Threading.Tasks;
```

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### ASP.NET Core

WebアプリケーションフレームワークでAPIエンドポイントを提供します。

```csharp
// ファイル: Src/WebUI/Controllers/CustomersController.cs:13-22
[Authorize]
public class CustomersController : BaseController
{
    [HttpGet]
    public async Task<ActionResult<CustomersListVm>> GetAll()
    {
        var vm = await Mediator.Send(new GetCustomersListQuery());

        return Ok(vm);
    }
}
```

**主要な属性:**
- `[ApiController]`: API コントローラーであることを示す
- `[Route]`: ルーティングパターンを定義
- `[HttpGet]`, `[HttpPost]`, `[HttpPut]`, `[HttpDelete]`: HTTPメソッドを指定
- `[Authorize]`: 認証が必要なエンドポイント

#### MediatR（メディエーターパターン）

コマンド/クエリとハンドラーを分離し、疎結合を実現します。

```csharp
// ファイル: Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQuery.cs:1-8
using MediatR;

namespace Northwind.Application.Customers.Queries.GetCustomersList
{
    public class GetCustomersListQuery : IRequest<CustomersListVm>
    {
    }
}
```

#### Entity Framework Core

ORMとしてデータベースアクセスを抽象化します。

```csharp
// ファイル: Src/Persistence/NorthwindDbContext.cs:31-51
public DbSet<Category> Categories { get; set; }
public DbSet<Customer> Customers { get; set; }
public DbSet<Employee> Employees { get; set; }
public DbSet<Product> Products { get; set; }
// ... 他のエンティティ
```

#### FluentValidation

宣言的なバリデーションルールを定義します。

```csharp
// ファイル: Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommandValidator.cs:1-22
using FluentValidation;

namespace Northwind.Application.Customers.Commands.CreateCustomer
{
    public class CreateCustomerCommandValidator : AbstractValidator<CreateCustomerCommand>
    {
        public CreateCustomerCommandValidator()
        {
            RuleFor(x => x.Id).Length(5).NotEmpty();
            RuleFor(x => x.Address).MaximumLength(60);
            RuleFor(x => x.City).MaximumLength(15);
            RuleFor(x => x.CompanyName).MaximumLength(40).NotEmpty();
            // ...
        }
    }
}
```

#### AutoMapper

オブジェクト間のマッピングを自動化します。

```csharp
// ファイル: Src/Application/Customers/Queries/GetCustomersList/CustomerLookupDto.cs:1-19
using AutoMapper;
using Northwind.Application.Common.Mappings;
using Northwind.Domain.Entities;

namespace Northwind.Application.Customers.Queries.GetCustomersList
{
    public class CustomerLookupDto : IMapFrom<Customer>
    {
        public string Id { get; set; }
        public string Name { get; set; }

        public void Mapping(Profile profile)
        {
            profile.CreateMap<Customer, CustomerLookupDto>()
                .ForMember(d => d.Id, opt => opt.MapFrom(s => s.CustomerId))
                .ForMember(d => d.Name, opt => opt.MapFrom(s => s.CompanyName));
        }
    }
}
```

### 2.2 プロジェクト独自のパターン

#### CQRS（Command Query Responsibility Segregation）

読み取り（Query）と書き込み（Command）を明確に分離しています。

- **Query**: データの読み取り専用。状態を変更しない。
- **Command**: データの作成・更新・削除。状態を変更する。

```
Application/
  Customers/
    Queries/           # 読み取り操作
      GetCustomersList/
      GetCustomerDetail/
    Commands/          # 書き込み操作
      CreateCustomer/
      UpdateCustomer/
      DeleteCustomer/
```

#### Dependency Injection パターン

各レイヤーで依存性注入の設定を`DependencyInjection.cs`に集約しています。

```csharp
// ファイル: Src/Application/DependencyInjection.cs:1-21
public static class DependencyInjection
{
    public static IServiceCollection AddApplication(this IServiceCollection services)
    {
        services.AddAutoMapper(Assembly.GetExecutingAssembly());
        services.AddMediatR(Assembly.GetExecutingAssembly());
        services.AddTransient(typeof(IPipelineBehavior<,>), typeof(RequestPerformanceBehaviour<,>));
        services.AddTransient(typeof(IPipelineBehavior<,>), typeof(RequestValidationBehavior<,>));

        return services;
    }
}
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `{Entity}Controller.cs` | APIコントローラー | `CustomersController.cs` |
| `Get{Entity}ListQuery.cs` | 一覧取得クエリ | `GetCustomersListQuery.cs` |
| `Get{Entity}DetailQuery.cs` | 詳細取得クエリ | `GetCustomerDetailQuery.cs` |
| `Create{Entity}Command.cs` | 作成コマンド | `CreateCustomerCommand.cs` |
| `Update{Entity}Command.cs` | 更新コマンド | `UpdateCustomerCommand.cs` |
| `Delete{Entity}Command.cs` | 削除コマンド | `DeleteCustomerCommand.cs` |
| `{Entity}Configuration.cs` | EF Core設定 | `CustomerConfiguration.cs` |
| `{Name}Dto.cs` | データ転送オブジェクト | `CustomerLookupDto.cs` |
| `{Name}Vm.cs` | ビューモデル | `CustomersListVm.cs` |
| `{Command/Query}Handler.cs` | ハンドラークラス | `GetCustomersListQueryHandler.cs` |
| `{Command/Query}Validator.cs` | バリデーター | `CreateCustomerCommandValidator.cs` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `I{Name}` | インターフェース | `INorthwindDbContext`, `INotificationService` |
| `_{name}` | プライベートフィールド | `_context`, `_mapper` |
| `{Name}Dto` | Data Transfer Object | `CustomerLookupDto` |
| `{Name}Vm` | ViewModel（APIレスポンス） | `CustomersListVm` |
| `Abstract{Name}` | 抽象クラス | `AbstractValidator` |

### 3.3 プログラム分類一覧

| 分類 | 場所 | 説明 |
|-----|------|------|
| エンティティ | `Src/Domain/Entities/` | ビジネスドメインのオブジェクト |
| 値オブジェクト | `Src/Domain/ValueObjects/` | 不変の値を表すオブジェクト |
| コマンド | `Src/Application/{Entity}/Commands/` | 書き込み操作の定義 |
| クエリ | `Src/Application/{Entity}/Queries/` | 読み取り操作の定義 |
| コントローラー | `Src/WebUI/Controllers/` | APIエンドポイント |
| 設定 | `Src/Persistence/Configurations/` | EF Coreエンティティ設定 |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
Northwind.sln
├── Src/
│   ├── Common/                    # 共通インターフェース
│   │   └── IDateTime.cs
│   ├── Domain/                    # ドメイン層（ビジネスロジックの中核）
│   │   ├── Common/
│   │   │   ├── AuditableEntity.cs
│   │   │   └── ValueObject.cs
│   │   ├── Entities/              # エンティティクラス
│   │   │   ├── Customer.cs
│   │   │   ├── Product.cs
│   │   │   ├── Order.cs
│   │   │   └── ...
│   │   ├── Exceptions/
│   │   └── ValueObjects/
│   ├── Application/               # アプリケーション層（ユースケース）
│   │   ├── Common/
│   │   │   ├── Behaviours/        # MediatRパイプライン
│   │   │   ├── Exceptions/
│   │   │   ├── Interfaces/        # 抽象インターフェース
│   │   │   ├── Mappings/
│   │   │   └── Models/
│   │   ├── Customers/
│   │   │   ├── Commands/
│   │   │   │   ├── CreateCustomer/
│   │   │   │   ├── UpdateCustomer/
│   │   │   │   └── DeleteCustomer/
│   │   │   └── Queries/
│   │   │       ├── GetCustomersList/
│   │   │       └── GetCustomerDetail/
│   │   ├── Products/
│   │   ├── Categories/
│   │   ├── Employees/
│   │   └── ...
│   ├── Infrastructure/            # インフラストラクチャ層
│   │   ├── Identity/              # 認証・認可
│   │   ├── Files/                 # ファイル操作
│   │   └── DependencyInjection.cs
│   ├── Persistence/               # 永続化層
│   │   ├── Configurations/        # EF Core設定
│   │   ├── Migrations/
│   │   ├── NorthwindDbContext.cs
│   │   └── DependencyInjection.cs
│   └── WebUI/                     # プレゼンテーション層
│       ├── ClientApp/             # Angular SPA
│       ├── Controllers/           # APIコントローラー
│       ├── Services/
│       ├── Program.cs
│       └── Startup.cs
└── Tests/
    ├── Application.UnitTests/
    ├── Domain.UnitTests/
    ├── WebUI.IntegrationTests/
    └── Persistence.IntegrationTests/
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `Src/Common` | プロジェクト共通のインターフェース | `IDateTime.cs` |
| `Src/Domain` | ビジネスドメインの定義（外部依存なし） | `Customer.cs`, `Product.cs` |
| `Src/Application` | ユースケースの実装（CQRS） | `GetCustomersListQuery.cs`, `CreateCustomerCommand.cs` |
| `Src/Infrastructure` | 外部サービスとの連携 | `NotificationService.cs`, `CsvFileBuilder.cs` |
| `Src/Persistence` | データベースアクセス | `NorthwindDbContext.cs` |
| `Src/WebUI` | APIエンドポイント・フロントエンド | `CustomersController.cs`, `Startup.cs` |
| `Tests/` | 各種テストプロジェクト | `*Tests.cs` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

本プロジェクトは **Clean Architecture** を採用しています。

```mermaid
graph TB
    subgraph "Presentation Layer"
        WebUI[WebUI<br/>Controllers, ClientApp]
    end

    subgraph "Infrastructure Layer"
        Infrastructure[Infrastructure<br/>Identity, Files, Services]
        Persistence[Persistence<br/>DbContext, Configurations]
    end

    subgraph "Application Layer"
        Application[Application<br/>Commands, Queries, Handlers]
    end

    subgraph "Domain Layer"
        Domain[Domain<br/>Entities, ValueObjects]
    end

    WebUI --> Application
    WebUI --> Infrastructure
    WebUI --> Persistence
    Infrastructure --> Application
    Persistence --> Application
    Application --> Domain
```

**依存関係の方向:**
- 外側から内側へ向かう（Domain層は何にも依存しない）
- 内側の層は外側の層を知らない

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Domain | ビジネスロジックの中核。エンティティ、値オブジェクトの定義 | `Customer.cs`, `AdAccount.cs` |
| Application | ユースケースの実装。コマンド/クエリハンドラー | `GetCustomersListQueryHandler.cs` |
| Infrastructure | 外部サービスとの連携。認証、ファイル操作 | `NotificationService.cs` |
| Persistence | データベースアクセス。EF Core実装 | `NorthwindDbContext.cs` |
| WebUI | APIエンドポイント。フロントエンド | `CustomersController.cs` |

### 5.3 データフロー

リクエストからレスポンスまでの流れ:

```
[HTTP Request]
    ↓
[Controller] - APIエンドポイント受付
    ↓
[MediatR] - コマンド/クエリをディスパッチ
    ↓
[Pipeline Behaviors] - バリデーション、ログ、パフォーマンス計測
    ↓
[Handler] - ビジネスロジック実行
    ↓
[DbContext] - データベースアクセス
    ↓
[Entity] - ドメインオブジェクト操作
    ↓
[AutoMapper] - DTO/ViewModelへ変換
    ↓
[HTTP Response]
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

アプリケーションは `Program.cs` から起動します。

```csharp
// ファイル: Src/WebUI/Program.cs:22-48
public static async Task Main(string[] args)
{
    var host = CreateWebHostBuilder(args).Build();

    using (var scope = host.Services.CreateScope())
    {
        var services = scope.ServiceProvider;

        try
        {
            var northwindContext = services.GetRequiredService<NorthwindDbContext>();
            northwindContext.Database.Migrate();

            var identityContext = services.GetRequiredService<ApplicationDbContext>();
            identityContext.Database.Migrate();

            var mediator = services.GetRequiredService<IMediator>();
            await mediator.Send(new SeedSampleDataCommand(), CancellationToken.None);
        }
        catch (Exception ex)
        {
            // エラーログ
        }
    }

    host.Run();
}
```

**起動時の処理:**
1. Webホストの構築
2. データベースマイグレーション実行
3. サンプルデータの投入
4. HTTPリクエストの受付開始

### 6.2 ビジネスロジック

CQRSパターンに従い、ビジネスロジックはHandlerクラスに集約されます。

```csharp
// ファイル: Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs:33-68
public class Handler : IRequestHandler<CreateCustomerCommand>
{
    private readonly INorthwindDbContext _context;
    private readonly IMediator _mediator;

    public Handler(INorthwindDbContext context, IMediator mediator)
    {
        _context = context;
        _mediator = mediator;
    }

    public async Task<Unit> Handle(CreateCustomerCommand request, CancellationToken cancellationToken)
    {
        var entity = new Customer
        {
            CustomerId = request.Id,
            Address = request.Address,
            // ... プロパティ設定
        };

        _context.Customers.Add(entity);
        await _context.SaveChangesAsync(cancellationToken);

        // ドメインイベント発行
        await _mediator.Publish(new CustomerCreated { CustomerId = entity.CustomerId }, cancellationToken);

        return Unit.Value;
    }
}
```

### 6.3 データアクセス

Entity Framework Coreを使用し、リポジトリパターンの代わりにDbContextを直接使用します。

```csharp
// ファイル: Src/Persistence/NorthwindDbContext.cs:9-78
public class NorthwindDbContext : DbContext, INorthwindDbContext
{
    // DbSetでテーブルへのアクセスを提供
    public DbSet<Category> Categories { get; set; }
    public DbSet<Customer> Customers { get; set; }
    // ...

    // 保存時に監査情報を自動設定
    public override Task<int> SaveChangesAsync(CancellationToken cancellationToken = new CancellationToken())
    {
        foreach (var entry in ChangeTracker.Entries<AuditableEntity>())
        {
            switch (entry.State)
            {
                case EntityState.Added:
                    entry.Entity.CreatedBy = _currentUserService.UserId;
                    entry.Entity.Created = _dateTime.Now;
                    break;
                case EntityState.Modified:
                    entry.Entity.LastModifiedBy = _currentUserService.UserId;
                    entry.Entity.LastModified = _dateTime.Now;
                    break;
            }
        }
        return base.SaveChangesAsync(cancellationToken);
    }
}
```

### 6.4 ユーティリティ/共通機能

#### バリデーションパイプライン

```csharp
// ファイル: Src/Application/Common/Behaviours/RequestValidationBehavior.cs:9-38
public class RequestValidationBehavior<TRequest, TResponse> : IPipelineBehavior<TRequest, TResponse>
    where TRequest : IRequest<TResponse>
{
    private readonly IEnumerable<IValidator<TRequest>> _validators;

    public Task<TResponse> Handle(TRequest request, CancellationToken cancellationToken, RequestHandlerDelegate<TResponse> next)
    {
        var context = new ValidationContext(request);

        var failures = _validators
            .Select(v => v.Validate(context))
            .SelectMany(result => result.Errors)
            .Where(f => f != null)
            .ToList();

        if (failures.Count != 0)
        {
            throw new ValidationException(failures);
        }

        return next();
    }
}
```

#### マッピングインターフェース

```csharp
// ファイル: Src/Application/Common/Mappings/IMapFrom.cs:1-9
public interface IMapFrom<T>
{
    void Mapping(Profile profile) => profile.CreateMap(typeof(T), GetType());
}
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| CQRS | コマンドとクエリの分離 | 高 | `GetCustomersListQuery.cs`, `CreateCustomerCommand.cs` |
| Mediator | リクエスト/レスポンスの仲介 | 高 | `CustomersController.cs` |
| Repository (implicit) | DbContextによるデータアクセス | 高 | `NorthwindDbContext.cs` |
| Value Object | 不変の値オブジェクト | 中 | `AdAccount.cs` |
| Factory Method | オブジェクト生成の抽象化 | 中 | `AdAccount.For()` |
| Dependency Injection | 依存性の注入 | 高 | `DependencyInjection.cs` |

### 各パターンの詳細

#### パターン1: CQRS（Command Query Responsibility Segregation）

**目的:** 読み取りと書き込みの処理を分離し、それぞれに最適化された実装を可能にする

**実装例:**
```csharp
// ファイル: Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQuery.cs:1-8
// Query: データを返すが状態を変更しない
public class GetCustomersListQuery : IRequest<CustomersListVm>
{
}

// ファイル: Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs:9
// Command: 状態を変更するがデータを返さない
public class CreateCustomerCommand : IRequest
{
    public string Id { get; set; }
    // ...
}
```

**解説:** Queryは`IRequest<TResponse>`を継承し結果を返す。Commandは`IRequest`（Unit型を返す）を継承し、状態変更後に成功/失敗のみを伝える。

#### パターン2: Value Object

**目的:** 不変の値を表現し、ドメインの概念を明確にする

**実装例:**
```csharp
// ファイル: Src/Domain/ValueObjects/AdAccount.cs:8-56
public class AdAccount : ValueObject
{
    private AdAccount()
    {
    }

    public static AdAccount For(string accountString)
    {
        var account = new AdAccount();
        // パース処理
        return account;
    }

    public string Domain { get; private set; }
    public string Name { get; private set; }

    protected override IEnumerable<object> GetAtomicValues()
    {
        yield return Domain;
        yield return Name;
    }
}
```

**解説:** コンストラクタをprivateにしてFactory Methodで生成。`GetAtomicValues`で値の等価性を定義。

#### パターン3: Fluent Validation

**目的:** 宣言的で読みやすいバリデーションルールを定義する

**実装例:**
```csharp
// ファイル: Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommandValidator.cs:5-22
public class CreateCustomerCommandValidator : AbstractValidator<CreateCustomerCommand>
{
    public CreateCustomerCommandValidator()
    {
        RuleFor(x => x.Id).Length(5).NotEmpty();
        RuleFor(x => x.CompanyName).MaximumLength(40).NotEmpty();
        // ...
    }
}
```

**解説:** `AbstractValidator<T>`を継承し、コンストラクタ内で`RuleFor`を使ってルールを定義。MediatRパイプラインで自動実行される。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（Controller）
2. 処理の流れを追跡（MediatR -> Handler）
3. データの変換を確認（DTO/ViewModel）
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: 顧客一覧取得

**概要:** 顧客一覧を取得するAPIエンドポイントの処理フロー

**処理フロー:**
```
GET /api/Customers/GetAll → CustomersController → MediatR → GetCustomersListQueryHandler → DbContext → AutoMapper → Response
```

**詳細な追跡:**

1. **Controllerでリクエスト受付** (`Src/WebUI/Controllers/CustomersController.cs:16-22`)
   ```csharp
   [HttpGet]
   public async Task<ActionResult<CustomersListVm>> GetAll()
   {
       var vm = await Mediator.Send(new GetCustomersListQuery());
       return Ok(vm);
   }
   ```

2. **MediatRがHandlerを呼び出し** (`Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQueryHandler.cs:22-34`)
   ```csharp
   public async Task<CustomersListVm> Handle(GetCustomersListQuery request, CancellationToken cancellationToken)
   {
       var customers = await _context.Customers
           .ProjectTo<CustomerLookupDto>(_mapper.ConfigurationProvider)
           .ToListAsync(cancellationToken);

       var vm = new CustomersListVm
       {
           Customers = customers
       };

       return vm;
   }
   ```

3. **DTOへのマッピング** (`Src/Application/Customers/Queries/GetCustomersList/CustomerLookupDto.cs:12-17`)
   ```csharp
   public void Mapping(Profile profile)
   {
       profile.CreateMap<Customer, CustomerLookupDto>()
           .ForMember(d => d.Id, opt => opt.MapFrom(s => s.CustomerId))
           .ForMember(d => d.Name, opt => opt.MapFrom(s => s.CompanyName));
   }
   ```

#### 例2: 顧客作成

**概要:** 新規顧客を作成するAPIエンドポイントの処理フロー

**処理フロー:**
```
POST /api/Customers/Create → Controller → Validation Pipeline → Handler → DbContext → Notification
```

**詳細な追跡:**

1. **Controllerでリクエスト受付** (`Src/WebUI/Controllers/CustomersController.cs:34-42`)
   ```csharp
   [HttpPost]
   public async Task<IActionResult> Create([FromBody]CreateCustomerCommand command)
   {
       await Mediator.Send(command);
       return NoContent();
   }
   ```

2. **バリデーション実行** (`Src/Application/Common/Behaviours/RequestValidationBehavior.cs:21-37`)
   ```csharp
   public Task<TResponse> Handle(TRequest request, ...)
   {
       var failures = _validators
           .Select(v => v.Validate(context))
           .SelectMany(result => result.Errors)
           .Where(f => f != null)
           .ToList();

       if (failures.Count != 0)
           throw new ValidationException(failures);

       return next();
   }
   ```

3. **Handlerでエンティティ作成** (`Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs:44-66`)
   ```csharp
   public async Task<Unit> Handle(CreateCustomerCommand request, CancellationToken cancellationToken)
   {
       var entity = new Customer { /* プロパティ設定 */ };

       _context.Customers.Add(entity);
       await _context.SaveChangesAsync(cancellationToken);

       await _mediator.Publish(new CustomerCreated { CustomerId = entity.CustomerId }, cancellationToken);

       return Unit.Value;
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイント（Controller）を特定したか
- [ ] MediatRによるDispatchを確認したか
- [ ] Pipeline Behaviors（バリデーション等）を確認したか
- [ ] Handlerのビジネスロジックを理解したか
- [ ] データアクセス（DbContext）の操作を確認したか
- [ ] DTO/ViewModelへの変換を確認したか
- [ ] ドメインイベント（Notification）の有無を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `Northwind.sln` - ソリューション構造
3. `Src/WebUI/Startup.cs` - DIとミドルウェア構成

#### 特定機能を理解したい場合
1. `Src/WebUI/Controllers/{Entity}Controller.cs` - APIエンドポイント
2. `Src/Application/{Entity}/` - コマンド/クエリ
3. `Src/Domain/Entities/{Entity}.cs` - エンティティ定義

#### 改修作業を行う場合
1. 対象機能のController
2. 関連するCommand/Query
3. バリデーションルール（Validator）
4. エンティティ設定（Configuration）
5. テストコード

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初に |
| `Src/Domain/README.md` | ドメイン層の説明 | ビジネスロジック理解時 |
| `docs/code-to-docs/` | 生成されたドキュメント群 | 詳細設計確認時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: Handlerがどこから呼ばれているかわからない
A: MediatRパターンを使用しています。`Mediator.Send(request)`でリクエストを送信すると、MediatRが対応するHandlerを自動で見つけて実行します。Handlerは`IRequestHandler<TRequest, TResponse>`インターフェースを実装しているクラスです。

#### Q: バリデーションはどこで実行されているか
A: `Src/Application/Common/Behaviours/RequestValidationBehavior.cs`がMediatRパイプラインとして登録されており、Handler実行前に自動でバリデーションが実行されます。Validatorは`AbstractValidator<T>`を継承したクラスで、FluentValidationにより自動で検出されます。

#### Q: DTOとViewModelの違いは何か
A: 本プロジェクトでは以下のように使い分けています:
- **Dto**: 内部的なデータ転送オブジェクト。リスト項目など
- **Vm (ViewModel)**: APIレスポンスの最上位オブジェクト。リストを含むコンテナなど

#### Q: 新しいエンティティを追加するには何が必要か
A: 以下のファイルを作成/更新する必要があります:
1. `Src/Domain/Entities/{Entity}.cs` - エンティティクラス
2. `Src/Persistence/Configurations/{Entity}Configuration.cs` - EF Core設定
3. `Src/Application/Common/Interfaces/INorthwindDbContext.cs` - DbSetプロパティ追加
4. `Src/Persistence/NorthwindDbContext.cs` - DbSetプロパティ追加
5. `Src/Application/{Entity}/` - Commands, Queriesフォルダ
6. `Src/WebUI/Controllers/{Entity}Controller.cs` - APIコントローラー

### デバッグのヒント

1. **MediatRのフロー確認**: `RequestPerformanceBehaviour`にブレークポイントを設置すると、すべてのリクエストをキャプチャできます。

2. **EF Coreクエリ確認**: `appsettings.Development.json`でログレベルを設定すると、生成されるSQLを確認できます。

3. **バリデーションエラー確認**: `ValidationException`がスローされた場合、`Errors`プロパティに詳細なエラー情報が含まれます。

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Clean Architecture | 依存関係を内側に向けることで、ビジネスロジックを外部依存から隔離するアーキテクチャパターン |
| CQRS | Command Query Responsibility Segregation。読み取りと書き込みを分離するパターン |
| MediatR | .NETのMediator実装ライブラリ。リクエスト/レスポンスパターンを実現 |
| Entity Framework Core | .NETのORM。データベースアクセスを抽象化 |
| FluentValidation | 宣言的なバリデーションルール定義ライブラリ |
| AutoMapper | オブジェクト間のマッピングを自動化するライブラリ |
| DbContext | EF Coreのデータベースセッションを表すクラス |
| DbSet | EF Coreでエンティティコレクションを表すプロパティ |
| Dto | Data Transfer Object。レイヤー間のデータ転送用オブジェクト |
| ViewModel | プレゼンテーション層向けのデータ構造 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `Northwind.sln` | ソリューションファイル | プロジェクト構成 |
| `Src/Domain/` | ドメイン層 | エンティティ、値オブジェクト |
| `Src/Application/` | アプリケーション層 | コマンド、クエリ、ハンドラー |
| `Src/Infrastructure/` | インフラストラクチャ層 | 認証、ファイル操作 |
| `Src/Persistence/` | 永続化層 | DbContext、マイグレーション |
| `Src/WebUI/` | プレゼンテーション層 | コントローラー、Angular SPA |
| `Tests/` | テストプロジェクト | 単体テスト、統合テスト |

### C. 参考資料

- [Clean Architecture with ASP.NET Core](https://github.com/jasontaylordev/CleanArchitecture) - 最新版のClean Architectureテンプレート
- [MediatR Wiki](https://github.com/jbogard/MediatR/wiki) - MediatRの公式ドキュメント
- [Entity Framework Core Documentation](https://docs.microsoft.com/ef/core/) - EF Coreの公式ドキュメント
- [FluentValidation Documentation](https://docs.fluentvalidation.net/) - FluentValidationの公式ドキュメント
- [AutoMapper Documentation](https://docs.automapper.org/) - AutoMapperの公式ドキュメント
- [ASP.NET Core Documentation](https://docs.microsoft.com/aspnet/core/) - ASP.NET Coreの公式ドキュメント
