# セキュリティ設計書

## 概要

本ドキュメントは、Northwind Tradersアプリケーションのセキュリティアーキテクチャとセキュリティ対策について記載しています。本アプリケーションはASP.NET Core 3.0をベースとしたWebアプリケーションであり、IdentityServerを使用したOAuth2/OpenID Connect認証、ASP.NET Core Identityによるユーザー管理、FluentValidationによる入力検証などのセキュリティ機能を実装しています。

## 認証設計

### 認証方式

本アプリケーションは以下の認証方式を採用しています：

| 認証方式 | 説明 | 実装ファイル |
| --- | --- | --- |
| ASP.NET Core Identity | ユーザー管理・認証基盤 | `Src/Infrastructure/DependencyInjection.cs` |
| IdentityServer4 | OAuth2/OpenID Connect認証サーバー | `Src/Infrastructure/DependencyInjection.cs` |
| JWT Bearer Token | APIアクセス認証 | `Src/WebUI/ClientApp/src/api-authorization/authorize.interceptor.ts` |
| Cookie Authentication | Webページ認証 | ASP.NET Core Identity標準機能 |

#### 認証フロー

1. **Webアプリケーション（Angular SPA）**: oidc-clientライブラリを使用したOIDC認証フロー
   - Silent認証（iframe）
   - ポップアップ認証
   - リダイレクト認証（フォールバック）

2. **API認証**: JWTベアラートークン
   - AuthorizeInterceptorによりリクエストヘッダーにBearerトークンを付与
   - 同一オリジンチェックによるトークン送信制御

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | ASP.NET Core Identity標準設定 | デフォルト値を使用 |
| セッション固定化対策 | 実装済み | ログイン時に外部Cookie削除（`SignOutAsync`） |
| アカウントロックアウト | 有効 | `lockoutOnFailure: true`設定 |
| Remember Me | 実装済み | ユーザー選択可能 |
| Silent Token Renewal | 有効 | `automaticSilentRenew: true` |

### パスワードポリシー

| 項目 | 設定 |
| --- | --- |
| 最小文字数 | 6文字 |
| 最大文字数 | 100文字 |
| パスワード確認 | 必須（ConfirmPassword） |

## 認可設計

### 権限体系

本アプリケーションでは、コントローラーレベルでの認可制御を実装しています：

| コントローラー | 認可設定 | 説明 |
| --- | --- | --- |
| ProductsController | `[Authorize]` | 認証済みユーザーのみアクセス可能 |
| CustomersController | `[Authorize]` | 認証済みユーザーのみアクセス可能 |
| CategoriesController | `[Authorize]` | 認証済みユーザーのみアクセス可能 |
| EmployeesController | 未確認 | 要確認 |

#### 公開エンドポイント（AllowAnonymous）

| エンドポイント | 説明 |
| --- | --- |
| `GET /api/Products/GetAll` | 製品一覧取得 |
| `GET /api/Products/Get/{id}` | 製品詳細取得 |
| `GET /api/Products/Download` | 製品一覧ファイルダウンロード |
| `/Identity/Account/Login` | ログインページ |
| `/Identity/Account/Register` | ユーザー登録ページ |

### アクセス制御

- **コントローラーレベル**: `[Authorize]`属性による認可
- **アクションレベル**: `[AllowAnonymous]`による例外設定
- **認証ユーザー情報**: `ICurrentUserService`によるユーザーID取得

## 通信セキュリティ

| 項目 | 対策 | 実装箇所 |
| --- | --- | --- |
| HTTPS | `UseHttpsRedirection()`により強制 | `Startup.cs` |
| HSTS | 本番環境で有効（30日間） | `Startup.cs` - `UseHsts()` |
| Same Origin Check | API呼び出し時に同一オリジンチェック | `authorize.interceptor.ts` |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 | 備考 |
| --- | --- | --- |
| 通信 | TLS/HTTPS | `UseHttpsRedirection()`による強制 |
| パスワード | ASP.NET Core Identity標準ハッシュ | PBKDF2ベースのハッシュ化 |
| JWT署名 | IdentityServer4標準 | 本番環境では証明書ストアから取得 |

### 機密情報管理

| 方式 | 説明 | 設定ファイル |
| --- | --- | --- |
| User Secrets | 開発環境での機密情報管理 | `UserSecretsId`設定済み |
| 環境変数 | 本番環境での設定上書き | `AddEnvironmentVariables()` |
| 証明書ストア | IdentityServer署名キー（本番） | `appsettings.Production.json` |

#### 本番環境のIdentityServer鍵設定

```json
{
  "IdentityServer": {
    "Key": {
      "Type": "Store",
      "StoreName": "My",
      "StoreLocation": "CurrentUser",
      "Name": "CN=northwindtraders.azurewebsites.net"
    }
  }
}
```

## 入出力対策

| 脅威 | 対策 | 実装箇所 |
| --- | --- | --- |
| XSS | Razor自動エスケープ、HtmlEncoder | Razorビュー、`Login.cshtml.cs` |
| SQLインジェクション | Entity Framework Core（パラメータ化クエリ） | 全データアクセス層 |
| CSRF | Razor Pages標準AntiForgeryToken | ASP.NET Core標準機能 |
| 入力検証エラー | カスタム例外ハンドラーで安全なエラー応答 | `CustomExceptionHandlerMiddleware.cs` |

### 入力検証

FluentValidationを使用した入力検証パイプラインを実装：

| コンポーネント | 説明 | 実装ファイル |
| --- | --- | --- |
| RequestValidationBehavior | MediatRパイプラインでの検証実行 | `RequestValidationBehavior.cs` |
| CreateCustomerCommandValidator | 顧客作成時の検証 | `CreateCustomerCommandValidator.cs` |
| UpdateCustomerCommandValidator | 顧客更新時の検証 | `UpdateCustomerCommandValidator.cs` |
| GetCustomerDetailQueryValidator | 顧客詳細取得時の検証 | `GetCustomerDetailQueryValidator.cs` |
| DeleteCustomerCommandValidator | 顧客削除時の検証 | `DeleteCustomerCommandValidator.cs` |

#### 検証ルール例（CreateCustomerCommandValidator）

| フィールド | 検証ルール |
| --- | --- |
| Id | 5文字固定、必須 |
| CompanyName | 最大40文字、必須 |
| ContactName | 最大30文字 |
| Address | 最大60文字 |
| PostalCode | 最大10文字、必須 |

## 監査ログ

| ログ種別 | 記録内容 | 実装箇所 |
| --- | --- | --- |
| リクエストログ | リクエスト名、ユーザーID、リクエスト内容 | `RequestLogger.cs` |
| パフォーマンスログ | 500ms以上の長時間リクエスト | `RequestPerformanceBehaviour.cs` |
| 認証ログ | ログイン成功/失敗、ロックアウト | `Login.cshtml.cs` |
| ユーザー作成ログ | 新規アカウント作成 | `Register.cshtml.cs` |
| データ変更監査 | 作成者、作成日時、更新者、更新日時 | `AuditableEntity.cs`, `NorthwindDbContext.cs` |

### リクエストログ形式

```
Northwind Request: {RequestName} {UserId} {Request}
```

### パフォーマンス警告ログ形式

```
Northwind Long Running Request: {RequestName} ({ElapsedMilliseconds} milliseconds) {UserId} {Request}
```

### データ変更監査

`AuditableEntity`を継承したエンティティは、`SaveChangesAsync`時に自動的に以下の情報が記録されます：

- `CreatedBy`: 作成ユーザーID
- `Created`: 作成日時
- `LastModifiedBy`: 最終更新ユーザーID
- `LastModified`: 最終更新日時

## 例外処理

カスタム例外ハンドラー（`CustomExceptionHandlerMiddleware`）による安全なエラー応答：

| 例外種別 | HTTPステータス | 応答内容 |
| --- | --- | --- |
| ValidationException | 400 Bad Request | 検証エラー詳細（フィールド別） |
| BadRequestException | 400 Bad Request | エラーメッセージ |
| NotFoundException | 404 Not Found | なし |
| その他の例外 | 500 Internal Server Error | 汎用エラーメッセージ |

## 開発環境のセキュリティ

開発環境では以下の追加機能が有効になります：

| 機能 | 説明 |
| --- | --- |
| DeveloperExceptionPage | 詳細な例外情報表示 |
| DatabaseErrorPage | データベースエラー詳細表示 |
| RegisteredServicesPage | DIコンテナ登録サービス一覧 |

**注意**: これらの機能は本番環境では無効化されます（`Environment.IsDevelopment()`チェック）。

## テスト環境のセキュリティ

テスト環境では以下のテストユーザーが設定されています：

| 項目 | 値 |
| --- | --- |
| SubjectId | f26da293-02fb-4c90-be75-e4aa51e0bb17 |
| Username | jason@northwind |
| Password | Northwind1! |
| Grant Type | ResourceOwnerPassword |

## 備考

### セキュリティ上の推奨事項

1. **CORS設定**: 現在CORS設定が明示的に行われていません。本番環境では適切なCORS設定を検討してください。

2. **CSP（Content Security Policy）**: Content Security Policyヘッダーの設定が確認できません。XSS対策強化のため、適切なCSP設定を検討してください。

3. **レート制限**: API呼び出しに対するレート制限が実装されていません。DDoS対策として検討してください。

4. **ロールベースアクセス制御**: 現在は認証済み/未認証の2レベルの認可のみです。より細かいロールベースのアクセス制御の実装を検討してください。

### 依存パッケージ

| パッケージ | バージョン | 用途 |
| --- | --- | --- |
| Microsoft.AspNetCore.ApiAuthorization.IdentityServer | 3.0.0 | IdentityServer統合 |
| Microsoft.AspNetCore.Identity.UI | 3.0.0 | Identity UI |
| FluentValidation.AspNetCore | 8.5.0 | 入力検証 |
| oidc-client | - | フロントエンド認証（Angular） |
