# 機能設計書 1-ユーザー登録

## 概要

本ドキュメントは、NorthwindTradersシステムにおけるユーザー登録機能の設計仕様を記載する。ASP.NET Core Identityを使用した新規ユーザーアカウント作成機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：NorthwindTradersシステムを利用するためには、ユーザーアカウントが必要である。本機能は、新規ユーザーがシステムにアクセスするための最初のステップとして、アカウント作成を可能にする。これにより、顧客管理や従業員管理などの認証が必要な機能へのアクセス制御が実現される。

**機能の利用シーン**：新規ユーザーがNorthwindTradersシステムを初めて利用する際に、新規登録画面（サーバー）からアカウント作成を行う。管理者が新しい従業員用のアカウントを作成する場合にも使用される。

**主要な処理内容**：
1. ユーザーがメールアドレス、パスワード、確認用パスワードを入力
2. 入力値のバリデーション（メールアドレス形式、パスワード長、パスワード一致確認）
3. UserManagerを使用したユーザーアカウント作成
4. メール確認トークンの生成
5. 確認メールの送信
6. アカウント確認設定に応じた後続処理（確認待ちページへのリダイレクト、または自動ログイン）

**関連システム・外部連携**：IEmailSenderインターフェースを通じたメール送信サービスとの連携がある。外部認証プロバイダー（OAuth）との連携も可能。

**権限による制御**：本機能は[AllowAnonymous]属性により、未認証ユーザーでもアクセス可能である。アカウント作成後の機能アクセスは、メール確認の完了状況に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | 新規登録画面（サーバー） | 主画面 | UserManager.CreateAsync()を使用して新規ユーザーアカウントを作成する |

## 機能種別

CRUD操作（Create） / 認証処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Email | string | Yes | ユーザーのメールアドレス | Required, EmailAddress形式 |
| Password | string | Yes | パスワード | Required, 6〜100文字 |
| ConfirmPassword | string | No | 確認用パスワード | Passwordと一致すること |
| returnUrl | string | No | 登録後のリダイレクト先URL | - |

### 入力データソース

Razor Pages形式の入力フォーム（Register.cshtml）からのPOSTリクエスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| IdentityResult | IdentityResult | ユーザー作成の成功/失敗結果 |
| UserId | string | 作成されたユーザーのID（成功時） |
| Errors | IEnumerable<IdentityError> | エラー情報（失敗時） |

### 出力先

- 成功時：RegisterConfirmationページへのリダイレクト、または自動ログイン後にreturnUrlへリダイレクト
- 失敗時：同一ページ（Register）にエラーメッセージを表示

## 処理フロー

### 処理シーケンス

```
1. OnGetAsync - 画面初期表示
   └─ 外部認証スキーム一覧を取得してExternalLoginsに設定
2. OnPostAsync - フォーム送信時
   ├─ ModelState.IsValidでバリデーション確認
   ├─ ApplicationUserオブジェクト作成（Email→UserName, Email）
   ├─ UserManager.CreateAsync()でユーザー作成
   │   └─ 成功時：
   │       ├─ ログ出力
   │       ├─ メール確認トークン生成
   │       ├─ 確認URLをBase64エンコード
   │       ├─ IEmailSender.SendEmailAsync()で確認メール送信
   │       └─ RequireConfirmedAccount設定による分岐
   │           ├─ true → RegisterConfirmationページへリダイレクト
   │           └─ false → SignInAsync()で自動ログイン後、returnUrlへリダイレクト
   └─ 失敗時：エラーをModelStateに追加して再表示
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[OnGetAsync: 外部認証スキーム取得]
    B --> C[フォーム表示]
    C --> D{POSTリクエスト?}
    D -->|No| C
    D -->|Yes| E{ModelState有効?}
    E -->|No| F[エラー表示]
    F --> C
    E -->|Yes| G[ApplicationUser作成]
    G --> H[UserManager.CreateAsync]
    H --> I{作成成功?}
    I -->|No| J[エラーをModelStateに追加]
    J --> F
    I -->|Yes| K[メール確認トークン生成]
    K --> L[確認メール送信]
    L --> M{RequireConfirmedAccount?}
    M -->|Yes| N[RegisterConfirmationへリダイレクト]
    M -->|No| O[SignInAsync自動ログイン]
    O --> P[returnUrlへリダイレクト]
    N --> Q[終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | メールアドレス一意性 | 同一メールアドレスでのユーザー登録は不可 | 常時 |
| BR-002 | パスワード強度 | パスワードは6文字以上100文字以内 | 常時 |
| BR-003 | メール確認要否 | RequireConfirmedAccountがtrueの場合、メール確認完了までログイン不可 | 設定依存 |
| BR-004 | ユーザー名設定 | ユーザー名としてメールアドレスを使用 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー登録 | AspNetUsers | INSERT | 新規ユーザーレコード作成 |

### テーブル別操作詳細

#### AspNetUsers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | Id | 自動生成GUID | |
| INSERT | UserName | Input.Email | メールアドレスをユーザー名として使用 |
| INSERT | NormalizedUserName | Input.Email.ToUpper() | 正規化されたユーザー名 |
| INSERT | Email | Input.Email | |
| INSERT | NormalizedEmail | Input.Email.ToUpper() | 正規化されたメールアドレス |
| INSERT | PasswordHash | ハッシュ化されたパスワード | Identity内部でハッシュ化 |
| INSERT | EmailConfirmed | false | 初期値、メール確認後にtrue |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| DuplicateUserName | バリデーションエラー | 同一メールアドレスが既に登録済み | 別のメールアドレスで再試行 |
| PasswordTooShort | バリデーションエラー | パスワードが6文字未満 | 6文字以上のパスワードを入力 |
| PasswordMismatch | バリデーションエラー | パスワードと確認用パスワードが不一致 | パスワードを再入力 |
| InvalidEmail | バリデーションエラー | メールアドレス形式が不正 | 正しい形式で入力 |

### リトライ仕様

リトライは不要。ユーザーが再度フォームを送信する。

## トランザクション仕様

UserManager.CreateAsync()内部でトランザクション管理される。ユーザー作成は単一のアトミック操作として実行される。

## パフォーマンス要件

- レスポンス時間：3秒以内（メール送信を含む）
- メール送信は非同期で実行

## セキュリティ考慮事項

- パスワードはASP.NET Core Identityによりハッシュ化されて保存
- CSRF対策としてAntiForgeryTokenを使用（Razor Pages標準機能）
- メール確認トークンはBase64Urlエンコードにより安全に送信
- [AllowAnonymous]属性により未認証アクセスを許可

## 備考

- ASP.NET Core Identity 3.0を使用
- 外部認証プロバイダー（Google、Facebook等）との連携も可能（ExternalLogins）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ユーザーデータを表すApplicationUserクラスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | IdentityUserを継承したユーザーエンティティ。現時点では拡張プロパティなし |

**読解のコツ**: ApplicationUserはASP.NET Core IdentityのIdentityUserを継承している。IdentityUserには既にId, UserName, Email, PasswordHash等のプロパティが定義されている。

#### Step 2: エントリーポイントを理解する

Razor PagesのPageModelであるRegisterModelがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | ユーザー登録のメインロジック |

**主要処理フロー**:
1. **28-38行目**: コンストラクタでDI経由でUserManager, SignInManager, ILogger, IEmailSenderを取得
2. **47-64行目**: InputModelクラスでフォーム入力のデータ構造とバリデーション属性を定義
3. **66-70行目**: OnGetAsync()で初期表示処理（外部認証スキーム取得）
4. **72-113行目**: OnPostAsync()でユーザー登録処理を実行
5. **78-79行目**: ApplicationUserオブジェクト作成
6. **79行目**: UserManager.CreateAsync()でユーザー作成
7. **84-93行目**: メール確認トークン生成とメール送信
8. **95-103行目**: RequireConfirmedAccountに応じた後続処理

#### Step 3: 依存サービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Startup.cs | `Src/WebUI/Startup.cs` | DIコンテナへのサービス登録、Identity設定 |

**主要処理フロー**:
- **35行目**: AddInfrastructure()でIdentity関連サービスを登録

### プログラム呼び出し階層図

```
Register.cshtml (View)
    │
    └─ RegisterModel (PageModel)
           │
           ├─ UserManager<ApplicationUser>.CreateAsync()
           │      └─ AspNetUsers テーブルへINSERT
           │
           ├─ UserManager<ApplicationUser>.GenerateEmailConfirmationTokenAsync()
           │
           ├─ IEmailSender.SendEmailAsync()
           │
           └─ SignInManager<ApplicationUser>.SignInAsync()
                  └─ 認証Cookie発行
```

### データフロー図

```
[入力]                    [処理]                              [出力]

Email            ───▶  InputModel                    ───▶  AspNetUsersテーブル
Password         ───▶  RegisterModel.OnPostAsync()   ───▶  確認メール送信
ConfirmPassword  ───▶  UserManager.CreateAsync()     ───▶  リダイレクト先画面
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | ソース | ユーザー登録のPageModel |
| Register.cshtml | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml` | ビュー | 登録フォームのRazor Page |
| ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | ソース | ユーザーエンティティ |
| Startup.cs | `Src/WebUI/Startup.cs` | ソース | サービス登録・設定 |
| IdentityHostingStartup.cs | `Src/WebUI/Areas/Identity/IdentityHostingStartup.cs` | ソース | Identity設定 |
