# 機能設計書 10-商品詳細取得

## 概要

本ドキュメントは、NorthwindTradersシステムにおける商品詳細取得機能の設計仕様を記載する。指定した商品IDに基づいて商品の詳細情報を取得するAPI機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：商品管理業務において、個別の商品情報を詳細に確認する必要がある。商品一覧画面から特定の商品を選択した際に、商品名、単価、カテゴリ、仕入先、在庫状況などの詳細情報を表示するために本機能が使用される。営業担当者が顧客への商品説明を行う際や、在庫管理担当者が商品状態を確認する際に活用される。

**機能の利用シーン**：
1. 商品一覧画面から特定の商品を選択して詳細を表示する場合
2. 商品の編集・削除を行う前に現在の状態を確認する場合
3. 外部システムから商品情報を参照する場合（API経由）

**主要な処理内容**：
1. クライアントから商品ID（整数）を受け取る
2. MediatRパターンを使用してGetProductDetailQueryを発行
3. データベースからProductsテーブルを検索し、関連するCategory、Supplierも含めて取得
4. AutoMapperを使用してProductDetailVmにマッピング
5. 商品が存在しない場合はNotFoundExceptionをスロー
6. JSONレスポンスとして商品詳細情報を返却

**関連システム・外部連携**：本機能は独立したREST APIとして動作し、SPAフロントエンド（Angular）から呼び出される。

**権限による制御**：本機能は`[AllowAnonymous]`属性により認証なしでアクセス可能である。商品情報は公開情報として扱われる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品一覧画面 | 参照元画面 | 商品一覧から詳細表示をトリガー |

## 機能種別

CRUD操作（Read） / データ参照

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 取得対象の商品ID | URLパラメータとして指定、整数値 |

### 入力データソース

HTTPリクエスト GET /api/Products/Get/{id}

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ProductId | int | 商品ID |
| ProductName | string | 商品名 |
| UnitPrice | decimal? | 単価 |
| SupplierId | int? | 仕入先ID |
| SupplierCompanyName | string | 仕入先会社名 |
| CategoryId | int? | カテゴリID |
| CategoryName | string | カテゴリ名 |
| Discontinued | bool | 販売終了フラグ |
| EditEnabled | bool | 編集可能フラグ |
| DeleteEnabled | bool | 削除可能フラグ |

### 出力先

- 成功時：HTTP 200 OK、JSON形式のProductDetailVm
- 失敗時（商品未存在）：HTTP 404 Not Found

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信（GET /api/Products/Get/{id}）
   └─ ProductsController.Get(id)が呼び出される
2. GetProductDetailQueryオブジェクト作成
   └─ Id = リクエストパラメータのid
3. MediatR.Send()でクエリ発行
   └─ GetProductDetailQueryHandlerが処理
4. データベースクエリ実行
   ├─ INorthwindDbContext.Products.ProjectTo<ProductDetailVm>()
   ├─ AutoMapperによるマッピング（Category、Supplier含む）
   └─ FirstOrDefaultAsync(p => p.ProductId == request.Id)
5. 結果判定
   ├─ 商品存在：ProductDetailVmを返却
   └─ 商品不在：NotFoundException("Product", id)をスロー
6. HTTP応答
   ├─ 成功：200 OK + JSON
   └─ 失敗：404 Not Found
```

### フローチャート

```mermaid
flowchart TD
    A[開始: GET /api/Products/Get/id] --> B[ProductsController.Get]
    B --> C[GetProductDetailQuery作成]
    C --> D[MediatR.Send]
    D --> E[GetProductDetailQueryHandler.Handle]
    E --> F[Products.ProjectTo.FirstOrDefaultAsync]
    F --> G{商品存在?}
    G -->|Yes| H[ProductDetailVm返却]
    G -->|No| I[NotFoundException]
    H --> J[HTTP 200 OK]
    I --> K[HTTP 404 Not Found]
    J --> L[終了]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 商品存在チェック | 指定されたIDの商品が存在しない場合は404エラー | 常時 |
| BR-002 | 関連情報取得 | 商品に関連するカテゴリ名、仕入先会社名も取得して返却 | 常時 |
| BR-003 | 認証不要 | 商品詳細は公開情報として認証なしでアクセス可能 | 常時 |
| BR-004 | 編集・削除フラグ | EditEnabled、DeleteEnabledは現時点では未設定（Ignore） | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品取得 | Products | SELECT | 指定IDの商品を取得 |
| カテゴリ取得 | Categories | SELECT (JOIN) | 商品のカテゴリ情報を取得 |
| 仕入先取得 | Suppliers | SELECT (JOIN) | 商品の仕入先情報を取得 |

### テーブル別操作詳細

#### Products

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | ProductId | WHERE ProductId = @id | 主キー検索 |
| SELECT | ProductName | - | 商品名 |
| SELECT | UnitPrice | - | 単価 |
| SELECT | SupplierId | - | 仕入先FK |
| SELECT | CategoryId | - | カテゴリFK |
| SELECT | Discontinued | - | 販売終了フラグ |

#### Categories (JOIN)

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CategoryName | JOIN on CategoryId | カテゴリ名 |

#### Suppliers (JOIN)

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CompanyName | JOIN on SupplierId | 仕入先会社名（SupplierCompanyNameとして返却） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NotFound | NotFoundException | 指定IDの商品が存在しない | 正しい商品IDで再試行 |
| InvalidId | バリデーションエラー | IDが整数でない | 正しい形式のIDを指定 |

### リトライ仕様

読み取り専用操作のため、リトライは不要。

## トランザクション仕様

読み取り専用操作のため、トランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間：500ms以内
- ProjectTo<>によるクエリ最適化（必要なカラムのみ取得）

## セキュリティ考慮事項

- `[AllowAnonymous]`属性により認証なしでアクセス可能
- 商品情報は公開情報として扱われるため、認証は不要
- SQLインジェクション対策：Entity Frameworkのパラメータ化クエリにより保護

## 備考

- AutoMapperによるマッピングで、Supplier?.CompanyName、Category?.CategoryNameがnullの場合は空文字列を返却
- EditEnabled、DeleteEnabledは現在のマッピング設定では無視（Ignore）されており、将来の権限制御のために予約されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、商品データを表すエンティティとDTOを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Product.cs | `Src/Domain/Entities/Product.cs` | 商品エンティティの定義。AuditableEntityを継承 |
| 1-2 | ProductDetailVm.cs | `Src/Application/Products/Queries/GetProductDetail/ProductDetailVm.cs` | レスポンスのビューモデル。AutoMapperマッピング設定を含む |
| 1-3 | GetProductDetailQuery.cs | `Src/Application/Products/Queries/GetProductDetail/GetProductDetailQuery.cs` | クエリオブジェクトの定義（Idプロパティのみ） |

**読解のコツ**:
- ProductエンティティはCategory、Supplierへのナビゲーションプロパティを持つ
- ProductDetailVmのMapping()メソッドでAutoMapperの設定が定義されている
- EditEnabled、DeleteEnabledはIgnore()されており、現時点では使用されていない

#### Step 2: エントリーポイントを理解する

APIコントローラーから処理が開始される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | REST APIエンドポイント |

**主要処理フロー**:
1. **26-33行目**: Get(int id)メソッド - [HttpGet("{id}")]、[AllowAnonymous]属性付き
2. **30行目**: GetProductDetailQueryを作成しMediatR.Send()で発行
3. **32行目**: Ok(vm)でJSONレスポンスを返却

#### Step 3: クエリハンドラを理解する

MediatRパターンによるクエリ処理の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GetProductDetailQueryHandler.cs | `Src/Application/Products/Queries/GetProductDetail/GetProductDetailQueryHandler.cs` | クエリの実際の処理ロジック |

**主要処理フロー**:
1. **12-20行目**: コンストラクタでINorthwindDbContext、IMapperを注入
2. **23-35行目**: Handle()メソッド - クエリの実行
3. **25-27行目**: ProjectTo<ProductDetailVm>でAutoMapperマッピングを適用しながらDBクエリ実行
4. **29-32行目**: 結果がnullの場合はNotFoundException("Product", request.Id)をスロー
5. **34行目**: ProductDetailVmを返却

#### Step 4: インフラストラクチャ層を理解する

データベースコンテキストとエンティティ設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | DbContextインターフェース定義 |
| 4-2 | ProductConfiguration.cs | `Src/Persistence/Configurations/ProductConfiguration.cs` | Productテーブルの設定 |

### プログラム呼び出し階層図

```
GET /api/Products/Get/{id}
    │
    └─ ProductsController.Get(id)
           │
           ├─ new GetProductDetailQuery { Id = id }
           │
           └─ Mediator.Send(query)
                  │
                  └─ GetProductDetailQueryHandler.Handle()
                         │
                         ├─ _context.Products.ProjectTo<ProductDetailVm>()
                         │      │
                         │      ├─ AutoMapper - Product → ProductDetailVm
                         │      │      ├─ Supplier?.CompanyName → SupplierCompanyName
                         │      │      └─ Category?.CategoryName → CategoryName
                         │      │
                         │      └─ FirstOrDefaultAsync(p => p.ProductId == request.Id)
                         │
                         └─ null判定
                                ├─ 存在: return vm
                                └─ 不在: throw NotFoundException
```

### データフロー図

```
[入力]                    [処理]                              [出力]

id (int)         ───▶  ProductsController.Get()       ───▶  HTTP Response
  │                          │
  │                          ▼
  │              GetProductDetailQuery
  │                          │
  │                          ▼
  │              GetProductDetailQueryHandler
  │                          │
  │                          ▼
  └──────────▶  Products + Categories + Suppliers   ───▶  ProductDetailVm (JSON)
               (Entity Framework LINQ Query)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | ソース | REST APIコントローラー |
| GetProductDetailQuery.cs | `Src/Application/Products/Queries/GetProductDetail/GetProductDetailQuery.cs` | ソース | クエリオブジェクト定義 |
| GetProductDetailQueryHandler.cs | `Src/Application/Products/Queries/GetProductDetail/GetProductDetailQueryHandler.cs` | ソース | クエリハンドラ |
| ProductDetailVm.cs | `Src/Application/Products/Queries/GetProductDetail/ProductDetailVm.cs` | ソース | レスポンスDTO |
| Product.cs | `Src/Domain/Entities/Product.cs` | ソース | 商品エンティティ |
| Category.cs | `Src/Domain/Entities/Category.cs` | ソース | カテゴリエンティティ |
| Supplier.cs | `Src/Domain/Entities/Supplier.cs` | ソース | 仕入先エンティティ |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DbContextインターフェース |
| BaseController.cs | `Src/WebUI/Controllers/BaseController.cs` | ソース | コントローラー基底クラス |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
