# 機能設計書 11-商品登録

## 概要

本ドキュメントは、NorthwindTradersシステムにおける商品登録機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、新規商品をデータベースに登録するためのAPIエンドポイントを提供する。商品名、単価、仕入先ID、カテゴリID、販売終了フラグを指定して新しい商品レコードを作成し、生成された商品IDを返却する。

**業務上の目的・背景**：商品マスタを適切に管理することは、受発注業務や在庫管理の基盤となる。新規商品を取り扱う際には、商品情報をシステムに登録する必要があり、この機能によって商品マスタへの新規データ投入を行う。

**機能の利用シーン**：新規取扱商品が発生した場合に、管理者または商品担当者が商品情報を登録する。仕入先から新商品の入荷通知があった際、商品一覧に追加するために使用される。

**主要な処理内容**：
1. APIリクエストから商品情報を受け取る
2. CreateProductCommandを通じてMediatRパイプラインへディスパッチ
3. CreateProductCommandHandlerがProductエンティティを作成
4. Entity Framework Coreを通じてProductsテーブルにINSERT
5. 自動生成された商品ID（ProductId）を返却

**関連システム・外部連携**：特になし。本機能は内部APIとして動作し、外部システムとの連携は行わない。

**権限による制御**：ProductsControllerには[Authorize]属性が付与されており、認証済みユーザーのみがこの機能を利用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 新規登録（Create）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ProductName | string | Yes | 商品名 | 最大40文字、必須 |
| UnitPrice | decimal? | No | 単価 | null許容 |
| SupplierId | int? | No | 仕入先ID | null許容、Suppliersテーブルの外部キー |
| CategoryId | int? | No | カテゴリID | null許容、Categoriesテーブルの外部キー |
| Discontinued | bool | Yes | 販売終了フラグ | true/false |

### 入力データソース

HTTPリクエストボディ（JSON形式）からCreateProductCommandオブジェクトにバインドされる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ProductId | int | 新規作成された商品のID |

### 出力先

HTTPレスポンス（JSON形式）として返却される。

## 処理フロー

### 処理シーケンス

```
1. ProductsController.Create()がHTTP POSTリクエストを受信
   └─ [FromBody]属性によりリクエストボディからCreateProductCommandを生成
2. MediatRを通じてCreateProductCommandをディスパッチ
   └─ Mediator.Send(command)を呼び出し
3. CreateProductCommandHandlerがコマンドを処理
   └─ Productエンティティを新規作成
4. INorthwindDbContextを通じてデータベースに保存
   └─ _context.Products.Add(entity)
   └─ _context.SaveChangesAsync()
5. 生成された商品IDを返却
   └─ return entity.ProductId
```

### フローチャート

```mermaid
flowchart TD
    A[HTTP POST /api/Products] --> B[CreateProductCommand生成]
    B --> C[MediatR.Send]
    C --> D[CreateProductCommandHandler.Handle]
    D --> E[Productエンティティ作成]
    E --> F[Products.Add]
    F --> G[SaveChangesAsync]
    G --> H[ProductId返却]
    H --> I[HTTP 200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 商品名必須 | 商品名は必ず指定する必要がある | 常に適用 |
| BR-02 | 単価デフォルト | 単価が指定されない場合、データベースデフォルト値0が設定される | UnitPriceがnullの場合 |
| BR-03 | 販売終了フラグデフォルト | Discontinuedはfalseがデフォルト | 明示的に指定されない場合 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品登録 | Products | INSERT | 新規商品レコードを挿入 |

### テーブル別操作詳細

#### Products

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | ProductID | 自動生成（IDENTITY） | 主キー |
| INSERT | ProductName | request.ProductName | 必須、最大40文字 |
| INSERT | CategoryID | request.CategoryId | null許容 |
| INSERT | SupplierID | request.SupplierId | null許容 |
| INSERT | UnitPrice | request.UnitPrice | money型、デフォルト0 |
| INSERT | Discontinued | request.Discontinued | bit型 |
| INSERT | CreatedBy | 現在ユーザーID | AuditableEntity |
| INSERT | Created | 現在日時 | AuditableEntity |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証ユーザーからのリクエスト | ログイン後に再実行 |
| 400 | BadRequest | 商品名が未指定または40文字超過 | 正しい商品名を指定 |
| 500 | InternalServerError | データベース接続エラー等 | システム管理者に連絡 |

### リトライ仕様

特になし。エラー発生時はクライアント側で再実行を行う。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。Entity Framework Coreのデフォルトトランザクション管理を使用。

## パフォーマンス要件

特に明示的な要件なし。通常のCRUD操作として数百ミリ秒以内のレスポンスを期待。

## セキュリティ考慮事項

- [Authorize]属性により認証済みユーザーのみアクセス可能
- 入力値はEntity Frameworkによりパラメータ化されSQLインジェクションを防止
- ProductNameの長さ制限によりバッファオーバーフローを防止

## 備考

- 現在、バリデーションロジック（FluentValidation等）は実装されていない
- 将来的にはカテゴリID・仕入先IDの存在チェックを追加することが望ましい

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Productエンティティの構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Product.cs | `Src/Domain/Entities/Product.cs` | 商品エンティティの全プロパティを確認。AuditableEntityを継承していることに注目 |
| 1-2 | ProductConfiguration.cs | `Src/Persistence/Configurations/ProductConfiguration.cs` | データベースカラムのマッピングと制約を確認 |

**読解のコツ**: Product.csではAuditableEntityを継承しているため、Created/CreatedBy/LastModified/LastModifiedByプロパティが自動的に付与される。

#### Step 2: エントリーポイントを理解する

APIコントローラーがリクエストを受け付ける起点。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | Create()メソッドの実装を確認。[Authorize]と[HttpPost]属性に注目 |

**主要処理フロー**:
1. **35-41行目**: Create()メソッドがCreateProductCommandを受け取りMediatRへディスパッチ
2. **14行目**: [Authorize]属性によりコントローラー全体が認証必須

#### Step 3: コマンドとハンドラーを理解する

CQRSパターンに基づくコマンド処理。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CreateProductCommand.cs | `Src/Application/Products/Commands/CreateProduct/CreateProductCommand.cs` | 入力パラメータの定義。IRequest<int>を実装し、商品IDを返却 |
| 3-2 | CreateProductCommandHandler.cs | `Src/Application/Products/Commands/CreateProduct/CreateProductCommandHandler.cs` | 実際の登録処理ロジック |

**主要処理フロー**:
- **18-34行目**: Handle()メソッドでProductエンティティを作成しデータベースに保存
- **20-27行目**: Productエンティティの各プロパティにリクエスト値をマッピング
- **29行目**: DbContextへのAdd()呼び出し
- **31行目**: SaveChangesAsync()でコミット

### プログラム呼び出し階層図

```
ProductsController.Create()
    │
    ├─ Mediator.Send(CreateProductCommand)
    │      │
    │      └─ CreateProductCommandHandler.Handle()
    │             │
    │             ├─ new Product() - エンティティ作成
    │             │
    │             ├─ _context.Products.Add() - DbSetへ追加
    │             │
    │             └─ _context.SaveChangesAsync() - データベースコミット
    │
    └─ return Ok(productId)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTP POST Body    ───▶  CreateProductCommand     ───▶  HTTP Response
(JSON)                        │                         (ProductId)
                              │
                              ▼
                    CreateProductCommandHandler
                              │
                              ▼
                    Product Entity作成
                              │
                              ▼
                    INorthwindDbContext.Products
                              │
                              ▼
                    SaveChangesAsync()
                              │
                              ▼
                    Productsテーブル (INSERT)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | ソース | APIエンドポイント定義 |
| CreateProductCommand.cs | `Src/Application/Products/Commands/CreateProduct/CreateProductCommand.cs` | ソース | コマンドモデル定義 |
| CreateProductCommandHandler.cs | `Src/Application/Products/Commands/CreateProduct/CreateProductCommandHandler.cs` | ソース | コマンド処理ハンドラー |
| Product.cs | `Src/Domain/Entities/Product.cs` | ソース | ドメインエンティティ |
| ProductConfiguration.cs | `Src/Persistence/Configurations/ProductConfiguration.cs` | ソース | EF Coreマッピング設定 |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DbContextインターフェース |
