# 機能設計書 12-商品更新

## 概要

本ドキュメントは、NorthwindTradersシステムにおける商品更新機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、既存の商品情報を更新するためのAPIエンドポイントを提供する。商品ID、商品名、単価、仕入先ID、カテゴリID、販売終了フラグを指定して既存の商品レコードを更新する。指定された商品IDが存在しない場合はNotFoundExceptionをスローする。

**業務上の目的・背景**：商品情報は価格改定、カテゴリ変更、販売終了設定など、運用中に頻繁に更新が必要となる。本機能により、商品マスタの変更を適切にシステムへ反映し、最新の商品情報を維持する。

**機能の利用シーン**：商品の単価変更、カテゴリの再分類、仕入先の変更、販売終了処理など、商品情報の修正が必要な際に管理者が使用する。

**主要な処理内容**：
1. APIリクエストから更新対象の商品情報を受け取る
2. UpdateProductCommandを通じてMediatRパイプラインへディスパッチ
3. UpdateProductCommandHandlerが既存商品を検索
4. 商品が存在しない場合はNotFoundExceptionをスロー
5. Entity Framework Coreを通じてProductsテーブルをUPDATE
6. 処理完了後、NoContent（204）を返却

**関連システム・外部連携**：特になし。本機能は内部APIとして動作する。

**権限による制御**：ProductsControllerには[Authorize]属性が付与されており、認証済みユーザーのみがこの機能を利用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 更新（Update）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ProductId | int | Yes | 更新対象の商品ID | 存在チェック |
| ProductName | string | Yes | 商品名 | 最大40文字、必須 |
| UnitPrice | decimal? | No | 単価 | null許容 |
| SupplierId | int? | No | 仕入先ID | null許容 |
| CategoryId | int? | No | カテゴリID | null許容 |
| Discontinued | bool | Yes | 販売終了フラグ | true/false |

### 入力データソース

HTTPリクエストボディ（JSON形式）からUpdateProductCommandオブジェクトにバインドされる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | 成功時はレスポンスボディなし（204 No Content） |

### 出力先

HTTPレスポンスステータスコード。

## 処理フロー

### 処理シーケンス

```
1. ProductsController.Update()がHTTP PUTリクエストを受信
   └─ [FromBody]属性によりリクエストボディからUpdateProductCommandを生成
2. MediatRを通じてUpdateProductCommandをディスパッチ
   └─ Mediator.Send(command)を呼び出し
3. UpdateProductCommandHandlerがコマンドを処理
   └─ FindAsync()で既存商品を検索
4. 商品存在チェック
   └─ 存在しない場合：NotFoundException("Product", productId)
5. エンティティのプロパティを更新
   └─ 各プロパティにリクエスト値をマッピング
6. INorthwindDbContextを通じてデータベースに保存
   └─ _context.SaveChangesAsync()
7. NoContent（204）を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTP PUT /api/Products] --> B[UpdateProductCommand生成]
    B --> C[MediatR.Send]
    C --> D[UpdateProductCommandHandler.Handle]
    D --> E{商品存在チェック}
    E -->|存在しない| F[NotFoundException]
    E -->|存在する| G[エンティティ更新]
    G --> H[SaveChangesAsync]
    H --> I[HTTP 204 No Content]
    F --> J[HTTP 404 Not Found]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 商品存在必須 | 更新対象の商品がデータベースに存在する必要がある | 常に適用 |
| BR-02 | 商品名必須 | 商品名は必ず指定する必要がある | 常に適用 |
| BR-03 | 全項目上書き | 指定された値で全てのフィールドが上書きされる | 常に適用 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品検索 | Products | SELECT | ProductIdで既存商品を検索 |
| 商品更新 | Products | UPDATE | 商品情報を更新 |

### テーブル別操作詳細

#### Products

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | ProductID | WHERE ProductID = request.ProductId | 主キー検索 |
| UPDATE | ProductID | request.ProductId | 変更不可だが代入される |
| UPDATE | ProductName | request.ProductName | 必須 |
| UPDATE | CategoryID | request.CategoryId | null許容 |
| UPDATE | SupplierID | request.SupplierId | null許容 |
| UPDATE | UnitPrice | request.UnitPrice | null許容 |
| UPDATE | Discontinued | request.Discontinued | bool |
| UPDATE | LastModifiedBy | 現在ユーザーID | AuditableEntity |
| UPDATE | LastModified | 現在日時 | AuditableEntity |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証ユーザーからのリクエスト | ログイン後に再実行 |
| 404 | NotFound | 指定されたProductIdが存在しない | 正しい商品IDを指定 |
| 400 | BadRequest | 商品名が未指定 | 正しい商品名を指定 |
| 500 | InternalServerError | データベース接続エラー等 | システム管理者に連絡 |

### リトライ仕様

特になし。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。Entity Framework Coreのデフォルトトランザクション管理を使用。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- [Authorize]属性により認証済みユーザーのみアクセス可能
- 入力値はEntity Frameworkによりパラメータ化されSQLインジェクションを防止

## 備考

- コード中でProductIdへの代入（28行目）は冗長だが、明示的な設定として実装されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Product.cs | `Src/Domain/Entities/Product.cs` | 更新対象となるエンティティの全プロパティを確認 |
| 1-2 | ProductConfiguration.cs | `Src/Persistence/Configurations/ProductConfiguration.cs` | 各カラムの制約を確認 |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | Update()メソッドの実装を確認 |

**主要処理フロー**:
1. **43-51行目**: Update()メソッドがUpdateProductCommandを受け取りMediatRへディスパッチ
2. **44行目**: [ProducesResponseType(StatusCodes.Status204NoContent)]により成功時204を返却

#### Step 3: コマンドとハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UpdateProductCommand.cs | `Src/Application/Products/Commands/UpdateProduct/UpdateProductCommand.cs` | 入力パラメータの定義。IRequestを実装（戻り値なし） |
| 3-2 | UpdateProductCommandHandler.cs | `Src/Application/Products/Commands/UpdateProduct/UpdateProductCommandHandler.cs` | 実際の更新処理ロジック |

**主要処理フロー**:
- **21行目**: FindAsync()で既存商品を検索
- **23-26行目**: 商品が存在しない場合NotFoundException
- **28-33行目**: エンティティの各プロパティを更新
- **35行目**: SaveChangesAsync()でコミット

### プログラム呼び出し階層図

```
ProductsController.Update()
    │
    ├─ Mediator.Send(UpdateProductCommand)
    │      │
    │      └─ UpdateProductCommandHandler.Handle()
    │             │
    │             ├─ _context.Products.FindAsync() - 商品検索
    │             │
    │             ├─ (商品なし) → throw NotFoundException
    │             │
    │             ├─ エンティティプロパティ更新
    │             │
    │             └─ _context.SaveChangesAsync() - コミット
    │
    └─ return NoContent()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTP PUT Body     ───▶  UpdateProductCommand     ───▶  HTTP 204
(JSON)                        │
                              │
                              ▼
                    UpdateProductCommandHandler
                              │
                              ▼
                    Products.FindAsync(ProductId)
                              │
                              ▼
                        存在チェック
                         /        \
                存在しない         存在する
                    │                │
                    ▼                ▼
            NotFoundException   エンティティ更新
                                     │
                                     ▼
                              SaveChangesAsync()
                                     │
                                     ▼
                              Productsテーブル (UPDATE)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | ソース | APIエンドポイント定義 |
| UpdateProductCommand.cs | `Src/Application/Products/Commands/UpdateProduct/UpdateProductCommand.cs` | ソース | コマンドモデル定義 |
| UpdateProductCommandHandler.cs | `Src/Application/Products/Commands/UpdateProduct/UpdateProductCommandHandler.cs` | ソース | コマンド処理ハンドラー |
| Product.cs | `Src/Domain/Entities/Product.cs` | ソース | ドメインエンティティ |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
