# 機能設計書 13-商品削除

## 概要

本ドキュメントは、NorthwindTradersシステムにおける商品削除機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、指定された商品IDに基づいて商品をデータベースから削除するAPIエンドポイントを提供する。削除前に、該当商品に関連する注文明細（OrderDetails）の存在をチェックし、存在する場合は削除を拒否してDeleteFailureExceptionをスローする。

**業務上の目的・背景**：商品マスタの整理において、不要な商品レコードを削除する必要がある。ただし、過去の注文履歴との整合性を保つため、注文実績のある商品は削除できないようにして、データの一貫性を保護する。

**機能の利用シーン**：取扱終了した商品で、かつ注文履歴が存在しない商品を完全に削除する際に使用する。通常は販売終了フラグ（Discontinued）を使用し、物理削除は限定的なケースでのみ使用される。

**主要な処理内容**：
1. APIリクエストから削除対象の商品IDを受け取る
2. DeleteProductCommandを通じてMediatRパイプラインへディスパッチ
3. DeleteProductCommandHandlerが既存商品を検索
4. 商品が存在しない場合はNotFoundExceptionをスロー
5. 関連する注文明細の存在チェック
6. 注文明細が存在する場合はDeleteFailureExceptionをスロー
7. Entity Framework Coreを通じてProductsテーブルからDELETE
8. 処理完了後、NoContent（204）を返却

**関連システム・外部連携**：特になし。

**権限による制御**：ProductsControllerには[Authorize]属性が付与されており、認証済みユーザーのみがこの機能を利用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 削除（Delete）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 削除対象の商品ID | URL パスパラメータ、存在チェック |

### 入力データソース

URLパスパラメータから商品IDを取得（/api/Products/{id}）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | 成功時はレスポンスボディなし（204 No Content） |

### 出力先

HTTPレスポンスステータスコード。

## 処理フロー

### 処理シーケンス

```
1. ProductsController.Delete()がHTTP DELETEリクエストを受信
   └─ URLパスから商品IDを取得
2. MediatRを通じてDeleteProductCommandをディスパッチ
   └─ Mediator.Send(new DeleteProductCommand { Id = id })
3. DeleteProductCommandHandlerがコマンドを処理
   └─ FindAsync()で既存商品を検索
4. 商品存在チェック
   └─ 存在しない場合：NotFoundException("Product", id)
5. 関連注文明細の存在チェック
   └─ OrderDetails.Any(od => od.ProductId == entity.ProductId)
6. 注文明細存在時
   └─ DeleteFailureException("Product", id, "There are existing orders...")
7. Products.Remove()で削除マーク
8. SaveChangesAsync()でDELETE実行
9. NoContent（204）を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTP DELETE /api/Products/id] --> B[DeleteProductCommand生成]
    B --> C[MediatR.Send]
    C --> D[DeleteProductCommandHandler.Handle]
    D --> E{商品存在チェック}
    E -->|存在しない| F[NotFoundException]
    E -->|存在する| G{注文明細存在チェック}
    G -->|存在する| H[DeleteFailureException]
    G -->|存在しない| I[Products.Remove]
    I --> J[SaveChangesAsync]
    J --> K[HTTP 204 No Content]
    F --> L[HTTP 404 Not Found]
    H --> M[HTTP 400 Bad Request]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 商品存在必須 | 削除対象の商品がデータベースに存在する必要がある | 常に適用 |
| BR-02 | 注文関連チェック | 注文明細に関連付けられている商品は削除不可 | 常に適用 |
| BR-03 | 物理削除 | 削除は物理削除として実行される（論理削除ではない） | 常に適用 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品検索 | Products | SELECT | ProductIdで既存商品を検索 |
| 注文明細チェック | OrderDetails | SELECT | 商品IDに関連する注文明細の存在確認 |
| 商品削除 | Products | DELETE | 商品レコードを物理削除 |

### テーブル別操作詳細

#### Products

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | ProductID | WHERE ProductID = request.Id | 主キー検索 |
| DELETE | 全カラム | WHERE ProductID = request.Id | 物理削除 |

#### OrderDetails

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | ProductID | WHERE ProductID = entity.ProductId | 存在チェック（Any） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証ユーザーからのリクエスト | ログイン後に再実行 |
| 404 | NotFound | 指定されたProductIdが存在しない | 正しい商品IDを指定 |
| 400 | DeleteFailure | 該当商品に注文明細が紐づいている | 関連注文を先に削除するか、商品をDiscontinued状態に変更 |
| 500 | InternalServerError | データベース接続エラー等 | システム管理者に連絡 |

### リトライ仕様

特になし。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- [Authorize]属性により認証済みユーザーのみアクセス可能
- 削除操作は監査ログ（AuditableEntity）により追跡可能な設計が望ましい（現在は未実装）

## 備考

- コード内のTODOコメントにより、DeleteFailureExceptionのテストが未実装であることが示されている（32行目）
- 削除の代わりにDiscontinuedフラグの使用が推奨される運用ケースが多い

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Product.cs | `Src/Domain/Entities/Product.cs` | OrderDetailsとのリレーションを確認（26行目） |
| 1-2 | OrderDetail.cs | `Src/Domain/Entities/OrderDetail.cs` | ProductIdプロパティを確認 |

**読解のコツ**: Product.csのOrderDetailsプロパティが削除可否チェックで参照される点に注目。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | Delete()メソッドの実装を確認 |

**主要処理フロー**:
1. **53-61行目**: Delete()メソッドがパスパラメータidを受け取りMediatRへディスパッチ
2. **54行目**: [ProducesResponseType(StatusCodes.Status204NoContent)]

#### Step 3: コマンドとハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DeleteProductCommand.cs | `Src/Application/Products/Commands/DeleteProduct/DeleteProductCommand.cs` | シンプルな入力パラメータ定義 |
| 3-2 | DeleteProductCommandHandler.cs | `Src/Application/Products/Commands/DeleteProduct/DeleteProductCommandHandler.cs` | 削除ロジックと関連チェック |

**主要処理フロー**:
- **22行目**: FindAsync()で既存商品を検索
- **24-27行目**: 商品が存在しない場合NotFoundException
- **29-34行目**: 注文明細存在チェックとDeleteFailureException
- **36行目**: Products.Remove()で削除マーク
- **38行目**: SaveChangesAsync()でコミット

### プログラム呼び出し階層図

```
ProductsController.Delete(id)
    │
    ├─ Mediator.Send(DeleteProductCommand { Id = id })
    │      │
    │      └─ DeleteProductCommandHandler.Handle()
    │             │
    │             ├─ _context.Products.FindAsync() - 商品検索
    │             │
    │             ├─ (商品なし) → throw NotFoundException
    │             │
    │             ├─ _context.OrderDetails.Any() - 注文チェック
    │             │
    │             ├─ (注文あり) → throw DeleteFailureException
    │             │
    │             ├─ _context.Products.Remove() - 削除マーク
    │             │
    │             └─ _context.SaveChangesAsync() - DELETE実行
    │
    └─ return NoContent()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL Path (id)     ───▶  DeleteProductCommand     ───▶  HTTP 204
                              │
                              │
                              ▼
                    DeleteProductCommandHandler
                              │
                              ▼
                    Products.FindAsync(Id)
                              │
                              ▼
                        存在チェック
                         /        \
                存在しない         存在する
                    │                │
                    ▼                ▼
            NotFoundException   OrderDetails.Any()
                                     │
                                     ▼
                              注文明細チェック
                               /         \
                          存在する      存在しない
                              │             │
                              ▼             ▼
                    DeleteFailureException  Products.Remove()
                                              │
                                              ▼
                                       SaveChangesAsync()
                                              │
                                              ▼
                                       Productsテーブル (DELETE)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | ソース | APIエンドポイント定義 |
| DeleteProductCommand.cs | `Src/Application/Products/Commands/DeleteProduct/DeleteProductCommand.cs` | ソース | コマンドモデル定義 |
| DeleteProductCommandHandler.cs | `Src/Application/Products/Commands/DeleteProduct/DeleteProductCommandHandler.cs` | ソース | コマンド処理ハンドラー |
| Product.cs | `Src/Domain/Entities/Product.cs` | ソース | ドメインエンティティ |
| OrderDetail.cs | `Src/Domain/Entities/OrderDetail.cs` | ソース | 関連チェック対象エンティティ |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
| DeleteFailureException.cs | `Src/Application/Common/Exceptions/DeleteFailureException.cs` | ソース | 削除失敗例外クラス |
