# 機能設計書 16-カテゴリ登録・更新

## 概要

本ドキュメントは、NorthwindTradersシステムにおけるカテゴリ登録・更新機能（Upsert）の設計仕様を定義する。

### 本機能の処理概要

本機能は、新規カテゴリの登録、または既存カテゴリの更新を単一のエンドポイントで行う（Upsert方式）。IDが指定されている場合は更新、IDがnullまたは未指定の場合は新規登録として処理される。

**業務上の目的・背景**：商品カテゴリは商品マスタの分類に使用される重要なマスタデータである。新規カテゴリの追加や既存カテゴリの名称・説明の修正が必要な場面で使用される。Upsert方式により、クライアント側で登録と更新を意識することなく統一的に操作できる。

**機能の利用シーン**：新しい商品カテゴリの追加、既存カテゴリの名称変更、説明文の修正、カテゴリ画像の更新などの場面で使用される。

**主要な処理内容**：
1. APIリクエストからカテゴリ情報を受け取る
2. UpsertCategoryCommandを通じてMediatRパイプラインへディスパッチ
3. UpsertCategoryCommandHandlerがIDの有無をチェック
4. IDがある場合: 既存カテゴリを取得して更新
5. IDがない場合: 新規カテゴリを作成
6. データベースに保存し、カテゴリIDを返却

**関連システム・外部連携**：特になし。

**権限による制御**：CategoriesControllerには[Authorize]属性が付与されており、認証済みユーザーのみがこの機能を利用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 登録・更新（Upsert）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | int? | No | カテゴリID（nullの場合は新規登録） | 更新時は存在するID |
| Name | string | Yes | カテゴリ名 | 必須 |
| Description | string | No | 説明 | - |
| Picture | byte[] | No | カテゴリ画像 | - |

### 入力データソース

HTTPリクエストボディ（JSON形式）からUpsertCategoryCommandオブジェクトにバインドされる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | カテゴリID（新規作成または更新されたカテゴリのID） |

### 出力先

HTTPレスポンス（200 OK）としてカテゴリIDを返却。

## 処理フロー

### 処理シーケンス

\`\`\`
1. CategoriesController.Upsert()がHTTP POSTリクエストを受信
2. MediatRを通じてUpsertCategoryCommandをディスパッチ
3. UpsertCategoryCommandHandlerがコマンドを処理
   └─ IDの有無をチェック
4. IDがある場合（更新）
   └─ FindAsync()で既存カテゴリを取得
   └─ 各プロパティを更新
5. IDがない場合（新規登録）
   └─ 新規Categoryエンティティを作成
   └─ DbContextに追加
6. 共通処理
   └─ SaveChangesAsync()
7. カテゴリIDを返却
\`\`\`

### フローチャート

\`\`\`mermaid
flowchart TD
    A[HTTP POST /api/Categories] --> B[UpsertCategoryCommand生成]
    B --> C[MediatR.Send]
    C --> D[UpsertCategoryCommandHandler.Handle]
    D --> E{Id.HasValue?}
    E -->|Yes| F[FindAsync で既存カテゴリ取得]
    E -->|No| G[新規Category作成]
    F --> H[プロパティ更新]
    G --> I[DbContextに追加]
    H --> J[SaveChangesAsync]
    I --> J
    J --> K[CategoryId返却]
    K --> L[HTTP 200 OK]
\`\`\`

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Upsert方式 | IDの有無で登録/更新を自動判定 | 常に適用 |
| BR-02 | カテゴリ名必須 | カテゴリ名は必ず指定する必要がある | 常に適用 |
| BR-03 | 認証必須 | 認証済みユーザーのみ実行可能 | 常に適用 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 新規登録 | Categories | INSERT | 新規カテゴリを挿入 |
| 更新 | Categories | SELECT | 既存カテゴリを取得 |
| 更新 | Categories | UPDATE | カテゴリ情報を更新 |

### テーブル別操作詳細

#### Categories

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CategoryID | WHERE CategoryID = request.Id | 更新時のみ |
| INSERT/UPDATE | CategoryName | request.Name | 必須 |
| INSERT/UPDATE | Description | request.Description | null許容 |
| INSERT/UPDATE | Picture | request.Picture | null許容 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証ユーザーからのリクエスト | ログイン後に再実行 |
| 400 | BadRequest | カテゴリ名が未指定 | カテゴリ名を指定 |
| 500 | InternalServerError | データベース接続エラー等 | システム管理者に連絡 |

### リトライ仕様

特になし。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- [Authorize]属性により認証済みユーザーのみアクセス可能
- 入力値はEntity Frameworkによりパラメータ化されSQLインジェクションを防止

## 備考

- 更新時にカテゴリが存在しない場合の処理は明示的に実装されていない（FindAsyncがnullを返す可能性）
- NullReferenceExceptionのリスクがあるため、実運用では存在チェックの追加を推奨

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Category.cs | \`Src/Domain/Entities/Category.cs\` | カテゴリエンティティの全プロパティ |
| 1-2 | UpsertCategoryCommand.cs | \`Src/Application/Categories/Commands/UpsertCategory/UpsertCategoryCommand.cs\` | コマンドとハンドラーの両方を含む |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CategoriesController.cs | \`Src/WebUI/Controllers/CategoriesController.cs\` | Upsert()メソッド（21-29行目） |

**主要処理フロー**:
- **21-22行目**: [HttpPost]属性とProducesResponseType
- **26行目**: Mediator.Send(command)でコマンド送信

#### Step 3: コマンドハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UpsertCategoryCommand.cs | \`Src/Application/Categories/Commands/UpsertCategory/UpsertCategoryCommand.cs\` | Handle()メソッド（28-50行目） |

**主要処理フロー**:
- **30行目**: Categoryエンティティ変数宣言
- **32-34行目**: Id.HasValueの場合、FindAsyncで既存取得
- **35-40行目**: Idがない場合、新規作成してAdd
- **43-45行目**: 各プロパティに値をセット
- **47行目**: SaveChangesAsync
- **49行目**: entity.CategoryIdを返却

### プログラム呼び出し階層図

\`\`\`
CategoriesController.Upsert()
    │
    ├─ Mediator.Send(UpsertCategoryCommand)
    │      │
    │      └─ UpsertCategoryCommandHandler.Handle()
    │             │
    │             ├─ [Id.HasValue] → FindAsync() - 既存取得
    │             │
    │             ├─ [!Id.HasValue] → new Category() + Add()
    │             │
    │             ├─ プロパティ設定
    │             │
    │             └─ SaveChangesAsync()
    │
    └─ return Ok(id)
\`\`\`

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CategoriesController.cs | \`Src/WebUI/Controllers/CategoriesController.cs\` | ソース | APIエンドポイント定義 |
| UpsertCategoryCommand.cs | \`Src/Application/Categories/Commands/UpsertCategory/UpsertCategoryCommand.cs\` | ソース | コマンド定義とハンドラー |
| Category.cs | \`Src/Domain/Entities/Category.cs\` | ソース | ドメインエンティティ |
