# 機能設計書 17-カテゴリ削除

## 概要

本ドキュメントは、NorthwindTradersシステムにおけるカテゴリ削除機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、指定されたカテゴリIDに基づいてカテゴリをデータベースから削除するためのAPIエンドポイントを提供する。カテゴリが存在しない場合はNotFoundExceptionがスローされる。

**業務上の目的・背景**：使用されなくなったカテゴリを整理し、マスタデータの健全性を維持する。ただし、商品が紐づいているカテゴリの削除は参照整合性の観点から注意が必要。

**機能の利用シーン**：カテゴリ統合時の重複削除、誤って作成したカテゴリの削除、カテゴリ再編成時などに使用される。

**主要な処理内容**：
1. APIリクエストから削除対象のカテゴリIDを受け取る
2. DeleteCategoryCommandを通じてMediatRパイプラインへディスパッチ
3. DeleteCategoryCommandHandlerが対象カテゴリを取得し存在確認
4. カテゴリが存在しない場合はNotFoundExceptionをスロー
5. カテゴリを削除
6. 204 No Contentを返却

**関連システム・外部連携**：特になし。

**権限による制御**：CategoriesControllerには[Authorize]属性が付与されており、認証済みユーザーのみがこの機能を利用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 削除（Delete）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 削除対象のカテゴリID | URLパスパラメータ |

### 入力データソース

URLパスパラメータ（/api/Categories/{id}）から取得される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | 成功時はレスポンスボディなし（204 No Content） |

### 出力先

HTTPレスポンスステータスコード。

## 処理フロー

### 処理シーケンス

\`\`\`
1. CategoriesController.Delete(id)がHTTP DELETEリクエストを受信
2. MediatRを通じてDeleteCategoryCommandをディスパッチ
3. DeleteCategoryCommandHandlerがコマンドを処理
   └─ FindAsync()で対象カテゴリを取得
4. カテゴリ存在チェック
   └─ 存在しない場合：NotFoundException("Category", id)
5. 削除処理
   └─ _context.Categories.Remove(entity)
6. SaveChangesAsync()
7. NoContent（204）を返却
\`\`\`

### フローチャート

\`\`\`mermaid
flowchart TD
    A[HTTP DELETE /api/Categories/id] --> B[DeleteCategoryCommand生成]
    B --> C[MediatR.Send]
    C --> D[DeleteCategoryCommandHandler.Handle]
    D --> E{カテゴリ存在チェック}
    E -->|存在しない| F[NotFoundException]
    E -->|存在する| G[Categories.Remove]
    G --> H[SaveChangesAsync]
    H --> I[HTTP 204 No Content]
    F --> J[HTTP 404 Not Found]
\`\`\`

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | カテゴリ存在必須 | 削除対象のカテゴリが存在する必要がある | 常に適用 |
| BR-02 | 認証必須 | 認証済みユーザーのみ実行可能 | 常に適用 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| カテゴリ検索 | Categories | SELECT | CategoryIdで既存カテゴリを検索 |
| カテゴリ削除 | Categories | DELETE | カテゴリレコードを削除 |

### テーブル別操作詳細

#### Categories

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CategoryID | WHERE CategoryID = request.Id | 対象カテゴリ取得 |
| DELETE | (全カラム) | WHERE CategoryID = entity.CategoryId | レコード削除 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証ユーザーからのリクエスト | ログイン後に再実行 |
| 404 | NotFound | 指定されたCategoryIdが存在しない | 正しいカテゴリIDを指定 |
| 500 | InternalServerError | 参照整合性エラー（商品が紐づいている場合）等 | 商品を別カテゴリに移動後に再実行 |

### リトライ仕様

特になし。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- [Authorize]属性により認証済みユーザーのみアクセス可能
- 削除操作のため、適切な権限チェックの追加を推奨

## 備考

- 商品が紐づいているカテゴリを削除しようとすると、データベースレベルの外部キー制約エラーが発生する可能性がある
- 商品の削除チェックは実装されていないため、運用での注意が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Category.cs | \`Src/Domain/Entities/Category.cs\` | カテゴリエンティティ、Productsとの関連 |
| 1-2 | DeleteCategoryCommand.cs | \`Src/Application/Categories/Commands/DeleteCategory/DeleteCategoryCommand.cs\` | コマンドとハンドラー |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CategoriesController.cs | \`Src/WebUI/Controllers/CategoriesController.cs\` | Delete()メソッド（31-39行目） |

**主要処理フロー**:
- **31行目**: [HttpDelete("{id}")]属性
- **36行目**: DeleteCategoryCommandを作成してディスパッチ

#### Step 3: コマンドハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DeleteCategoryCommand.cs | \`Src/Application/Categories/Commands/DeleteCategory/DeleteCategoryCommand.cs\` | Handle()メソッド（23-38行目） |

**主要処理フロー**:
- **25-26行目**: FindAsyncで対象カテゴリを取得
- **28-31行目**: nullチェックしてNotFoundExceptionをスロー
- **33行目**: Categories.Remove()で削除
- **35行目**: SaveChangesAsync()でコミット

### プログラム呼び出し階層図

\`\`\`
CategoriesController.Delete(id)
    │
    ├─ Mediator.Send(DeleteCategoryCommand)
    │      │
    │      └─ DeleteCategoryCommandHandler.Handle()
    │             │
    │             ├─ _context.Categories.FindAsync() - カテゴリ検索
    │             │
    │             ├─ (カテゴリなし) → throw NotFoundException
    │             │
    │             ├─ _context.Categories.Remove() - 削除マーク
    │             │
    │             └─ _context.SaveChangesAsync() - コミット
    │
    └─ return NoContent()
\`\`\`

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CategoriesController.cs | \`Src/WebUI/Controllers/CategoriesController.cs\` | ソース | APIエンドポイント定義 |
| DeleteCategoryCommand.cs | \`Src/Application/Categories/Commands/DeleteCategory/DeleteCategoryCommand.cs\` | ソース | コマンド定義とハンドラー |
| Category.cs | \`Src/Domain/Entities/Category.cs\` | ソース | ドメインエンティティ |
| NotFoundException.cs | \`Src/Application/Common/Exceptions/NotFoundException.cs\` | ソース | 例外クラス |
