# 機能設計書 19-従業員詳細取得

## 概要

本ドキュメントは、NorthwindTradersシステムにおける従業員詳細取得機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、指定された従業員IDに基づいて従業員の詳細情報（担当地域含む）を取得するAPIエンドポイントを提供する。基本情報に加え、担当テリトリー（地域）の一覧も含めた詳細データを返却する。

**業務上の目的・背景**：従業員の詳細情報は、人事管理、担当地域の確認、組織構造の把握など多くの場面で必要となる。本機能により、特定の従業員の完全な情報を取得できる。

**機能の利用シーン**：従業員詳細画面での情報表示、人事レポートの作成、担当地域の確認、管理者情報の参照などに使用される。

**主要な処理内容**：
1. APIリクエストから従業員IDを受け取る
2. GetEmployeeDetailQueryを通じてMediatRパイプラインへディスパッチ
3. GetEmployeeDetailQueryHandlerが指定IDの従業員を取得
4. AutoMapperでEmployeeDetailVmにマッピング（担当地域含む）
5. 従業員詳細情報を返却

**関連システム・外部連携**：特になし。

**権限による制御**：認証設定は明示的に定義されていない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を直接呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 詳細取得（Read）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 従業員ID | URLパスパラメータ |

### 入力データソース

URLパスパラメータから従業員IDを取得（/api/Employees/{id}）。

## 出力仕様

### 出力データ（EmployeeDetailVm）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | 従業員ID |
| Title | string | 敬称（TitleOfCourtesy） |
| FirstName | string | 名 |
| LastName | string | 姓 |
| BirthDate | DateTime? | 生年月日 |
| Address | string | 住所 |
| City | string | 市区町村 |
| Region | string | 地域 |
| PostalCode | string | 郵便番号 |
| Country | string | 国 |
| HomePhone | string | 自宅電話番号 |
| Position | string | 役職（Title） |
| Extension | string | 内線番号 |
| HireDate | DateTime? | 入社日 |
| Notes | string | 備考 |
| Photo | byte[] | 写真 |
| ManagerId | int? | 上司ID（ReportsTo） |
| Territories | List<EmployeeTerritoryDto> | 担当地域リスト |

### EmployeeTerritoryDto

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TerritoryId | string | テリトリーID |
| Territory | string | テリトリー名 |
| Region | string | リージョン名 |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. EmployeesController.Get()がHTTP GETリクエストを受信
   └─ URLパスから従業員IDを取得
2. MediatRを通じてGetEmployeeDetailQueryをディスパッチ
   └─ Mediator.Send(new GetEmployeeDetailQuery { Id = id })
3. GetEmployeeDetailQueryHandlerがクエリを処理
   └─ Where(e => e.EmployeeId == request.Id)
   └─ ProjectTo<EmployeeDetailVm>()
   └─ SingleOrDefaultAsync()
4. 従業員詳細情報を返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTP GET /api/Employees/id] --> B[GetEmployeeDetailQuery生成]
    B --> C[MediatR.Send]
    C --> D[GetEmployeeDetailQueryHandler.Handle]
    D --> E[Where EmployeeId == id]
    E --> F[ProjectTo EmployeeDetailVm]
    F --> G[SingleOrDefaultAsync]
    G --> H{従業員存在?}
    H -->|Yes| I[HTTP 200 OK]
    H -->|No| J[null返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 単一取得 | 指定されたIDの従業員のみを取得 | 常に適用 |
| BR-02 | 担当地域含む | EmployeeTerritoriesを含めてマッピング | 常に適用 |
| BR-03 | プロパティマッピング | TitleOfCourtesy→Title、Title→Position、ReportsTo→ManagerId | 常に適用 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 従業員取得 | Employees | SELECT | 指定IDの従業員を取得 |
| 担当地域取得 | EmployeeTerritories | SELECT (JOIN) | 従業員の担当地域を取得 |
| テリトリー取得 | Territories | SELECT (JOIN) | テリトリー名を取得 |
| リージョン取得 | Region | SELECT (JOIN) | リージョン名を取得 |

### テーブル別操作詳細

#### Employees

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE EmployeeID = request.Id | 主キー検索 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 200 | OK (null) | 指定されたEmployeeIdが存在しない | null返却（例外ではない） |
| 500 | InternalServerError | データベース接続エラー等 | システム管理者に連絡 |

### リトライ仕様

特になし。

## トランザクション仕様

読み取り専用のため、トランザクション管理は不要。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- 従業員詳細には個人情報（住所、電話番号、生年月日等）が含まれる
- 適切なアクセス制御の実装を検討すべき

## 備考

- 存在しない従業員IDが指定された場合、nullが返却される（例外はスローされない）
- 担当地域はEmployeeTerritoriesを通じて取得される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Employee.cs | `Src/Domain/Entities/Employee.cs` | 従業員エンティティの全プロパティ |
| 1-2 | EmployeeDetailVm.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/EmployeeDetailVm.cs` | ViewModelとマッピングルール |
| 1-3 | EmployeeTerritoryDto.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/EmployeeTerritoryDto.cs` | 担当地域DTO |

**読解のコツ**: EmployeeDetailVm.csのMapping()メソッドで、プロパティ名の変換を確認。特にTitleOfCourtesy→Title、Title→Position、ReportsTo→ManagerIdの変換に注目。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | Get()メソッドの実装 |

**主要処理フロー**:
1. **21-26行目**: Get()メソッドがパスパラメータidを受け取りMediatRへディスパッチ

#### Step 3: クエリとハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GetEmployeeDetailQuery.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/GetEmployeeDetailQuery.cs` | クエリとハンドラー |

**主要処理フロー**:
- **14行目**: Idプロパティ
- **27-34行目**: Handle()メソッド
- **29-32行目**: Where、ProjectTo、SingleOrDefaultAsyncの流れ

### プログラム呼び出し階層図

```
EmployeesController.Get(id)
    │
    ├─ Mediator.Send(GetEmployeeDetailQuery { Id = id })
    │      │
    │      └─ GetEmployeeDetailQueryHandler.Handle()
    │             │
    │             ├─ _context.Employees.Where(e => e.EmployeeId == request.Id)
    │             │
    │             ├─ .ProjectTo<EmployeeDetailVm>()
    │             │      │
    │             │      └─ EmployeeTerritories → Territories
    │             │
    │             └─ .SingleOrDefaultAsync()
    │
    └─ return Ok(vm)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL Path (id)     ───▶  GetEmployeeDetailQuery   ───▶  JSON
                              │                         EmployeeDetailVm
                              │
                              ▼
                    GetEmployeeDetailQueryHandler
                              │
                              ▼
                    Employees.Where(EmployeeId == id)
                              │
                              ▼
                    ProjectTo<EmployeeDetailVm>
                              │
                              ▼
                    EmployeeTerritories JOIN
                              │
                              ▼
                    SingleOrDefaultAsync
                              │
                              ▼
                    EmployeeDetailVm
                    { 基本情報, Territories[] }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | ソース | APIエンドポイント定義 |
| GetEmployeeDetailQuery.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/GetEmployeeDetailQuery.cs` | ソース | クエリとハンドラー |
| EmployeeDetailVm.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/EmployeeDetailVm.cs` | ソース | 従業員詳細ViewModel |
| EmployeeTerritoryDto.cs | `Src/Application/Employees/Queries/GetEmployeeDetail/EmployeeTerritoryDto.cs` | ソース | 担当地域DTO |
| Employee.cs | `Src/Domain/Entities/Employee.cs` | ソース | ドメインエンティティ |
| EmployeeTerritory.cs | `Src/Domain/Entities/EmployeeTerritory.cs` | ソース | 中間エンティティ |
