# 機能設計書 2-ログイン

## 概要

本ドキュメントは、NorthwindTradersシステムにおけるログイン機能の設計仕様を記載する。ASP.NET Core Identityを使用した認証機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：NorthwindTradersシステムの保護されたリソース（顧客管理、従業員管理等）にアクセスするためには、ユーザー認証が必要である。本機能は、登録済みユーザーがシステムにログインするための認証処理を提供する。セキュリティ強化のため、二要素認証やアカウントロックアウト機能も実装されている。

**機能の利用シーン**：登録済みユーザーがシステムにアクセスする際、ログイン画面（サーバー）またはSPA経由のOAuth2/OIDC認証フローでログインを行う。セッションがタイムアウトした場合や、明示的にログアウトした後の再ログインにも使用される。

**主要な処理内容**：
1. ユーザーがメールアドレスとパスワードを入力
2. 入力値のバリデーション
3. SignInManager.PasswordSignInAsync()による認証処理
4. 認証結果に応じた処理分岐（成功、二要素認証要求、ロックアウト、失敗）
5. 成功時は認証Cookieを発行し、指定されたリダイレクト先へ遷移
6. メール確認未完了ユーザーへの確認メール再送信機能

**関連システム・外部連携**：IdentityServerとの連携によるOAuth2/OIDCフロー対応。IEmailSenderを通じたメール送信サービスとの連携（確認メール再送信時）。

**権限による制御**：本機能は[AllowAnonymous]属性により、未認証ユーザーでもアクセス可能である。ログイン成功後に各機能のAuthorize属性による権限制御が適用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | ログイン画面 | 主画面 | AuthorizeServiceを通じてIdentityServerと連携し、OAuth2/OIDC認証フローによるログイン処理を行う |
| 6 | ログアウト画面 | 参照画面 | AuthorizeServiceを通じてログアウト処理を実行し、セッションを終了する |
| 7 | ログイン画面（サーバー） | 主画面 | SignInManager.PasswordSignInAsync()を使用してメールアドレスとパスワードでのローカルログインを処理する |

## 機能種別

認証処理 / セッション管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Email | string | Yes | ユーザーのメールアドレス | Required, EmailAddress形式 |
| Password | string | Yes | パスワード | Required |
| RememberMe | bool | No | ログイン状態を保持するか | デフォルトfalse |
| returnUrl | string | No | ログイン後のリダイレクト先URL | - |

### 入力データソース

Razor Pages形式の入力フォーム（Login.cshtml）からのPOSTリクエスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| SignInResult | SignInResult | ログイン処理の結果 |
| Succeeded | bool | 認証成功フラグ |
| RequiresTwoFactor | bool | 二要素認証が必要かどうか |
| IsLockedOut | bool | アカウントロックアウト状態 |

### 出力先

- 成功時：returnUrl（指定がない場合はホーム画面）へリダイレクト
- 二要素認証要求時：LoginWith2faページへリダイレクト
- ロックアウト時：Lockoutページへリダイレクト
- 失敗時：同一ページ（Login）にエラーメッセージを表示

## 処理フロー

### 処理シーケンス

```
1. OnGetAsync - 画面初期表示
   ├─ エラーメッセージがある場合はModelStateに追加
   ├─ 既存の外部認証Cookieをクリア
   └─ 外部認証スキーム一覧を取得してExternalLoginsに設定
2. OnPostAsync - ログインフォーム送信時
   ├─ ModelState.IsValidでバリデーション確認
   ├─ SignInManager.PasswordSignInAsync()で認証
   │   └─ lockoutOnFailure: true でロックアウト機能有効
   ├─ 結果による分岐：
   │   ├─ Succeeded → returnUrlへリダイレクト
   │   ├─ RequiresTwoFactor → LoginWith2faへリダイレクト
   │   ├─ IsLockedOut → Lockoutページへリダイレクト
   │   └─ それ以外 → エラーメッセージ表示
   └─ 失敗時：エラーをModelStateに追加して再表示
3. OnPostSendVerificationEmailAsync - 確認メール再送信
   ├─ UserManager.FindByEmailAsync()でユーザー検索
   ├─ メール確認トークン生成
   └─ IEmailSender.SendEmailAsync()で確認メール送信
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[OnGetAsync: 初期化処理]
    B --> C[外部認証Cookie削除]
    C --> D[外部認証スキーム取得]
    D --> E[フォーム表示]
    E --> F{POSTリクエスト?}
    F -->|No| E
    F -->|Yes| G{ModelState有効?}
    G -->|No| H[エラー表示]
    H --> E
    G -->|Yes| I[PasswordSignInAsync実行]
    I --> J{認証結果}
    J -->|Succeeded| K[returnUrlへリダイレクト]
    J -->|RequiresTwoFactor| L[LoginWith2faへリダイレクト]
    J -->|IsLockedOut| M[Lockoutページへリダイレクト]
    J -->|失敗| N[エラーメッセージ追加]
    N --> H
    K --> O[終了]
    L --> O
    M --> O
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | アカウントロックアウト | ログイン失敗が規定回数を超えるとアカウントをロック | lockoutOnFailure: true |
| BR-002 | 二要素認証 | 二要素認証が有効なユーザーは追加認証が必要 | 二要素認証設定済みユーザー |
| BR-003 | セッション保持 | RememberMeがtrueの場合、永続的なCookieを発行 | RememberMe選択時 |
| BR-004 | 外部認証Cookie削除 | ログイン画面表示時に既存の外部認証Cookieを削除 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー検索 | AspNetUsers | SELECT | メールアドレスによるユーザー検索 |
| ロックアウトカウント更新 | AspNetUsers | UPDATE | ログイン失敗時にカウント増加 |

### テーブル別操作詳細

#### AspNetUsers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | * | NormalizedEmail = Input.Email.ToUpper() | ユーザー検索 |
| UPDATE | AccessFailedCount | +1 | ログイン失敗時 |
| UPDATE | LockoutEnd | 現在時刻 + ロックアウト期間 | ロックアウト発動時 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidLogin | 認証エラー | メールアドレスまたはパスワードが不正 | 正しい認証情報で再試行 |
| LockedOut | アカウントロック | 規定回数のログイン失敗 | ロックアウト期間終了を待つ |
| RequiresTwoFactor | 追加認証要求 | 二要素認証が必要 | 二要素認証コードを入力 |
| EmailNotConfirmed | メール未確認 | メールアドレスが未確認状態 | 確認メールから確認を完了 |

### リトライ仕様

ログイン失敗時はユーザーが再度フォームを送信する。ロックアウト後は一定期間経過後に再試行可能。

## トランザクション仕様

SignInManager.PasswordSignInAsync()内部でトランザクション管理される。ログイン試行とロックアウトカウント更新はアトミック操作として実行される。

## パフォーマンス要件

- レスポンス時間：1秒以内
- パスワードハッシュ検証は計算コストが高いが、セキュリティのため許容

## セキュリティ考慮事項

- パスワードはデータベースにハッシュ化されて保存、平文では送信されない
- CSRF対策としてAntiForgeryTokenを使用（Razor Pages標準機能）
- アカウントロックアウトによるブルートフォース攻撃対策
- 二要素認証による追加セキュリティ層
- [AllowAnonymous]属性により未認証アクセスを許可

## 備考

- ASP.NET Core Identity 3.0を使用
- IdentityServer4によるOAuth2/OIDC対応
- 外部認証プロバイダー（Google、Facebook等）との連携も可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

入力データモデルとユーザーエンティティを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Login.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Login.cshtml.cs` | InputModelクラス（47-59行目）で入力データ構造を定義 |
| 1-2 | ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | IdentityUserを継承したユーザーエンティティ |

**読解のコツ**: InputModelはRazor PagesのBindPropertyとして使用され、フォームデータの受け渡しに使われる。

#### Step 2: エントリーポイントを理解する

Razor PagesのPageModelであるLoginModelがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Login.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Login.cshtml.cs` | ログインのメインロジック |

**主要処理フロー**:
1. **26-35行目**: コンストラクタでDI経由でUserManager, SignInManager, ILogger, IEmailSenderを取得
2. **61-76行目**: OnGetAsync()で初期表示処理
3. **71行目**: HttpContext.SignOutAsync()で外部認証Cookieをクリア
4. **78-110行目**: OnPostAsync()でログイン処理を実行
5. **86行目**: PasswordSignInAsync()で認証（lockoutOnFailure: true）
6. **87-100行目**: 認証結果による分岐処理
7. **112-139行目**: OnPostSendVerificationEmailAsync()で確認メール再送信

#### Step 3: 認証サービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Startup.cs | `Src/WebUI/Startup.cs` | 認証・認可設定 |

**主要処理フロー**:
- **105-107行目**: UseAuthentication, UseIdentityServer, UseAuthorization

### プログラム呼び出し階層図

```
Login.cshtml (View)
    │
    └─ LoginModel (PageModel)
           │
           ├─ OnGetAsync()
           │      └─ SignOutAsync() - 外部認証Cookie削除
           │
           ├─ OnPostAsync()
           │      ├─ SignInManager.PasswordSignInAsync()
           │      │      ├─ AspNetUsers テーブル参照
           │      │      └─ 認証Cookie発行
           │      │
           │      └─ 結果に応じたリダイレクト
           │
           └─ OnPostSendVerificationEmailAsync()
                  ├─ UserManager.FindByEmailAsync()
                  ├─ UserManager.GenerateEmailConfirmationTokenAsync()
                  └─ IEmailSender.SendEmailAsync()
```

### データフロー図

```
[入力]                    [処理]                              [出力]

Email            ───▶  InputModel                    ───▶  認証Cookie
Password         ───▶  LoginModel.OnPostAsync()      ───▶  リダイレクト先画面
RememberMe       ───▶  PasswordSignInAsync()         ───▶  エラーメッセージ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Login.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Login.cshtml.cs` | ソース | ログインのPageModel |
| Login.cshtml | `Src/WebUI/Areas/Identity/Pages/Account/Login.cshtml` | ビュー | ログインフォームのRazor Page |
| ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | ソース | ユーザーエンティティ |
| Startup.cs | `Src/WebUI/Startup.cs` | ソース | 認証・認可設定 |
| Lockout.cshtml | `Src/WebUI/Areas/Identity/Pages/Account/Lockout.cshtml` | ビュー | ロックアウト画面 |
| LoginWith2fa.cshtml | `Src/WebUI/Areas/Identity/Pages/Account/LoginWith2fa.cshtml` | ビュー | 二要素認証画面 |
