# 機能設計書 20-従業員登録・更新

## 概要

本ドキュメントは、NorthwindTradersシステムにおける従業員登録・更新機能の設計仕様を定義する。

### 本機能の処理概要

本機能は、従業員情報を新規登録または既存の従業員情報を更新するためのAPIエンドポイントを提供する。Upsertパターンを採用しており、IDの有無によって新規登録と更新を自動的に判別して処理する。新規登録時はIDなし（null）でリクエストし、更新時は既存の従業員IDを指定する。

**業務上の目的・背景**：従業員マスタは、受注処理における担当者割り当て、組織構造の管理、権限管理など多くの業務で参照される。本機能により、従業員の入社・退社・異動などに伴う情報の登録・更新を行う。

**機能の利用シーン**：新入社員の登録、従業員の連絡先変更、役職変更、上司（Manager）の変更、住所変更などの人事情報の更新に使用される。

**主要な処理内容**：
1. APIリクエストから従業員情報（UpsertEmployeeCommand）を受け取る
2. MediatRを通じてUpsertEmployeeCommandをディスパッチ
3. UpsertEmployeeCommandHandlerがID有無で新規/更新を判別
4. IDあり：既存従業員を取得して更新、IDなし：新規従業員エンティティを作成
5. 15項目のプロパティを設定
6. SaveChangesAsync()でデータベースに保存
7. 従業員IDを返却

**関連システム・外部連携**：特になし。

**権限による制御**：EmployeesControllerには明示的な[Authorize]属性は付与されていない。BaseControllerの設定による認証制御が適用される可能性がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現在、この機能を直接呼び出す画面は実装されていない |

## 機能種別

CRUD操作 / 登録・更新（Create/Update - Upsert）

## 入力仕様

### 入力パラメータ（UpsertEmployeeCommand）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | int? | No | 従業員ID（更新時のみ指定） | null=新規、値あり=更新 |
| Title | string | No | 敬称（Mr., Mrs.等） | TitleOfCourtesyに設定 |
| FirstName | string | No | 名 | - |
| LastName | string | No | 姓 | - |
| BirthDate | DateTime? | No | 生年月日 | - |
| Address | string | No | 住所 | - |
| City | string | No | 市区町村 | - |
| Region | string | No | 地域 | - |
| PostalCode | string | No | 郵便番号 | - |
| Country | string | No | 国 | - |
| HomePhone | string | No | 自宅電話番号 | - |
| Position | string | No | 役職 | Titleに設定 |
| Extension | string | No | 内線番号 | - |
| HireDate | DateTime? | No | 入社日 | - |
| Notes | string | No | 備考 | - |
| Photo | byte[] | No | 写真 | - |
| ManagerId | int? | No | 上司ID | ReportsToに設定 |

### 入力データソース

HTTPリクエストボディ（JSON形式）から取得される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id | int | 登録・更新された従業員ID |

### 出力先

HTTPレスポンス（JSON形式）。新規登録時は新しく採番されたID、更新時は既存のIDが返却される。

## 処理フロー

### 処理シーケンス

```
1. EmployeesController.Upsert()がHTTP POSTリクエストを受信
   └─ リクエストボディからUpsertEmployeeCommandを取得
2. MediatRを通じてUpsertEmployeeCommandをディスパッチ
   └─ Mediator.Send(command)
3. UpsertEmployeeCommandHandlerがコマンドを処理
   └─ request.Id.HasValueで新規/更新を判別
4-A. 更新の場合（Id有り）
   └─ FindAsync(request.Id.Value)で既存従業員を取得
4-B. 新規の場合（Id無し）
   └─ new Employee()で新規エンティティ作成
   └─ _context.Employees.Add(entity)
5. プロパティマッピング（15項目）
   └─ entity.TitleOfCourtesy = request.Title
   └─ entity.Title = request.Position
   └─ entity.ReportsTo = request.ManagerId
   └─ その他12項目
6. SaveChangesAsync()でデータベースに保存
7. entity.EmployeeIdを返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTP POST /api/Employees] --> B[UpsertEmployeeCommand受信]
    B --> C[MediatR.Send]
    C --> D[UpsertEmployeeCommandHandler.Handle]
    D --> E{Id.HasValue?}
    E -->|Yes 更新| F[FindAsync id]
    E -->|No 新規| G[new Employee]
    G --> H[Employees.Add]
    F --> I[プロパティ設定 15項目]
    H --> I
    I --> J[SaveChangesAsync]
    J --> K[return EmployeeId]
    K --> L[HTTP 200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Upsertパターン | IDの有無で新規登録/更新を自動判別 | 常に適用 |
| BR-02 | プロパティマッピング | Title→TitleOfCourtesy、Position→Title、ManagerId→ReportsTo | 常に適用 |
| BR-03 | ID自動採番 | 新規登録時はEmployeeIdが自動採番される | 新規登録時 |

### 計算ロジック

特になし。

### プロパティマッピング詳細

| コマンドプロパティ | エンティティプロパティ | 説明 |
|------------------|---------------------|------|
| Title | TitleOfCourtesy | 敬称（Mr., Mrs.等） |
| Position | Title | 役職名 |
| ManagerId | ReportsTo | 上司の従業員ID |
| その他12項目 | 同名プロパティ | FirstName, LastName, BirthDate, Address, City, Region, PostalCode, Country, HomePhone, Extension, HireDate, Notes, Photo |

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 従業員検索（更新時） | Employees | SELECT | EmployeeIdで既存従業員を検索 |
| 従業員登録 | Employees | INSERT | 新規従業員レコードを追加 |
| 従業員更新 | Employees | UPDATE | 既存従業員レコードを更新 |

### テーブル別操作詳細

#### Employees

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | EmployeeID | WHERE EmployeeID = request.Id | 更新時のみ |
| INSERT/UPDATE | TitleOfCourtesy | request.Title | 敬称 |
| INSERT/UPDATE | FirstName | request.FirstName | 名 |
| INSERT/UPDATE | LastName | request.LastName | 姓 |
| INSERT/UPDATE | BirthDate | request.BirthDate | 生年月日 |
| INSERT/UPDATE | Address | request.Address | 住所 |
| INSERT/UPDATE | City | request.City | 市区町村 |
| INSERT/UPDATE | Region | request.Region | 地域 |
| INSERT/UPDATE | PostalCode | request.PostalCode | 郵便番号 |
| INSERT/UPDATE | Country | request.Country | 国 |
| INSERT/UPDATE | HomePhone | request.HomePhone | 自宅電話番号 |
| INSERT/UPDATE | Title | request.Position | 役職 |
| INSERT/UPDATE | Extension | request.Extension | 内線番号 |
| INSERT/UPDATE | HireDate | request.HireDate | 入社日 |
| INSERT/UPDATE | Notes | request.Notes | 備考 |
| INSERT/UPDATE | Photo | request.Photo | 写真 |
| INSERT/UPDATE | ReportsTo | request.ManagerId | 上司ID |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | 不正なリクエストデータ | 入力データを確認して再送信 |
| 500 | InternalServerError | データベース接続エラー、制約違反等 | システム管理者に連絡 |

### リトライ仕様

特になし。

## トランザクション仕様

SaveChangesAsync()呼び出し時に単一トランザクションとして実行される。新規登録と更新の両方で同一のトランザクション制御が適用される。

## パフォーマンス要件

特に明示的な要件なし。

## セキュリティ考慮事項

- 従業員情報には個人情報（住所、電話番号、生年月日、写真等）が含まれる
- Photo項目はbyte[]でバイナリデータを直接受け取るため、サイズ制限の検討が必要
- 適切なアクセス制御の実装を推奨
- 更新時に存在しないIDを指定した場合のnull参照エラーに注意

## 備考

- 更新時に存在しないIDを指定した場合、FindAsync()がnullを返し、その後のプロパティ設定でNullReferenceExceptionが発生する可能性がある（例外ハンドリング未実装）
- バリデーション（必須チェック、文字数制限等）は明示的に実装されていない
- 担当地域（EmployeeTerritories）の登録・更新は本機能では対応していない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Employee.cs | `Src/Domain/Entities/Employee.cs` | 従業員エンティティの全プロパティ |
| 1-2 | UpsertEmployeeCommand.cs | `Src/Application/Employees/Commands/UpsertEmployee/UpsertEmployeeCommand.cs` | コマンドプロパティの定義（12-44行目） |

**読解のコツ**: Employee.csとUpsertEmployeeCommand.csのプロパティ名を比較し、マッピングルールを把握する。特にTitle↔TitleOfCourtesy、Position↔Title、ManagerId↔ReportsToの対応関係に注目。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | Upsert()メソッドの実装 |

**主要処理フロー**:
1. **29行目**: [HttpPost]属性でPOSTリクエストを受け付け
2. **32行目**: Upsert()メソッドがUpsertEmployeeCommandを受け取る
3. **34行目**: Mediator.Send(command)でMediatRパイプラインへディスパッチ
4. **36行目**: Ok(id)で従業員IDを返却

#### Step 3: コマンドハンドラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UpsertEmployeeCommand.cs | `Src/Application/Employees/Commands/UpsertEmployee/UpsertEmployeeCommand.cs` | Handle()メソッド（55-90行目） |

**主要処理フロー**:
- **59-68行目**: Id.HasValueで新規/更新を分岐
- **61行目**: 更新時はFindAsync()で既存従業員を取得
- **65-67行目**: 新規時はnew Employee()してAdd()
- **70-85行目**: 15項目のプロパティをコマンドからエンティティにコピー
- **87行目**: SaveChangesAsync()でデータベースに保存
- **89行目**: entity.EmployeeIdを返却

### プログラム呼び出し階層図

```
EmployeesController.Upsert(command)
    │
    ├─ Mediator.Send(UpsertEmployeeCommand)
    │      │
    │      └─ UpsertEmployeeCommandHandler.Handle()
    │             │
    │             ├─ (Id有り) _context.Employees.FindAsync() - 既存従業員取得
    │             │
    │             ├─ (Id無し) new Employee() + _context.Employees.Add()
    │             │
    │             ├─ entity.TitleOfCourtesy = request.Title
    │             ├─ entity.FirstName = request.FirstName
    │             ├─ entity.LastName = request.LastName
    │             ├─ entity.BirthDate = request.BirthDate
    │             ├─ entity.Address = request.Address
    │             ├─ entity.City = request.City
    │             ├─ entity.Region = request.Region
    │             ├─ entity.PostalCode = request.PostalCode
    │             ├─ entity.Country = request.Country
    │             ├─ entity.HomePhone = request.HomePhone
    │             ├─ entity.Title = request.Position
    │             ├─ entity.Extension = request.Extension
    │             ├─ entity.HireDate = request.HireDate
    │             ├─ entity.Notes = request.Notes
    │             ├─ entity.Photo = request.Photo
    │             ├─ entity.ReportsTo = request.ManagerId
    │             │
    │             └─ _context.SaveChangesAsync() - コミット
    │
    └─ return Ok(entity.EmployeeId)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

JSON Body         ───▶  UpsertEmployeeCommand     ───▶  JSON
{                            │                          { id: xxx }
  Id?,                       │
  Title,                     ▼
  FirstName,          UpsertEmployeeCommandHandler
  LastName,                  │
  BirthDate?,                ▼
  Address,            ┌──────┴──────┐
  City,               │ Id有り?     │
  Region,             ▼             ▼
  PostalCode,    FindAsync     new Employee
  Country,            │             │
  HomePhone,          └──────┬──────┘
  Position,                  │
  Extension,                 ▼
  HireDate?,          プロパティマッピング
  Notes,              (15項目)
  Photo,                     │
  ManagerId?                 ▼
}                     SaveChangesAsync
                             │
                             ▼
                      return EmployeeId
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | ソース | APIエンドポイント定義 |
| UpsertEmployeeCommand.cs | `Src/Application/Employees/Commands/UpsertEmployee/UpsertEmployeeCommand.cs` | ソース | コマンド定義とハンドラー |
| Employee.cs | `Src/Domain/Entities/Employee.cs` | ソース | ドメインエンティティ |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DBコンテキストインターフェース |
