# 機能設計書 21-従業員削除

## 概要

本ドキュメントは、Northwind Tradersシステムにおける従業員削除機能の設計仕様を記載する。

### 本機能の処理概要

この機能は、指定した従業員IDに基づいて従業員情報をシステムから削除する。従業員に紐づくユーザーアカウントがある場合は同時に削除を行う。

**業務上の目的・背景**：退職者や誤登録された従業員データを適切にシステムから削除し、データの整合性と正確性を維持する必要がある。従業員データは注文処理や顧客対応履歴と紐づいているため、適切な権限管理の下で削除操作を行う必要がある。

**機能の利用シーン**：人事管理者が従業員の退職処理を行う際、または誤って登録された従業員データを削除する際に使用される。管理者権限を持つユーザーのみが実行可能であり、自分自身のアカウントは削除できないよう制限されている。

**主要な処理内容**：
1. 削除対象の従業員IDを受け取る
2. 従業員の存在確認を行う
3. 自分自身の削除ではないことを確認する
4. 関連するユーザーアカウントが存在する場合は先に削除する
5. 従業員データをデータベースから削除する

**関連システム・外部連携**：ASP.NET Core Identityと連携し、従業員に紐づくユーザーアカウントの削除も同時に行う。

**権限による制御**：認証済みユーザーのみが実行可能。自分自身のアカウントは削除できない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面機能マッピングに定義されていないが、従業員管理画面等から呼び出されることを想定 |

## 機能種別

CRUD操作（Delete）/ データ削除

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 削除対象の従業員ID | 正の整数であること |

### 入力データソース

HTTPリクエストのURLパラメータ（DELETE /api/Employees/{id}）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| なし | - | 成功時はステータスコード204（No Content）を返す |

### 出力先

HTTPレスポンス

## 処理フロー

### 処理シーケンス

```
1. APIエンドポイント受信
   └─ EmployeesController.Delete(id)がリクエストを受信
2. コマンド発行
   └─ DeleteEmployeeCommandをMediatorに送信
3. 従業員存在確認
   └─ INorthwindDbContext.Employees.FindAsync(id)で検索
4. 存在チェック
   └─ 存在しない場合はNotFoundExceptionをスロー
5. 自己削除チェック
   └─ 削除対象が自分自身の場合はBadRequestExceptionをスロー
6. ユーザーアカウント削除
   └─ UserIdが設定されている場合はIUserManager.DeleteUserAsyncを呼び出し
7. 従業員データ削除
   └─ INorthwindDbContext.Employees.Remove(entity)
8. 変更保存
   └─ SaveChangesAsync(cancellationToken)
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[従業員ID受信]
    B --> C{従業員存在確認}
    C -->|存在しない| D[NotFoundException]
    C -->|存在する| E{自己削除チェック}
    E -->|自分自身| F[BadRequestException]
    E -->|他者| G{UserId存在確認}
    G -->|存在する| H[ユーザーアカウント削除]
    G -->|存在しない| I[従業員データ削除]
    H --> I
    I --> J[SaveChanges]
    J --> K[204 No Content]
    D --> L[終了]
    F --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-21-01 | 自己削除禁止 | ログイン中のユーザーは自分自身の従業員レコードを削除できない | 削除対象のUserIdが現在のユーザーIDと一致する場合 |
| BR-21-02 | 関連ユーザー同時削除 | 従業員に紐づくユーザーアカウントが存在する場合は同時に削除する | UserIdがnullでない場合 |
| BR-21-03 | 存在確認必須 | 削除対象の従業員が存在しない場合はエラーとする | 指定IDの従業員が見つからない場合 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 従業員削除 | Employees | DELETE | 指定IDの従業員レコードを削除 |
| 存在確認 | Employees | SELECT | 指定IDの従業員存在確認 |

### テーブル別操作詳細

#### Employees

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | EmployeeId | WHERE EmployeeId = @id | FindAsyncで検索 |
| DELETE | - | WHERE EmployeeId = @id | レコード全体を削除 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | NotFoundException | 指定IDの従業員が存在しない | Entity "Employee" ({id}) was not found. |
| 400 | BadRequestException | 自分自身を削除しようとした | Employees cannot delete their own account. |

### リトライ仕様

特になし（同期処理のため）

## トランザクション仕様

- Entity Frameworkのデフォルトトランザクション管理を使用
- SaveChangesAsyncの呼び出し時にコミットされる
- ユーザーアカウント削除とEmployees削除は別トランザクションで実行される（注意：UserManager.DeleteUserAsyncは別コンテキストで実行）

## パフォーマンス要件

- レスポンス時間：1秒以内を推奨
- FindAsyncによる主キー検索のため、インデックスが有効活用される

## セキュリティ考慮事項

- 認証済みユーザーのみアクセス可能（BaseControllerで制御）
- 自分自身のアカウント削除を防止するセキュリティチェックを実装
- 関連するユーザーアカウントも同時に削除されるため、認証情報の残存を防止

## 備考

- TODOコメントに記載がある通り、現在の実装では従業員に関連するテリトリーや注文がある場合、外部キー制約によりエラーとなる可能性がある
- 今後の改善として、関連データの整合性チェックや論理削除への変更を検討する必要がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、従業員エンティティとCQRSパターンのコマンドを理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Employee.cs | `Src/Domain/Entities/Employee.cs` | 従業員エンティティの構造、UserIdフィールドとの関連 |
| 1-2 | DeleteEmployeeCommand.cs | `Src/Application/Employees/Commands/DeleteEmployee/DeleteEmployeeCommand.cs` | コマンドの構造と必要なパラメータ |

**読解のコツ**: ASP.NET Coreでは、CQRSパターンとMediatRを使用している。Commandはデータ変更操作、Queryはデータ取得操作を担当する。

#### Step 2: エントリーポイントを理解する

処理の起点となるAPIコントローラーを特定。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | Deleteアクションメソッドの実装 |

**主要処理フロー**:
1. **39-47行目**: Deleteメソッドでリクエストを受け取り、DeleteEmployeeCommandを発行
2. **44行目**: Mediator.Sendでコマンドをハンドラーに送信

#### Step 3: ビジネスロジックを理解する

削除処理の本体であるコマンドハンドラーを読み解く。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DeleteEmployeeCommand.cs | `Src/Application/Employees/Commands/DeleteEmployee/DeleteEmployeeCommand.cs` | DeleteEmployeeCommandHandlerクラスのHandleメソッド |

**主要処理フロー**:
- **27-54行目**: Handleメソッドの全体フロー
- **29-30行目**: FindAsyncで従業員を検索
- **32-35行目**: 存在しない場合はNotFoundExceptionをスロー
- **37-40行目**: 自分自身の削除チェック、違反時はBadRequestException
- **42-45行目**: UserIdが存在する場合はユーザーアカウントを削除
- **49行目**: Employees.Remove(entity)で従業員を削除
- **51行目**: SaveChangesAsyncで変更を保存

#### Step 4: インターフェースを理解する

依存するサービスインターフェースを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ICurrentUserService.cs | `Src/Application/Common/Interfaces/ICurrentUserService.cs` | 現在ユーザー情報取得インターフェース |
| 4-2 | IUserManager.cs | `Src/Application/Common/Interfaces/IUserManager.cs` | ユーザー管理インターフェース |
| 4-3 | INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | データベースコンテキストインターフェース |

### プログラム呼び出し階層図

```
EmployeesController.Delete(id)
    │
    └─ Mediator.Send(DeleteEmployeeCommand)
           │
           └─ DeleteEmployeeCommandHandler.Handle()
                  │
                  ├─ INorthwindDbContext.Employees.FindAsync(id)
                  │
                  ├─ ICurrentUserService.UserId [比較用]
                  │
                  ├─ IUserManager.DeleteUserAsync(userId) [条件付き]
                  │
                  ├─ INorthwindDbContext.Employees.Remove(entity)
                  │
                  └─ INorthwindDbContext.SaveChangesAsync()
```

### データフロー図

```
[入力]                      [処理]                           [出力]

HTTPリクエスト          DeleteEmployeeCommand           HTTPレスポンス
DELETE /api/Employees/{id}                             204 No Content
       │                        │                           │
       ▼                        ▼                           │
    int id ──────────▶ FindAsync(id) ──────────▶ Employee    │
                              │                              │
                              ▼                              │
                       存在チェック                          │
                              │                              │
                              ▼                              │
                       自己削除チェック ◀─── ICurrentUserService.UserId
                              │                              │
                              ▼                              │
                    [UserId != null]                         │
                       DeleteUser ◀─── IUserManager          │
                              │                              │
                              ▼                              │
                    Employees.Remove() ─────────────────────▶│
                              │                              │
                              ▼                              │
                    SaveChangesAsync() ─────────────────────▶│
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| EmployeesController.cs | `Src/WebUI/Controllers/EmployeesController.cs` | コントローラー | APIエンドポイント定義 |
| DeleteEmployeeCommand.cs | `Src/Application/Employees/Commands/DeleteEmployee/DeleteEmployeeCommand.cs` | コマンド | 削除処理のビジネスロジック |
| Employee.cs | `Src/Domain/Entities/Employee.cs` | エンティティ | 従業員データモデル |
| ICurrentUserService.cs | `Src/Application/Common/Interfaces/ICurrentUserService.cs` | インターフェース | 現在ユーザー情報取得 |
| IUserManager.cs | `Src/Application/Common/Interfaces/IUserManager.cs` | インターフェース | ユーザー管理操作 |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | インターフェース | データベースアクセス |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | 例外クラス | 存在しないエンティティ例外 |
| BadRequestException.cs | `Src/Application/Common/Exceptions/BadRequestException.cs` | 例外クラス | 不正リクエスト例外 |
