# 機能設計書 23-例外ハンドリング

## 概要

本ドキュメントは、Northwind Tradersシステムにおける例外ハンドリング機能の設計仕様を記載する。本機能はアプリケーション全体で発生する例外を統一的に処理し、適切なHTTPレスポンスを返却するミドルウェア機能である。

### 本機能の処理概要

本機能は、ASP.NET Coreのミドルウェアパイプラインに組み込まれ、アプリケーション内で発生した例外をキャッチし、例外の種類に応じた適切なHTTPステータスコードとエラーメッセージをクライアントに返却する。

**業務上の目的・背景**：アプリケーション全体で統一されたエラー処理を実現し、開発者が各コントローラーで個別にエラー処理を実装する負担を軽減する。また、クライアントに対して一貫性のあるエラーレスポンス形式を提供することで、フロントエンド開発者が予測可能なエラー処理を実装できるようにする。

**機能の利用シーン**：
- バリデーションエラー発生時（ValidationException）
- リソースが見つからない場合（NotFoundException）
- 不正なリクエストの場合（BadRequestException）
- その他の予期しないエラー発生時

**主要な処理内容**：
1. HTTPリクエスト処理中に発生した例外をキャッチ
2. 例外の種類を判定し、対応するHTTPステータスコードを決定
3. エラーメッセージをJSON形式で整形
4. HTTPレスポンスとしてクライアントに返却

**関連システム・外部連携**：FluentValidationライブラリと連携し、バリデーションエラーの詳細を構造化された形式で返却する。

**権限による制御**：本機能は全てのHTTPリクエストに対して適用されるミドルウェアであり、権限による制御は行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | エラー画面 | 主機能 | CustomExceptionHandlerMiddlewareで処理されたエラー情報を受け取り、エラーメッセージとリクエストIDを表示する |

## 機能種別

バリデーション / エラー処理 / 横断的関心事

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| HttpContext | HttpContext | Yes | ASP.NET CoreのHTTPコンテキスト | - |
| Exception | Exception | Yes | キャッチされた例外オブジェクト | - |

### 入力データソース

ASP.NET Coreのミドルウェアパイプラインから提供されるHttpContextおよび発生した例外オブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTP Status Code | int | エラー種別に応じたステータスコード |
| Response Body | JSON | エラー詳細を含むJSONオブジェクト |

### 例外種別とHTTPステータスコードの対応

| 例外クラス | HTTPステータスコード | レスポンス形式 |
|-----------|---------------------|--------------|
| ValidationException | 400 Bad Request | `{ "PropertyName": ["Error1", "Error2"] }` |
| BadRequestException | 400 Bad Request | エラーメッセージ（テキスト） |
| NotFoundException | 404 Not Found | `{ "error": "Exception message" }` |
| その他の例外 | 500 Internal Server Error | `{ "error": "Exception message" }` |

### 出力先

HTTPレスポンスボディ（Content-Type: application/json）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ ミドルウェアパイプラインでInvoke()が呼び出される

2. 後続ミドルウェア・コントローラー呼び出し
   └─ await _next(context) で処理を委譲

3. 例外発生時のキャッチ
   └─ try-catchブロックで例外をキャッチ

4. 例外種別の判定
   └─ switch文で例外の型を判定

5. HTTPステータスコード決定
   └─ ValidationException → 400
   └─ BadRequestException → 400
   └─ NotFoundException → 404
   └─ その他 → 500

6. レスポンス整形
   └─ JsonConvert.SerializeObjectでJSON化
   └─ Content-Type: application/jsonを設定

7. レスポンス返却
   └─ context.Response.WriteAsync(result)
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[Invoke メソッド]
    B --> C[try: await _next context]
    C --> D{例外発生?}
    D -->|No| E[正常レスポンス返却]
    D -->|Yes| F[HandleExceptionAsync]
    F --> G{例外種別判定}
    G -->|ValidationException| H[400 + Failures JSON]
    G -->|BadRequestException| I[400 + Message]
    G -->|NotFoundException| J[404 + empty result]
    G -->|その他| K[500 + error JSON]
    H --> L[Response.WriteAsync]
    I --> L
    J --> L
    K --> L
    L --> M[終了]
    E --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | バリデーションエラー形式 | ValidationExceptionのFailuresプロパティをそのままJSON化して返却 | ValidationExceptionの場合 |
| BR-23-02 | BadRequest形式 | エラーメッセージをそのまま返却（JSON化しない） | BadRequestExceptionの場合 |
| BR-23-03 | NotFound形式 | ステータスコード404のみ返却、resultは空文字 | NotFoundExceptionの場合 |
| BR-23-04 | 汎用エラー形式 | エラーメッセージを`{ "error": "message" }`形式で返却 | その他の例外の場合 |
| BR-23-05 | Content-Type固定 | 全てのエラーレスポンスでContent-Typeは`application/json` | 常に適用 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能はデータベース操作を行わない |

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | ValidationException | FluentValidationによるバリデーション失敗 | クライアント側で入力値を修正 |
| 400 | BadRequestException | ビジネスロジックでの不正リクエスト検知 | クライアント側でリクエスト内容を修正 |
| 404 | NotFoundException | 指定されたリソースが存在しない | 正しいリソースIDを指定 |
| 500 | InternalServerError | 予期しないサーバーエラー | システム管理者に連絡 |

### リトライ仕様

本機能自体にリトライ仕様はない。クライアント側で必要に応じてリトライを実装する。

## トランザクション仕様

該当なし（本機能はデータベース操作を行わない）

## パフォーマンス要件

- エラー処理は即座に完了すること（追加の遅延なし）
- JSONシリアライズのオーバーヘッドは最小限に抑える

## セキュリティ考慮事項

- **エラーメッセージの内容**：本番環境では、例外の詳細（スタックトレースなど）をクライアントに返却しないよう注意が必要。現在の実装ではException.Messageをそのまま返却しているため、本番環境でのセンシティブな情報漏洩に注意。
- **例外の種類による情報漏洩**：NotFoundException等の発生により、リソースの存在有無が推測可能になる点に注意。

## 備考

- Newtonsoft.Json（Json.NET）を使用してJSONシリアライズを行っている
- 拡張メソッド`UseCustomExceptionHandler()`により、Startup.csで簡潔にミドルウェアを登録可能
- 開発環境ではUseDeveloperExceptionPage()が先に適用されるため、このミドルウェアの動作が異なる場合がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: ミドルウェアの構造を理解する

ASP.NET Coreのミドルウェアパターンを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CustomExceptionHandlerMiddleware.cs | `Src/WebUI/Common/CustomExceptionHandlerMiddleware.cs` | ミドルウェアクラス全体の構造 |

**主要処理フロー**:
- **11-18行目**: クラス定義とコンストラクタ。RequestDelegate _nextを保持
- **20-30行目**: Invoke()メソッド。try-catchで後続処理をラップ
- **22-24行目**: await _next(context)で後続ミドルウェアに処理を委譲
- **26-29行目**: 例外発生時にHandleExceptionAsyncを呼び出し

**読解のコツ**: ASP.NET Coreのミドルウェアパターンでは、コンストラクタでRequestDelegateを受け取り、Invoke()メソッドで処理を行う。後続ミドルウェアへの委譲は`await _next(context)`で行う。

#### Step 2: 例外処理ロジックを理解する

例外の種類に応じた処理分岐を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomExceptionHandlerMiddleware.cs | `Src/WebUI/Common/CustomExceptionHandlerMiddleware.cs` | HandleExceptionAsyncメソッド |

**主要処理フロー**:
- **32-62行目**: HandleExceptionAsync()メソッド全体
- **34-36行目**: デフォルトはInternalServerError（500）
- **38-51行目**: switch文で例外種別を判定
- **40-43行目**: ValidationException → 400、Failuresをシリアライズ
- **44-47行目**: BadRequestException → 400、メッセージをそのまま使用
- **48-50行目**: NotFoundException → 404、resultは空
- **53-54行目**: Content-TypeとStatusCodeを設定
- **56-59行目**: resultが空の場合はデフォルトのエラーJSON生成
- **61行目**: レスポンス書き込み

#### Step 3: 例外クラスを理解する

アプリケーションで使用される例外クラスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ValidationException.cs | `Src/Application/Common/Exceptions/ValidationException.cs` | Failuresプロパティの構造 |
| 3-2 | BadRequestException.cs | `Src/Application/Common/Exceptions/BadRequestException.cs` | シンプルなメッセージ例外 |
| 3-3 | NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | エンティティ名とキーを含むメッセージ |

**ValidationExceptionの主要処理**:
- **10-13行目**: デフォルトコンストラクタでFailuresを初期化
- **15-31行目**: ValidationFailureリストから、プロパティ名をキーとしたDictionaryを構築
- **34行目**: `IDictionary<string, string[]> Failures`プロパティ

#### Step 4: ミドルウェア登録を理解する

ミドルウェアがパイプラインに登録される仕組みを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CustomExceptionHandlerMiddleware.cs | `Src/WebUI/Common/CustomExceptionHandlerMiddleware.cs` | 拡張メソッドクラス |
| 4-2 | Startup.cs | `Src/WebUI/Startup.cs` | Configure()でのミドルウェア登録 |

**拡張メソッドの実装**:
- **65-70行目**: CustomExceptionHandlerMiddlewareExtensionsクラス
- **67-69行目**: UseCustomExceptionHandler()拡張メソッドの定義

**Startup.csでの使用**:
- **89行目**: `app.UseCustomExceptionHandler();`でパイプラインに登録

### プログラム呼び出し階層図

```
HTTP Request
    │
    ▼
CustomExceptionHandlerMiddleware.Invoke(context)
    │
    ├── try { await _next(context) }
    │         │
    │         └── [後続ミドルウェア/コントローラー]
    │                    │
    │                    └── [例外発生の可能性]
    │
    └── catch (Exception ex)
              │
              └── HandleExceptionAsync(context, ex)
                        │
                        ├── [例外種別判定]
                        │      │
                        │      ├── ValidationException
                        │      │      └── JsonConvert.SerializeObject(failures)
                        │      │
                        │      ├── BadRequestException
                        │      │      └── message直接使用
                        │      │
                        │      ├── NotFoundException
                        │      │      └── 空のresult
                        │      │
                        │      └── その他
                        │             └── JsonConvert.SerializeObject({ error })
                        │
                        └── context.Response.WriteAsync(result)
```

### データフロー図

```
[入力]                    [処理]                              [出力]

HTTPリクエスト       CustomExceptionHandler              HTTPレスポンス
     │                      Middleware
     │                         │
     ▼                         ▼
HttpContext ─────▶ try { _next(context) } ─────▶ 正常レスポンス
                              │
                              │ [例外発生時]
                              ▼
Exception ─────────▶ HandleExceptionAsync()
                              │
                              ▼
                      [例外種別判定]
                              │
         ┌─────────────┼─────────────┬─────────────┐
         ▼             ▼             ▼             ▼
ValidationException BadRequest  NotFoundException  Other
         │             │             │             │
         ▼             ▼             ▼             ▼
   Failures JSON   Message      Empty result   Error JSON
         │             │             │             │
         └─────────────┴─────────────┴─────────────┘
                              │
                              ▼
                    context.Response
                    .WriteAsync(result)
                              │
                              ▼
                    JSON Response
                    (Content-Type: application/json)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomExceptionHandlerMiddleware.cs | `Src/WebUI/Common/CustomExceptionHandlerMiddleware.cs` | ソース | 例外ハンドリングミドルウェア本体 |
| ValidationException.cs | `Src/Application/Common/Exceptions/ValidationException.cs` | ソース | バリデーションエラー例外クラス |
| BadRequestException.cs | `Src/Application/Common/Exceptions/BadRequestException.cs` | ソース | 不正リクエスト例外クラス |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | リソース未検出例外クラス |
| Startup.cs | `Src/WebUI/Startup.cs` | ソース | ミドルウェア登録設定 |
