# 機能設計書 24-現在ユーザー情報取得

## 概要

本ドキュメントは、Northwind Tradersシステムにおける現在ユーザー情報取得機能の設計仕様を記載する。

### 本機能の処理概要

この機能は、現在HTTPリクエストを実行しているユーザーの情報（ユーザーID、認証状態）を取得するサービス機能である。ASP.NET Core IdentityとHTTPコンテキストを連携させ、アプリケーション層のビジネスロジックで現在ユーザーの情報を参照可能にする。

**業務上の目的・背景**：多くのビジネスロジックでは「誰が」操作を実行しているかを知る必要がある。例えば、従業員削除時に自分自身を削除できないようにする制約や、リクエストログに実行者情報を含めるためには、現在のユーザー情報へのアクセスが必須である。WebUI層に依存せずにApplication層でユーザー情報を参照できるよう、インターフェースを通じた依存性注入パターンで実装されている。

**機能の利用シーン**：
- 従業員削除時の自己削除チェック
- リクエストログへのユーザーID記録
- パフォーマンス監視ログへのユーザーID記録
- 監査証跡（Audit Trail）の記録

**主要な処理内容**：
1. HTTPコンテキストからClaimsPrincipalを取得
2. ClaimsPrincipalからNameIdentifierクレームを抽出しUserIdとして返却
3. UserIdの有無から認証状態（IsAuthenticated）を判定

**関連システム・外部連携**：ASP.NET Core Identity、IdentityServer4と連携し、JWT/Cookieから認証情報を取得する。

**権限による制御**：本機能自体は権限による制御を行わない。呼び出し元で認証・認可の判定に使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接連携しないが、すべての認証済みリクエストで間接的に使用される |

## 機能種別

共通基盤機能 / サービス / 横断的関心事

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| IHttpContextAccessor | IHttpContextAccessor | Yes | HTTPコンテキストへのアクセサ（DIで注入） | - |

### 入力データソース

HTTPリクエストのClaimsPrincipal（HttpContext.User）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| UserId | string | 現在ユーザーのID（未認証時はnull） |
| IsAuthenticated | bool | 認証状態（UserIdがnullでなければtrue） |

### 出力先

呼び出し元のコンポーネントへの直接返却（プロパティアクセス）

## 処理フロー

### 処理シーケンス

```
1. CurrentUserServiceのインスタンス生成（DI）
   └─ コンストラクタでIHttpContextAccessorを受け取る
2. UserId取得
   └─ httpContextAccessor.HttpContext?.User?.FindFirstValue(ClaimTypes.NameIdentifier)
3. IsAuthenticated判定
   └─ UserId != null
```

### フローチャート

```mermaid
flowchart TD
    A[サービスインスタンス生成] --> B[IHttpContextAccessor注入]
    B --> C{HttpContext存在?}
    C -->|No| D[UserId = null]
    C -->|Yes| E{User存在?}
    E -->|No| D
    E -->|Yes| F[FindFirstValue NameIdentifier]
    F --> G{クレーム存在?}
    G -->|No| D
    G -->|Yes| H[UserId = クレーム値]
    D --> I[IsAuthenticated = false]
    H --> J[IsAuthenticated = true]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-01 | ユーザーID取得 | ClaimTypes.NameIdentifierクレームからユーザーIDを取得 | HTTPコンテキストが存在する場合 |
| BR-24-02 | 認証状態判定 | UserIdがnullでなければ認証済みとみなす | 常に適用 |
| BR-24-03 | null安全性 | HttpContextやUserがnullの場合も例外を発生させずnullを返却 | HttpContext/Userがnullの場合 |

### 計算ロジック

```
IsAuthenticated = (UserId != null)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| なし | - | - | 本機能はデータベースを操作しない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | HTTPコンテキスト外での呼び出し | UserIdがnull、IsAuthenticatedがfalseとなる |

### リトライ仕様

リトライ不要（同期的なプロパティアクセス）

## トランザクション仕様

該当なし（データベース操作を行わない）

## パフォーマンス要件

- プロパティアクセスは即座に完了すること
- コンストラクタでの値取得は軽量であること

## セキュリティ考慮事項

- **クレーム改竄防止**：ClaimsPrincipalはASP.NET Core Identityにより署名/暗号化されているため、改竄を検知可能
- **認証状態の正確性**：IsAuthenticatedプロパティはUserIdの有無のみで判定しており、トークンの有効期限等は考慮していない点に注意
- **HTTPコンテキスト依存**：バックグラウンドタスク等、HTTPコンテキストがない環境では使用不可

## 備考

- ICurrentUserServiceインターフェースはApplication層に定義され、実装（CurrentUserService）はWebUI層に配置される
- このパターンにより、Application層はWebUI層に依存せず、テスト時にモックを注入可能
- Startup.csでScoped（リクエストごと）としてDIコンテナに登録される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

まず、Application層で定義されているインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ICurrentUserService.cs | `Src/Application/Common/Interfaces/ICurrentUserService.cs` | インターフェース定義 |

**主要処理フロー**:
- **3-8行目**: インターフェース定義全体
- **5行目**: `string UserId { get; }` - 現在ユーザーIDのプロパティ
- **7行目**: `bool IsAuthenticated { get; }` - 認証状態のプロパティ

**読解のコツ**: このインターフェースはApplication層に定義されており、実装詳細を隠蔽している。これによりApplication層はWebUI層に依存せず、クリーンアーキテクチャの依存関係の方向を維持している。

#### Step 2: 実装クラスを理解する

インターフェースの実装であるCurrentUserServiceを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CurrentUserService.cs | `Src/WebUI/Services/CurrentUserService.cs` | 実装クラス |

**主要処理フロー**:
- **7-18行目**: CurrentUserServiceクラス全体
- **9-13行目**: コンストラクタ - IHttpContextAccessorから情報を取得
- **11行目**: `UserId = httpContextAccessor.HttpContext?.User?.FindFirstValue(ClaimTypes.NameIdentifier)` - クレームからユーザーID取得
- **12行目**: `IsAuthenticated = UserId != null` - 認証状態判定
- **15行目**: `public string UserId { get; }` - ユーザーIDプロパティ
- **17行目**: `public bool IsAuthenticated { get; }` - 認証状態プロパティ

**読解のコツ**: C#のnull条件演算子（?.）を使用して、nullチェーンを安全にナビゲートしている。HttpContext、User、クレームのいずれかがnullでも例外は発生しない。

#### Step 3: DI登録を理解する

サービスがどのようにDIコンテナに登録されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Startup.cs | `Src/WebUI/Startup.cs` | ConfigureServicesでのDI登録 |

**主要処理フロー**:
- **42行目**: `services.AddScoped<ICurrentUserService, CurrentUserService>()` - Scopedとして登録
- **44行目**: `services.AddHttpContextAccessor()` - IHttpContextAccessorの登録

**読解のコツ**: AddScoped()により、HTTPリクエストごとに新しいインスタンスが生成される。これにより各リクエストで異なるユーザー情報を保持できる。

#### Step 4: 使用箇所を理解する

ICurrentUserServiceがどこで使用されているかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | RequestLogger.cs | `Src/Application/Common/Behaviours/RequestLogger.cs` | リクエストログでの使用 |
| 4-2 | RequestPerformanceBehaviour.cs | `Src/Application/Common/Behaviours/RequestPerformanceBehaviour.cs` | パフォーマンス監視での使用 |
| 4-3 | DeleteEmployeeCommand.cs | `Src/Application/Employees/Commands/DeleteEmployee/DeleteEmployeeCommand.cs` | 従業員削除での使用 |

**RequestLoggerでの使用**:
- **12行目**: `private readonly ICurrentUserService _currentUserService;` - フィールド定義
- **14-17行目**: コンストラクタで注入
- **24-25行目**: ログにユーザーIDを含める

**DeleteEmployeeCommandでの使用**:
- **18行目**: `private readonly ICurrentUserService _currentUser;` - フィールド定義
- **37行目**: `if (entity.UserId == _currentUser.UserId)` - 自己削除チェック

### プログラム呼び出し階層図

```
[DIコンテナ]
    │
    └─ AddScoped<ICurrentUserService, CurrentUserService>()
           │
           ▼
HTTPリクエスト
    │
    └─ CurrentUserService インスタンス生成
           │
           ├─ IHttpContextAccessor.HttpContext.User
           │         │
           │         └─ FindFirstValue(ClaimTypes.NameIdentifier)
           │                   │
           │                   └─ UserId プロパティ設定
           │
           └─ IsAuthenticated = (UserId != null)

[使用箇所]
    │
    ├─ RequestLogger
    │      └─ _currentUserService.UserId → ログ出力
    │
    ├─ RequestPerformanceBehaviour
    │      └─ _currentUserService.UserId → 遅延ログ出力
    │
    └─ DeleteEmployeeCommand
           └─ _currentUser.UserId → 自己削除チェック
```

### データフロー図

```
[入力]                      [処理]                           [出力]

HTTPリクエスト          CurrentUserService               プロパティ値
     │                        │                              │
     ▼                        ▼                              │
 認証ヘッダー ─────▶ ASP.NET Core Identity              │
 (JWT/Cookie)              │                              │
     │                        ▼                              │
     │                 ClaimsPrincipal                      │
     │                        │                              │
     │                        ▼                              │
     │                 HttpContext.User                     │
     │                        │                              │
     │                        ▼                              │
     │          FindFirstValue(NameIdentifier) ─────────▶ UserId: string
     │                        │                              │
     │                        ▼                              │
     │              (UserId != null) ─────────────────▶ IsAuthenticated: bool
     │                        │
     │                        │
     ▼                        ▼
[使用例]            ICurrentUserService.UserId
     │                        │
     ├─ RequestLogger ◀───────┤
     │                        │
     ├─ RequestPerformanceBehaviour ◀─┤
     │                        │
     └─ DeleteEmployeeCommand ◀──────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ICurrentUserService.cs | `Src/Application/Common/Interfaces/ICurrentUserService.cs` | インターフェース | 現在ユーザー情報取得の抽象化 |
| CurrentUserService.cs | `Src/WebUI/Services/CurrentUserService.cs` | 実装 | HTTPコンテキストからユーザー情報を取得 |
| Startup.cs | `Src/WebUI/Startup.cs` | 構成 | DIコンテナへのサービス登録 |
| RequestLogger.cs | `Src/Application/Common/Behaviours/RequestLogger.cs` | ビヘイビア | リクエストログ記録（使用箇所） |
| RequestPerformanceBehaviour.cs | `Src/Application/Common/Behaviours/RequestPerformanceBehaviour.cs` | ビヘイビア | パフォーマンス監視（使用箇所） |
| DeleteEmployeeCommand.cs | `Src/Application/Employees/Commands/DeleteEmployee/DeleteEmployeeCommand.cs` | コマンド | 従業員削除（使用箇所） |
