# 機能設計書 4-顧客一覧取得

## 概要

本ドキュメントは、NorthwindTradersシステムにおける顧客一覧取得機能の設計仕様を記載する。REST APIを通じて全ての顧客情報を一覧形式で取得する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：顧客管理業務において、登録されている全顧客を一覧で確認する必要がある。営業担当者が顧客リストを参照したり、顧客への連絡先リストを作成したりする際に利用される。CQRSパターンに基づくQuery処理として実装されている。

**機能の利用シーン**：
1. 顧客一覧画面の初期表示時に全顧客データを取得
2. 顧客検索後に結果一覧を表示する際のベースデータ取得
3. 顧客選択ドロップダウン等のマスタデータ取得

**主要な処理内容**：
1. HTTPクライアントからGET /api/Customersへリクエスト
2. CustomersControllerがリクエストを受信
3. MediatRを通じてGetCustomersListQueryを発行
4. GetCustomersListQueryHandlerがクエリを処理
5. INorthwindDbContextを通じてCustomersテーブルからデータ取得
6. AutoMapperでCustomerエンティティをCustomerLookupDtoにマッピング
7. CustomersListVmとしてレスポンス返却

**関連システム・外部連携**：Angular SPAクライアントのCustomersClientからの呼び出し。

**権限による制御**：[Authorize]属性により認証済みユーザーのみアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 顧客一覧画面 | 主画面 | 画面初期表示時にCustomersClient.getAll()を呼び出し、全顧客の一覧を取得・表示する |

## 機能種別

CRUD操作（Read） / データ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | パラメータなし | - |

### 入力データソース

HTTPリクエスト: GET /api/Customers

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Customers | IList<CustomerLookupDto> | 顧客情報のリスト |
| Customers[].Id | string | 顧客ID（5文字） |
| Customers[].Name | string | 会社名 |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. CustomersController.GetAll()
   └─ Mediator.Send(new GetCustomersListQuery())
       └─ GetCustomersListQueryHandler.Handle()
           ├─ _context.Customers（DbSet<Customer>）からデータ取得
           ├─ ProjectTo<CustomerLookupDto>()でマッピング
           ├─ ToListAsync()で非同期実行
           └─ CustomersListVmにラップして返却
2. Ok(vm)でHTTP 200レスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/Customers] --> B[CustomersController.GetAll]
    B --> C[Mediator.Send]
    C --> D[GetCustomersListQueryHandler.Handle]
    D --> E[_context.Customers]
    E --> F[ProjectTo CustomerLookupDto]
    F --> G[ToListAsync]
    G --> H[CustomersListVm作成]
    H --> I[Ok vm 返却]
    I --> J[HTTP 200 JSON Response]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 全件取得 | 登録されている全ての顧客を取得 | 常時 |
| BR-002 | 認証必須 | 認証済みユーザーのみアクセス可能 | 常時 |
| BR-003 | マッピング | CustomerIdはIdに、CompanyNameはNameにマッピング | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 顧客一覧取得 | Customers | SELECT | 全顧客レコード取得 |

### テーブル別操作詳細

#### Customers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CustomerID | 全件 | IdにマッピングされてDTOに出力 |
| SELECT | CompanyName | 全件 | NameにマッピングされてDTOに出力 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証アクセス | ログイン後に再試行 |
| 500 | InternalServerError | データベースエラー等 | システム管理者へ連絡 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。

## トランザクション仕様

読み取り専用クエリのため、明示的なトランザクション管理は不要。Entity Frameworkのデフォルトの分離レベルが適用される。

## パフォーマンス要件

- レスポンス時間：1秒以内
- ProjectTo<>()によるクエリ最適化（必要なカラムのみSELECT）

## セキュリティ考慮事項

- [Authorize]属性による認証チェック
- SQLインジェクション対策はEntity Frameworkが自動で実施
- 出力データは必要最小限（Id, Name）に絞り込み

## 備考

- MediatRによるCQRSパターンを採用
- AutoMapperによるオブジェクトマッピング

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

DTOとViewModelの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CustomerLookupDto.cs | `Src/Application/Customers/Queries/GetCustomersList/CustomerLookupDto.cs` | 顧客一覧用DTO。Id, Nameの2プロパティのみ |
| 1-2 | CustomersListVm.cs | `Src/Application/Customers/Queries/GetCustomersList/CustomersListVm.cs` | DTOリストをラップするViewModel |
| 1-3 | Customer.cs | `Src/Domain/Entities/Customer.cs` | ドメインエンティティ。全プロパティを確認 |

**読解のコツ**:
- CustomerLookupDto.csの**12-17行目**にAutoMapperのマッピング設定がある
- `CustomerId → Id`、`CompanyName → Name`のマッピングを確認

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **13行目**: [Authorize]属性で認証必須を設定
2. **16-22行目**: GetAll()メソッドでHTTP GETリクエストを処理
3. **19行目**: Mediator.Send()でGetCustomersListQueryを発行
4. **21行目**: Ok(vm)でHTTP 200レスポンス返却

#### Step 3: クエリハンドラを理解する

MediatRのRequestHandlerがビジネスロジックを実行する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GetCustomersListQuery.cs | `Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQuery.cs` | クエリ定義（パラメータなし） |
| 3-2 | GetCustomersListQueryHandler.cs | `Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQueryHandler.cs` | クエリハンドラ実装 |

**主要処理フロー**:
- **22-34行目**: Handle()メソッドの実装
- **24-26行目**: INorthwindDbContext.CustomersからProjectTo<>()でマッピングしながら取得
- **28-31行目**: CustomersListVmにラップして返却

#### Step 4: データアクセス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | DbSetインターフェース定義 |
| 4-2 | NorthwindDbContext.cs | `Src/Persistence/NorthwindDbContext.cs` | DbContext実装 |

**主要処理フロー**:
- **32行目**: DbSet<Customer> Customersプロパティ

### プログラム呼び出し階層図

```
Angular SPA (CustomersClient.getAll())
    │
    ▼
GET /api/Customers
    │
    └─ CustomersController.GetAll()
           │
           └─ Mediator.Send(GetCustomersListQuery)
                  │
                  └─ GetCustomersListQueryHandler.Handle()
                         │
                         ├─ INorthwindDbContext.Customers
                         │      └─ DbSet<Customer>
                         │
                         ├─ ProjectTo<CustomerLookupDto>()
                         │      └─ AutoMapper マッピング
                         │
                         └─ CustomersListVm
```

### データフロー図

```
[入力]                    [処理]                              [出力]

HTTP GET Request  ───▶  CustomersController              ───▶  HTTP 200 Response
                  ───▶  GetCustomersListQueryHandler     ───▶  CustomersListVm
Customers Table   ───▶  ProjectTo<CustomerLookupDto>     ───▶  JSON
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | ソース | APIコントローラー |
| GetCustomersListQuery.cs | `Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQuery.cs` | ソース | クエリ定義 |
| GetCustomersListQueryHandler.cs | `Src/Application/Customers/Queries/GetCustomersList/GetCustomersListQueryHandler.cs` | ソース | クエリハンドラ |
| CustomerLookupDto.cs | `Src/Application/Customers/Queries/GetCustomersList/CustomerLookupDto.cs` | ソース | DTO定義 |
| CustomersListVm.cs | `Src/Application/Customers/Queries/GetCustomersList/CustomersListVm.cs` | ソース | ViewModel定義 |
| Customer.cs | `Src/Domain/Entities/Customer.cs` | ソース | ドメインエンティティ |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DbContextインターフェース |
| NorthwindDbContext.cs | `Src/Persistence/NorthwindDbContext.cs` | ソース | DbContext実装 |
| CustomerConfiguration.cs | `Src/Persistence/Configurations/CustomerConfiguration.cs` | ソース | EF Coreエンティティ設定 |
