# 機能設計書 5-顧客詳細取得

## 概要

本ドキュメントは、NorthwindTradersシステムにおける顧客詳細取得機能の設計仕様を記載する。REST APIを通じて指定した顧客IDに基づいて顧客の詳細情報を取得する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：顧客管理業務において、特定の顧客の詳細情報（会社名、担当者名、住所、連絡先等）を確認する必要がある。顧客との連絡時や請求書作成時など、詳細な顧客情報が必要な場面で利用される。

**機能の利用シーン**：
1. 顧客一覧画面から特定の顧客を選択した際に詳細情報を取得
2. 顧客詳細モーダルに表示するデータの取得
3. 顧客情報の確認・編集前の現在値取得

**主要な処理内容**：
1. HTTPクライアントからGET /api/Customers/{id}へリクエスト
2. CustomersControllerがリクエストを受信
3. MediatRを通じてGetCustomerDetailQueryを発行
4. GetCustomerDetailQueryHandlerがクエリを処理
5. バリデーション（ID形式チェック）
6. INorthwindDbContext.Customers.FindAsync()でデータ取得
7. 存在しない場合はNotFoundExceptionをスロー
8. AutoMapperでCustomerエンティティをCustomerDetailVmにマッピング
9. CustomerDetailVmとしてレスポンス返却

**関連システム・外部連携**：Angular SPAクライアントのCustomersClientからの呼び出し。

**権限による制御**：[Authorize]属性により認証済みユーザーのみアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 顧客一覧画面 | 参照画面 | 顧客行クリック時にCustomersClient.get(id)を呼び出し、顧客詳細モーダルに表示するデータを取得する |
| 4 | 顧客詳細モーダル | 主画面 | 顧客一覧画面から渡された顧客詳細情報を表示する。データ取得は親画面で実行済み |

## 機能種別

CRUD操作（Read） / データ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | string | Yes | 顧客ID | NotEmpty, 5文字固定 |

### 入力データソース

HTTPリクエスト: GET /api/Customers/{id}

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | string | 顧客ID |
| Address | string | 住所 |
| City | string | 市区町村 |
| CompanyName | string | 会社名 |
| ContactName | string | 担当者名 |
| ContactTitle | string | 担当者役職 |
| Country | string | 国名 |
| Fax | string | FAX番号 |
| Phone | string | 電話番号 |
| PostalCode | string | 郵便番号 |
| Region | string | 地域 |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. CustomersController.Get(id)
   └─ Mediator.Send(new GetCustomerDetailQuery { Id = id })
       └─ FluentValidation: GetCustomerDetailQueryValidator
           └─ Id: NotEmpty, Length(5)
       └─ GetCustomerDetailQueryHandler.Handle()
           ├─ _context.Customers.FindAsync(request.Id)
           ├─ entity == null の場合、NotFoundException throw
           └─ _mapper.Map<CustomerDetailVm>(entity)
2. 成功時：Ok(vm)でHTTP 200レスポンス返却
3. 失敗時：CustomExceptionHandlerMiddlewareでHTTP 404変換
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/Customers/id] --> B[CustomersController.Get]
    B --> C[Mediator.Send]
    C --> D[GetCustomerDetailQueryValidator]
    D --> E{バリデーション成功?}
    E -->|No| F[ValidationException]
    F --> G[HTTP 400 Bad Request]
    E -->|Yes| H[GetCustomerDetailQueryHandler.Handle]
    H --> I[_context.Customers.FindAsync]
    I --> J{顧客存在?}
    J -->|No| K[NotFoundException]
    K --> L[HTTP 404 Not Found]
    J -->|Yes| M[AutoMapper Map]
    M --> N[Ok vm 返却]
    N --> O[HTTP 200 JSON Response]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ID形式検証 | 顧客IDは5文字固定 | 常時 |
| BR-002 | 認証必須 | 認証済みユーザーのみアクセス可能 | 常時 |
| BR-003 | 存在確認 | 指定IDの顧客が存在しない場合は404エラー | 常時 |
| BR-004 | マッピング | CustomerId→Idにマッピング、他は同名プロパティで自動マッピング | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 顧客詳細取得 | Customers | SELECT | 主キーによる1件取得 |

### テーブル別操作詳細

#### Customers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | CustomerID | WHERE CustomerID = @id | 主キー検索 |
| SELECT | CompanyName | 上記条件に合致 | |
| SELECT | ContactName | 上記条件に合致 | |
| SELECT | ContactTitle | 上記条件に合致 | |
| SELECT | Address | 上記条件に合致 | |
| SELECT | City | 上記条件に合致 | |
| SELECT | Region | 上記条件に合致 | |
| SELECT | PostalCode | 上記条件に合致 | |
| SELECT | Country | 上記条件に合致 | |
| SELECT | Phone | 上記条件に合致 | |
| SELECT | Fax | 上記条件に合致 | |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | IDが空または5文字でない | 正しいID形式で再試行 |
| 401 | Unauthorized | 未認証アクセス | ログイン後に再試行 |
| 404 | NotFound | 指定IDの顧客が存在しない | IDを確認して再試行 |
| 500 | InternalServerError | データベースエラー等 | システム管理者へ連絡 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。

## トランザクション仕様

読み取り専用クエリのため、明示的なトランザクション管理は不要。

## パフォーマンス要件

- レスポンス時間：500ms以内
- 主キー検索のため高速なレスポンスが期待できる

## セキュリティ考慮事項

- [Authorize]属性による認証チェック
- SQLインジェクション対策はEntity Frameworkが自動で実施
- IDのバリデーションによる不正入力防止

## 備考

- MediatRによるCQRSパターンを採用
- FluentValidationによる入力検証
- AutoMapperによるオブジェクトマッピング

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ViewModelとドメインエンティティの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CustomerDetailVm.cs | `Src/Application/Customers/Queries/GetCustomerDetail/CustomerDetailVm.cs` | 顧客詳細ViewModel。全プロパティを確認 |
| 1-2 | Customer.cs | `Src/Domain/Entities/Customer.cs` | ドメインエンティティ |

**読解のコツ**:
- CustomerDetailVm.csの**21-25行目**にAutoMapperのマッピング設定がある
- `CustomerId → Id`のマッピング以外は同名プロパティで自動マッピング

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **24-32行目**: Get(id)メソッドでHTTP GETリクエストを処理
2. **25-26行目**: ProducesResponseType属性でレスポンス型を定義（200, 404）
3. **29行目**: Mediator.Send()でGetCustomerDetailQueryを発行
4. **31行目**: Ok(vm)でHTTP 200レスポンス返却

#### Step 3: バリデーションを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GetCustomerDetailQueryValidator.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQueryValidator.cs` | FluentValidationによるバリデーション |

**主要処理フロー**:
- **7-10行目**: RuleFor(v => v.Id).NotEmpty().Length(5) - IDの必須・長さチェック

#### Step 4: クエリハンドラを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | GetCustomerDetailQuery.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQuery.cs` | クエリ定義（Idパラメータ） |
| 4-2 | GetCustomerDetailQueryHandler.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQueryHandler.cs` | クエリハンドラ実装 |

**主要処理フロー**:
- **22-33行目**: Handle()メソッドの実装
- **24-25行目**: FindAsync()で主キー検索
- **27-30行目**: 存在チェックとNotFoundException
- **32行目**: AutoMapperでマッピング

### プログラム呼び出し階層図

```
Angular SPA (CustomersClient.get(id))
    │
    ▼
GET /api/Customers/{id}
    │
    └─ CustomersController.Get(id)
           │
           └─ Mediator.Send(GetCustomerDetailQuery)
                  │
                  ├─ GetCustomerDetailQueryValidator
                  │      └─ FluentValidation
                  │
                  └─ GetCustomerDetailQueryHandler.Handle()
                         │
                         ├─ INorthwindDbContext.Customers.FindAsync()
                         │      └─ SELECT * FROM Customers WHERE CustomerID = @id
                         │
                         ├─ NotFoundException（存在しない場合）
                         │      └─ CustomExceptionHandlerMiddleware
                         │             └─ HTTP 404 Not Found
                         │
                         └─ AutoMapper.Map<CustomerDetailVm>()
                                └─ HTTP 200 OK
```

### データフロー図

```
[入力]                    [処理]                              [出力]

id (パスパラメータ) ───▶  GetCustomerDetailQueryValidator  ───▶  ValidationException or OK
                    ───▶  GetCustomerDetailQueryHandler    ───▶  NotFoundException or CustomerDetailVm
Customers Table     ───▶  FindAsync                        ───▶  Customer Entity
Customer Entity     ───▶  AutoMapper                       ───▶  CustomerDetailVm (JSON)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | ソース | APIコントローラー |
| GetCustomerDetailQuery.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQuery.cs` | ソース | クエリ定義 |
| GetCustomerDetailQueryHandler.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQueryHandler.cs` | ソース | クエリハンドラ |
| GetCustomerDetailQueryValidator.cs | `Src/Application/Customers/Queries/GetCustomerDetail/GetCustomerDetailQueryValidator.cs` | ソース | バリデーター |
| CustomerDetailVm.cs | `Src/Application/Customers/Queries/GetCustomerDetail/CustomerDetailVm.cs` | ソース | ViewModel定義 |
| Customer.cs | `Src/Domain/Entities/Customer.cs` | ソース | ドメインエンティティ |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
