# 機能設計書 6-顧客登録

## 概要

本ドキュメントは、NorthwindTradersシステムにおける顧客登録機能の設計仕様を記載する。REST APIを通じて新規顧客を登録する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：新規取引先との取引開始時に、顧客情報をシステムに登録する必要がある。会社名、担当者名、住所、連絡先等の情報を保存し、受注管理や請求書発行等の業務で活用する。CQRSパターンに基づくCommand処理として実装されている。

**機能の利用シーン**：
1. 新規取引先との契約締結時に顧客情報を登録
2. 展示会やキャンペーン等で獲得した見込み顧客を登録
3. 既存顧客データの移行・インポート時

**主要な処理内容**：
1. HTTPクライアントからPOST /api/Customersへリクエスト（JSON形式のボディ）
2. CustomersControllerがリクエストを受信
3. MediatRを通じてCreateCustomerCommandを発行
4. FluentValidationによる入力値バリデーション
5. CreateCustomerCommand.Handlerがコマンドを処理
6. Customerエンティティを作成してDbContextに追加
7. SaveChangesAsync()でデータベースに永続化
8. CustomerCreatedドメインイベントをPublish
9. HTTP 204 No Contentレスポンスを返却

**関連システム・外部連携**：MediatRのNotificationによるドメインイベント通知（CustomerCreated）。

**権限による制御**：[Authorize]属性により認証済みユーザーのみアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現時点で関連画面なし。API経由で直接呼び出し |

## 機能種別

CRUD操作（Create） / データ登録

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | string | Yes | 顧客ID | 5文字固定, NotEmpty |
| CompanyName | string | Yes | 会社名 | 最大40文字, NotEmpty |
| PostalCode | string | Yes | 郵便番号 | 最大10文字, NotEmpty |
| Address | string | No | 住所 | 最大60文字 |
| City | string | No | 市区町村 | 最大15文字 |
| ContactName | string | No | 担当者名 | 最大30文字 |
| ContactTitle | string | No | 担当者役職 | 最大30文字 |
| Country | string | No | 国名 | 最大15文字 |
| Fax | string | No | FAX番号 | 最大24文字 |
| Phone | string | No | 電話番号 | 最大24文字 |
| Region | string | No | 地域 | 最大15文字 |

### 入力データソース

HTTPリクエスト: POST /api/Customers（JSON形式のリクエストボディ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | レスポンスボディなし（HTTP 204） |

### 出力先

HTTPレスポンス（HTTP 204 No Content）

## 処理フロー

### 処理シーケンス

```
1. CustomersController.Create(command)
   └─ Mediator.Send(command)
       └─ FluentValidation: CreateCustomerCommandValidator
           └─ Id, CompanyName, PostalCode等のバリデーション
       └─ CreateCustomerCommand.Handler.Handle()
           ├─ new Customer { ... } でエンティティ作成
           ├─ _context.Customers.Add(entity)
           ├─ _context.SaveChangesAsync()
           └─ _mediator.Publish(new CustomerCreated { CustomerId = entity.CustomerId })
2. NoContent()でHTTP 204レスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[POST /api/Customers] --> B[CustomersController.Create]
    B --> C[Mediator.Send]
    C --> D[CreateCustomerCommandValidator]
    D --> E{バリデーション成功?}
    E -->|No| F[ValidationException]
    F --> G[HTTP 400 Bad Request]
    E -->|Yes| H[CreateCustomerCommand.Handler.Handle]
    H --> I[new Customer エンティティ作成]
    I --> J[_context.Customers.Add]
    J --> K[SaveChangesAsync]
    K --> L[Publish CustomerCreated Event]
    L --> M[NoContent 返却]
    M --> N[HTTP 204 No Content]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ID形式 | 顧客IDは5文字固定 | 常時 |
| BR-002 | 必須項目 | Id, CompanyName, PostalCodeは必須 | 常時 |
| BR-003 | 文字数制限 | 各フィールドに最大文字数制限あり | 常時 |
| BR-004 | 認証必須 | 認証済みユーザーのみ実行可能 | 常時 |
| BR-005 | イベント発行 | 登録成功時にCustomerCreatedイベントを発行 | 登録成功時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 顧客登録 | Customers | INSERT | 新規顧客レコード作成 |

### テーブル別操作詳細

#### Customers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | CustomerID | request.Id | 主キー、5文字固定 |
| INSERT | CompanyName | request.CompanyName | 必須、最大40文字 |
| INSERT | ContactName | request.ContactName | 任意、最大30文字 |
| INSERT | ContactTitle | request.ContactTitle | 任意、最大30文字 |
| INSERT | Address | request.Address | 任意、最大60文字 |
| INSERT | City | request.City | 任意、最大15文字 |
| INSERT | PostalCode | request.PostalCode | 必須、最大10文字 |
| INSERT | Country | request.Country | 任意、最大15文字 |
| INSERT | Phone | request.Phone | 任意、最大24文字 |
| INSERT | Fax | request.Fax | 任意、最大24文字 |

※Regionはコマンドにプロパティがあるがエンティティ作成時に設定されていない

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | バリデーションエラー（ID長、必須項目等） | 入力値を修正して再試行 |
| 401 | Unauthorized | 未認証アクセス | ログイン後に再試行 |
| 500 | InternalServerError | データベースエラー（重複IDなど） | IDを変更して再試行 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。主キー重複エラーの場合はIDを変更して再試行。

## トランザクション仕様

SaveChangesAsync()内でトランザクション管理される。エンティティの追加とドメインイベントの発行は同一コンテキスト内で実行される。

## パフォーマンス要件

- レスポンス時間：500ms以内
- 単一レコードのINSERT操作のため高速なレスポンスが期待できる

## セキュリティ考慮事項

- [Authorize]属性による認証チェック
- SQLインジェクション対策はEntity Frameworkが自動で実施
- FluentValidationによる入力値検証
- 顧客IDはユーザー入力を許可（セキュリティ上の考慮が必要）

## 備考

- MediatRによるCQRSパターンを採用
- FluentValidationによる入力検証
- ドメインイベント（CustomerCreated）による拡張性を確保

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コマンドとドメインエンティティの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CreateCustomerCommand.cs | `Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs` | コマンド定義と内部ハンドラ |
| 1-2 | Customer.cs | `Src/Domain/Entities/Customer.cs` | ドメインエンティティ |
| 1-3 | CustomerCreated.cs | `Src/Application/Customers/Commands/CreateCustomer/CustomerCreated.cs` | ドメインイベント |

**読解のコツ**:
- CreateCustomerCommand.csはコマンドとハンドラが同一ファイル内に定義されている
- **11-31行目**: コマンドのプロパティ定義
- **33-68行目**: 内部Handlerクラスの実装

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **34-42行目**: Create()メソッドでHTTP POSTリクエストを処理
2. **35-36行目**: ProducesResponseType属性でレスポンス型を定義（204）
3. **37行目**: [FromBody]でJSONリクエストボディをデシリアライズ
4. **39行目**: Mediator.Send()でCreateCustomerCommandを発行
5. **41行目**: NoContent()でHTTP 204レスポンス返却

#### Step 3: バリデーションを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CreateCustomerCommandValidator.cs | `Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommandValidator.cs` | FluentValidationによるバリデーション |

**主要処理フロー**:
- **7-22行目**: 各フィールドのバリデーションルール
- **9行目**: Id - 5文字固定、必須
- **12行目**: CompanyName - 最大40文字、必須
- **18行目**: PostalCode - 最大10文字、必須

#### Step 4: コマンドハンドラを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CreateCustomerCommand.cs | `Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs` | 内部Handlerクラス |

**主要処理フロー**:
- **44-67行目**: Handle()メソッドの実装
- **46-58行目**: Customerエンティティの生成
- **60行目**: DbSetへの追加
- **62行目**: SaveChangesAsync()で永続化
- **64行目**: CustomerCreatedイベントのPublish

### プログラム呼び出し階層図

```
HTTP Client
    │
    ▼
POST /api/Customers
    │
    └─ CustomersController.Create(command)
           │
           └─ Mediator.Send(CreateCustomerCommand)
                  │
                  ├─ CreateCustomerCommandValidator
                  │      └─ FluentValidation
                  │
                  └─ CreateCustomerCommand.Handler.Handle()
                         │
                         ├─ new Customer { ... }
                         │
                         ├─ INorthwindDbContext.Customers.Add()
                         │
                         ├─ SaveChangesAsync()
                         │      └─ INSERT INTO Customers ...
                         │
                         └─ Mediator.Publish(CustomerCreated)
                                └─ ドメインイベントハンドラ
```

### データフロー図

```
[入力]                    [処理]                              [出力]

JSON Request Body  ───▶  CreateCustomerCommandValidator    ───▶  ValidationException or OK
                   ───▶  CreateCustomerCommand.Handler     ───▶  Unit.Value
Customer Data      ───▶  new Customer                      ───▶  Entity
Entity             ───▶  DbContext.Add + SaveChangesAsync  ───▶  DB INSERT
CustomerId         ───▶  Mediator.Publish                  ───▶  CustomerCreated Event
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | ソース | APIコントローラー |
| CreateCustomerCommand.cs | `Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommand.cs` | ソース | コマンド定義・ハンドラ |
| CreateCustomerCommandValidator.cs | `Src/Application/Customers/Commands/CreateCustomer/CreateCustomerCommandValidator.cs` | ソース | バリデーター |
| CustomerCreated.cs | `Src/Application/Customers/Commands/CreateCustomer/CustomerCreated.cs` | ソース | ドメインイベント |
| Customer.cs | `Src/Domain/Entities/Customer.cs` | ソース | ドメインエンティティ |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DbContextインターフェース |
| CustomerConfiguration.cs | `Src/Persistence/Configurations/CustomerConfiguration.cs` | ソース | EF Coreエンティティ設定 |
