# 機能設計書 7-顧客更新

## 概要

本ドキュメントは、NorthwindTradersシステムにおける顧客更新機能の設計仕様を記載する。REST APIを通じて既存の顧客情報を更新する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：顧客の連絡先変更、住所移転、担当者変更等が発生した際に、システム内の顧客情報を最新の状態に更新する必要がある。正確な顧客情報の維持は、受注処理や請求書送付等の業務の正常な遂行に不可欠である。

**機能の利用シーン**：
1. 顧客から連絡先変更の連絡を受けた際の情報更新
2. 顧客の本社移転等による住所変更
3. 担当者異動に伴う担当者名・役職の変更
4. 定期的な顧客情報の棚卸し・クレンジング

**主要な処理内容**：
1. HTTPクライアントからPUT /api/Customers/{id}へリクエスト（JSON形式のボディ）
2. CustomersControllerがリクエストを受信
3. MediatRを通じてUpdateCustomerCommandを発行
4. FluentValidationによる入力値バリデーション（国別の追加ルールあり）
5. UpdateCustomerCommand.Handlerがコマンドを処理
6. 既存顧客を検索し、存在確認
7. エンティティのプロパティを更新
8. SaveChangesAsync()でデータベースに永続化
9. HTTP 204 No Contentレスポンスを返却

**関連システム・外部連携**：なし

**権限による制御**：[Authorize]属性により認証済みユーザーのみアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現時点で関連画面なし。API経由で直接呼び出し |

## 機能種別

CRUD操作（Update） / データ更新

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | string | Yes | 顧客ID | 最大5文字, NotEmpty |
| CompanyName | string | Yes | 会社名 | 最大40文字, NotEmpty |
| Fax | string | Yes | FAX番号 | 最大24文字, NotEmpty |
| Phone | string | Yes | 電話番号 | 最大24文字, NotEmpty |
| Address | string | No | 住所 | 最大60文字 |
| City | string | No | 市区町村 | 最大15文字 |
| ContactName | string | No | 担当者名 | 最大30文字 |
| ContactTitle | string | No | 担当者役職 | 最大30文字 |
| Country | string | No | 国名 | 最大15文字 |
| PostalCode | string | No | 郵便番号 | 最大10文字, オーストラリアは4桁 |
| Region | string | No | 地域 | 最大15文字 |

### 入力データソース

HTTPリクエスト: PUT /api/Customers/{id}（JSON形式のリクエストボディ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | レスポンスボディなし（HTTP 204） |

### 出力先

HTTPレスポンス（HTTP 204 No Content または HTTP 404 Not Found）

## 処理フロー

### 処理シーケンス

```
1. CustomersController.Update(command)
   └─ Mediator.Send(command)
       └─ FluentValidation: UpdateCustomerCommandValidator
           ├─ 基本バリデーション（ID, CompanyName等）
           ├─ オーストラリアの郵便番号は4桁
           └─ クイーンズランド州の場合は07から始まる電話番号必須
       └─ UpdateCustomerCommand.Handler.Handle()
           ├─ _context.Customers.SingleOrDefaultAsync(c => c.CustomerId == request.Id)
           ├─ entity == null の場合、NotFoundException throw
           ├─ エンティティのプロパティを更新
           └─ _context.SaveChangesAsync()
2. 成功時：NoContent()でHTTP 204レスポンス返却
3. 存在しない場合：CustomExceptionHandlerMiddlewareでHTTP 404変換
```

### フローチャート

```mermaid
flowchart TD
    A[PUT /api/Customers/id] --> B[CustomersController.Update]
    B --> C[Mediator.Send]
    C --> D[UpdateCustomerCommandValidator]
    D --> E{バリデーション成功?}
    E -->|No| F[ValidationException]
    F --> G[HTTP 400 Bad Request]
    E -->|Yes| H[UpdateCustomerCommand.Handler.Handle]
    H --> I[SingleOrDefaultAsync 検索]
    I --> J{顧客存在?}
    J -->|No| K[NotFoundException]
    K --> L[HTTP 404 Not Found]
    J -->|Yes| M[エンティティ更新]
    M --> N[SaveChangesAsync]
    N --> O[NoContent 返却]
    O --> P[HTTP 204 No Content]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ID形式 | 顧客IDは最大5文字 | 常時 |
| BR-002 | 必須項目 | Id, CompanyName, Fax, Phoneは必須 | 常時 |
| BR-003 | 存在確認 | 指定IDの顧客が存在しない場合は404エラー | 常時 |
| BR-004 | 認証必須 | 認証済みユーザーのみ実行可能 | 常時 |
| BR-005 | オーストラリア郵便番号 | オーストラリアの場合、郵便番号は4桁 | Country="Australia" |
| BR-006 | クイーンズランド電話番号 | オーストラリアでPostalCodeが4で始まる場合、PhoneまたはFaxは07で始まる必要あり | Country="Australia" && PostalCode.StartsWith("4") |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 顧客検索 | Customers | SELECT | 主キーによる1件検索 |
| 顧客更新 | Customers | UPDATE | 既存レコードの更新 |

### テーブル別操作詳細

#### Customers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | * | WHERE CustomerID = @id | 更新対象の検索 |
| UPDATE | CompanyName | request.CompanyName | 会社名 |
| UPDATE | ContactName | request.ContactName | 担当者名 |
| UPDATE | ContactTitle | request.ContactTitle | 担当者役職 |
| UPDATE | Address | request.Address | 住所 |
| UPDATE | City | request.City | 市区町村 |
| UPDATE | PostalCode | request.PostalCode | 郵便番号 |
| UPDATE | Country | request.Country | 国名 |
| UPDATE | Phone | request.Phone | 電話番号 |
| UPDATE | Fax | request.Fax | FAX番号 |

※Regionは更新対象に含まれていない

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | バリデーションエラー | 入力値を修正して再試行 |
| 401 | Unauthorized | 未認証アクセス | ログイン後に再試行 |
| 404 | NotFound | 指定IDの顧客が存在しない | IDを確認して再試行 |
| 500 | InternalServerError | データベースエラー | システム管理者へ連絡 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。

## トランザクション仕様

SaveChangesAsync()内でトランザクション管理される。更新処理は単一のアトミック操作として実行される。

## パフォーマンス要件

- レスポンス時間：500ms以内
- 主キー検索+単一レコードのUPDATE操作のため高速なレスポンスが期待できる

## セキュリティ考慮事項

- [Authorize]属性による認証チェック
- SQLインジェクション対策はEntity Frameworkが自動で実施
- FluentValidationによる入力値検証
- 楽観的ロックは実装されていない（後勝ち更新）

## 備考

- MediatRによるCQRSパターンを採用
- FluentValidationによる入力検証（国別の追加ルールあり）
- Regionプロパティは更新対象外

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コマンドとドメインエンティティの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | UpdateCustomerCommand.cs | `Src/Application/Customers/Commands/UpdateCustomer/UpdateCustomerCommand.cs` | コマンド定義と内部ハンドラ |
| 1-2 | Customer.cs | `Src/Domain/Entities/Customer.cs` | ドメインエンティティ |

**読解のコツ**:
- UpdateCustomerCommand.csはコマンドとハンドラが同一ファイル内に定義されている
- **11-23行目**: コマンドのプロパティ定義
- **25-59行目**: 内部Handlerクラスの実装

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **44-52行目**: Update()メソッドでHTTP PUTリクエストを処理
2. **45-46行目**: ProducesResponseType属性でレスポンス型を定義（204, 404）
3. **47行目**: [FromBody]でJSONリクエストボディをデシリアライズ
4. **49行目**: Mediator.Send()でUpdateCustomerCommandを発行
5. **51行目**: NoContent()でHTTP 204レスポンス返却

#### Step 3: バリデーションを理解する（重要）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UpdateCustomerCommandValidator.cs | `Src/Application/Customers/Commands/UpdateCustomer/UpdateCustomerCommandValidator.cs` | FluentValidationによるバリデーション |

**主要処理フロー**:
- **8-21行目**: 基本的なバリデーションルール
- **22-24行目**: オーストラリアの郵便番号は4桁ルール
- **26-29行目**: クイーンズランド州の電話番号ルール
- **32-35行目**: HaveQueenslandLandLineカスタムバリデーション

#### Step 4: コマンドハンドラを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | UpdateCustomerCommand.cs | `Src/Application/Customers/Commands/UpdateCustomer/UpdateCustomerCommand.cs` | 内部Handlerクラス |

**主要処理フロー**:
- **34-57行目**: Handle()メソッドの実装
- **36-37行目**: SingleOrDefaultAsync()で顧客検索
- **39-42行目**: 存在チェックとNotFoundException
- **44-52行目**: エンティティのプロパティ更新
- **54行目**: SaveChangesAsync()で永続化

### プログラム呼び出し階層図

```
HTTP Client
    │
    ▼
PUT /api/Customers/{id}
    │
    └─ CustomersController.Update(command)
           │
           └─ Mediator.Send(UpdateCustomerCommand)
                  │
                  ├─ UpdateCustomerCommandValidator
                  │      ├─ 基本バリデーション
                  │      └─ 国別追加バリデーション
                  │
                  └─ UpdateCustomerCommand.Handler.Handle()
                         │
                         ├─ INorthwindDbContext.Customers.SingleOrDefaultAsync()
                         │      └─ SELECT * FROM Customers WHERE CustomerID = @id
                         │
                         ├─ NotFoundException（存在しない場合）
                         │
                         ├─ entity.Property = request.Property（更新）
                         │
                         └─ SaveChangesAsync()
                                └─ UPDATE Customers SET ...
```

### データフロー図

```
[入力]                    [処理]                              [出力]

JSON Request Body  ───▶  UpdateCustomerCommandValidator    ───▶  ValidationException or OK
id (パスパラメータ)───▶  UpdateCustomerCommand.Handler     ───▶  NotFoundException or Unit.Value
Customers Table    ───▶  SingleOrDefaultAsync              ───▶  Customer Entity
Updated Entity     ───▶  SaveChangesAsync                  ───▶  DB UPDATE
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | ソース | APIコントローラー |
| UpdateCustomerCommand.cs | `Src/Application/Customers/Commands/UpdateCustomer/UpdateCustomerCommand.cs` | ソース | コマンド定義・ハンドラ |
| UpdateCustomerCommandValidator.cs | `Src/Application/Customers/Commands/UpdateCustomer/UpdateCustomerCommandValidator.cs` | ソース | バリデーター |
| Customer.cs | `Src/Domain/Entities/Customer.cs` | ソース | ドメインエンティティ |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
