# 機能設計書 8-顧客削除

## 概要

本ドキュメントは、NorthwindTradersシステムにおける顧客削除機能の設計仕様を記載する。REST APIを通じて指定した顧客IDに基づいて顧客を削除する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：取引終了した顧客や、誤って登録した顧客情報をシステムから削除する必要がある。データ整合性を保つため、関連する受注データが存在する場合は削除を禁止し、ビジネスルールに基づいた安全な削除処理を実現する。

**機能の利用シーン**：
1. 取引終了した顧客データの削除
2. 誤登録した顧客データの修正（削除後に再登録）
3. テストデータのクリーンアップ

**主要な処理内容**：
1. HTTPクライアントからDELETE /api/Customers/{id}へリクエスト
2. CustomersControllerがリクエストを受信
3. MediatRを通じてDeleteCustomerCommandを発行
4. FluentValidationによる入力値バリデーション
5. DeleteCustomerCommandHandlerがコマンドを処理
6. 顧客の存在確認
7. 関連する受注データの存在確認
8. 受注データが存在する場合はDeleteFailureExceptionをスロー
9. 受注データがなければ顧客を削除
10. SaveChangesAsync()でデータベースに永続化
11. HTTP 204 No Contentレスポンスを返却

**関連システム・外部連携**：なし

**権限による制御**：[Authorize]属性により認証済みユーザーのみアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 現時点で関連画面なし。API経由で直接呼び出し |

## 機能種別

CRUD操作（Delete） / データ削除

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | string | Yes | 顧客ID | NotEmpty, 5文字固定 |

### 入力データソース

HTTPリクエスト: DELETE /api/Customers/{id}

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | レスポンスボディなし（HTTP 204） |

### 出力先

HTTPレスポンス（HTTP 204 No Content または HTTP 404 Not Found または HTTP 400 Bad Request）

## 処理フロー

### 処理シーケンス

```
1. CustomersController.Delete(id)
   └─ Mediator.Send(new DeleteCustomerCommand { Id = id })
       └─ FluentValidation: DeleteCustomerCommandValidator
           └─ Id: NotEmpty, Length(5)
       └─ DeleteCustomerCommandHandler.Handle()
           ├─ _context.Customers.FindAsync(request.Id)
           ├─ entity == null の場合、NotFoundException throw
           ├─ _context.Orders.Any(o => o.CustomerId == entity.CustomerId)
           ├─ hasOrders == true の場合、DeleteFailureException throw
           ├─ _context.Customers.Remove(entity)
           └─ _context.SaveChangesAsync()
2. 成功時：NoContent()でHTTP 204レスポンス返却
3. 存在しない場合：CustomExceptionHandlerMiddlewareでHTTP 404変換
4. 受注あり場合：CustomExceptionHandlerMiddlewareでHTTP 400変換
```

### フローチャート

```mermaid
flowchart TD
    A[DELETE /api/Customers/id] --> B[CustomersController.Delete]
    B --> C[Mediator.Send]
    C --> D[DeleteCustomerCommandValidator]
    D --> E{バリデーション成功?}
    E -->|No| F[ValidationException]
    F --> G[HTTP 400 Bad Request]
    E -->|Yes| H[DeleteCustomerCommandHandler.Handle]
    H --> I[FindAsync 検索]
    I --> J{顧客存在?}
    J -->|No| K[NotFoundException]
    K --> L[HTTP 404 Not Found]
    J -->|Yes| M{関連受注あり?}
    M -->|Yes| N[DeleteFailureException]
    N --> O[HTTP 400 Bad Request]
    M -->|No| P[_context.Customers.Remove]
    P --> Q[SaveChangesAsync]
    Q --> R[NoContent 返却]
    R --> S[HTTP 204 No Content]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ID形式検証 | 顧客IDは5文字固定 | 常時 |
| BR-002 | 存在確認 | 指定IDの顧客が存在しない場合は404エラー | 常時 |
| BR-003 | 受注データチェック | 関連する受注データが存在する場合は削除不可 | 常時 |
| BR-004 | 認証必須 | 認証済みユーザーのみ実行可能 | 常時 |
| BR-005 | 物理削除 | 削除は物理削除（論理削除ではない） | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 顧客検索 | Customers | SELECT | 主キーによる1件検索 |
| 受注存在確認 | Orders | SELECT | 顧客IDによる存在チェック |
| 顧客削除 | Customers | DELETE | レコード削除 |

### テーブル別操作詳細

#### Customers

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | * | WHERE CustomerID = @id | 削除対象の検索 |
| DELETE | - | WHERE CustomerID = @id | 物理削除 |

#### Orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | COUNT/ANY | WHERE CustomerID = @customerId | 関連受注の存在チェック |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | バリデーションエラー（ID長など） | 正しいID形式で再試行 |
| 400 | BadRequest | 関連する受注データが存在 | 先に受注データを削除または移行 |
| 401 | Unauthorized | 未認証アクセス | ログイン後に再試行 |
| 404 | NotFound | 指定IDの顧客が存在しない | IDを確認して再試行 |
| 500 | InternalServerError | データベースエラー | システム管理者へ連絡 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。

## トランザクション仕様

SaveChangesAsync()内でトランザクション管理される。削除処理は単一のアトミック操作として実行される。

## パフォーマンス要件

- レスポンス時間：500ms以内
- 主キー検索+外部キー存在チェック+単一レコードのDELETE操作

## セキュリティ考慮事項

- [Authorize]属性による認証チェック
- SQLインジェクション対策はEntity Frameworkが自動で実施
- FluentValidationによる入力値検証
- 参照整合性チェックによるデータ整合性の保護

## 備考

- MediatRによるCQRSパターンを採用
- FluentValidationによる入力検証
- 参照整合性違反時は明示的なビジネスエラーとして処理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コマンドと例外クラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | DeleteCustomerCommand.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommand.cs` | コマンド定義（Idのみ） |
| 1-2 | DeleteFailureException.cs | `Src/Application/Common/Exceptions/DeleteFailureException.cs` | 削除失敗時の例外クラス |

**読解のコツ**:
- DeleteCustomerCommandはIdプロパティのみを持つシンプルなコマンド
- DeleteFailureExceptionはエンティティ名、キー、メッセージを含む

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **54-62行目**: Delete()メソッドでHTTP DELETEリクエストを処理
2. **55-56行目**: ProducesResponseType属性でレスポンス型を定義（204, 404）
3. **59行目**: Mediator.Send()でDeleteCustomerCommandを発行
4. **61行目**: NoContent()でHTTP 204レスポンス返却

#### Step 3: バリデーションを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DeleteCustomerCommandValidator.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommandValidator.cs` | FluentValidationによるバリデーション |

**主要処理フロー**:
- **7-10行目**: RuleFor(v => v.Id).NotEmpty().Length(5) - IDの必須・長さチェック

#### Step 4: コマンドハンドラを理解する（重要）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | DeleteCustomerCommandHandler.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommandHandler.cs` | コマンドハンドラ実装 |

**主要処理フロー**:
- **20-41行目**: Handle()メソッドの実装
- **22-23行目**: FindAsync()で顧客検索
- **25-28行目**: 存在チェックとNotFoundException
- **30-34行目**: 関連受注の存在チェックとDeleteFailureException
- **36行目**: Remove()でエンティティ削除マーク
- **38行目**: SaveChangesAsync()で永続化

### プログラム呼び出し階層図

```
HTTP Client
    │
    ▼
DELETE /api/Customers/{id}
    │
    └─ CustomersController.Delete(id)
           │
           └─ Mediator.Send(DeleteCustomerCommand)
                  │
                  ├─ DeleteCustomerCommandValidator
                  │      └─ FluentValidation (Id: NotEmpty, Length(5))
                  │
                  └─ DeleteCustomerCommandHandler.Handle()
                         │
                         ├─ INorthwindDbContext.Customers.FindAsync()
                         │      └─ SELECT * FROM Customers WHERE CustomerID = @id
                         │
                         ├─ NotFoundException（存在しない場合）
                         │
                         ├─ INorthwindDbContext.Orders.Any()
                         │      └─ SELECT EXISTS(...) FROM Orders WHERE CustomerID = @id
                         │
                         ├─ DeleteFailureException（受注あり場合）
                         │
                         ├─ INorthwindDbContext.Customers.Remove()
                         │
                         └─ SaveChangesAsync()
                                └─ DELETE FROM Customers WHERE CustomerID = @id
```

### データフロー図

```
[入力]                    [処理]                              [出力]

id (パスパラメータ) ───▶  DeleteCustomerCommandValidator    ───▶  ValidationException or OK
                    ───▶  DeleteCustomerCommandHandler      ───▶  NotFoundException or DeleteFailureException or Unit.Value
Customers Table     ───▶  FindAsync                         ───▶  Customer Entity or null
Orders Table        ───▶  Any                               ───▶  bool (hasOrders)
Customer Entity     ───▶  Remove + SaveChangesAsync         ───▶  DB DELETE
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CustomersController.cs | `Src/WebUI/Controllers/CustomersController.cs` | ソース | APIコントローラー |
| DeleteCustomerCommand.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommand.cs` | ソース | コマンド定義 |
| DeleteCustomerCommandHandler.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommandHandler.cs` | ソース | コマンドハンドラ |
| DeleteCustomerCommandValidator.cs | `Src/Application/Customers/Commands/DeleteCustomer/DeleteCustomerCommandValidator.cs` | ソース | バリデーター |
| Customer.cs | `Src/Domain/Entities/Customer.cs` | ソース | ドメインエンティティ |
| NotFoundException.cs | `Src/Application/Common/Exceptions/NotFoundException.cs` | ソース | 例外クラス |
| DeleteFailureException.cs | `Src/Application/Common/Exceptions/DeleteFailureException.cs` | ソース | 削除失敗例外クラス |
