# 機能設計書 9-商品一覧取得

## 概要

本ドキュメントは、NorthwindTradersシステムにおける商品一覧取得機能の設計仕様を記載する。REST APIを通じて全ての商品情報を一覧形式で取得する機能の詳細を定義する。

### 本機能の処理概要

**業務上の目的・背景**：商品管理業務において、登録されている全商品を一覧で確認する必要がある。在庫確認、価格リストの作成、商品カタログの更新等の業務で利用される。認証不要で公開されており、一般ユーザーも商品情報を閲覧できる。

**機能の利用シーン**：
1. 商品一覧画面の初期表示時に全商品データを取得
2. 商品検索・フィルタリングのベースデータ取得
3. 受注入力時の商品選択リストの取得
4. 外部システムとの商品データ連携

**主要な処理内容**：
1. HTTPクライアントからGET /api/Productsへリクエスト
2. ProductsControllerがリクエストを受信
3. MediatRを通じてGetProductsListQueryを発行
4. GetProductsListQueryHandlerがクエリを処理
5. INorthwindDbContextを通じてProductsテーブルからデータ取得
6. AutoMapperでProductエンティティをProductDtoにマッピング
7. 商品名でソートして返却
8. ProductsListVmとしてレスポンス返却

**関連システム・外部連携**：Angular SPAクライアントのProductsClientからの呼び出し。

**権限による制御**：[AllowAnonymous]属性により認証不要でアクセス可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品一覧画面 | 主画面 | 画面初期表示時にProductsClient.getAll()を呼び出し、全商品の一覧を取得・表示する |

## 機能種別

CRUD操作（Read） / データ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | パラメータなし | - |

### 入力データソース

HTTPリクエスト: GET /api/Products

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Products | IList<ProductDto> | 商品情報のリスト |
| Products[].ProductId | int | 商品ID |
| Products[].ProductName | string | 商品名 |
| Products[].UnitPrice | decimal? | 単価 |
| Products[].SupplierId | int? | 仕入先ID |
| Products[].SupplierCompanyName | string | 仕入先会社名 |
| Products[].CategoryId | int? | カテゴリID |
| Products[].CategoryName | string | カテゴリ名 |
| Products[].Discontinued | bool | 販売中止フラグ |
| CreateEnabled | bool | 作成権限フラグ |

### 出力先

HTTPレスポンス（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. ProductsController.GetAll()
   └─ Mediator.Send(new GetProductsListQuery())
       └─ GetProductsListQueryHandler.Handle()
           ├─ _context.Products（DbSet<Product>）からデータ取得
           ├─ ProjectTo<ProductDto>()でマッピング（Supplier, Category含む）
           ├─ OrderBy(p => p.ProductName)でソート
           ├─ ToListAsync()で非同期実行
           └─ ProductsListVmにラップして返却（CreateEnabled = true）
2. Ok(vm)でHTTP 200レスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/Products] --> B[ProductsController.GetAll]
    B --> C[Mediator.Send]
    C --> D[GetProductsListQueryHandler.Handle]
    D --> E[_context.Products]
    E --> F[ProjectTo ProductDto]
    F --> G[OrderBy ProductName]
    G --> H[ToListAsync]
    H --> I[ProductsListVm作成]
    I --> J[CreateEnabled = true]
    J --> K[Ok vm 返却]
    K --> L[HTTP 200 JSON Response]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 全件取得 | 登録されている全ての商品を取得 | 常時 |
| BR-002 | 認証不要 | 認証なしでアクセス可能 | 常時 |
| BR-003 | ソート順 | 商品名（ProductName）の昇順でソート | 常時 |
| BR-004 | 関連データ取得 | 仕入先名、カテゴリ名も同時に取得 | 常時 |
| BR-005 | CreateEnabled | 現在は常にtrue（TODO: ユーザー権限で設定予定） | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品一覧取得 | Products | SELECT | 全商品レコード取得 |
| 仕入先情報取得 | Suppliers | LEFT JOIN | 仕入先名の取得 |
| カテゴリ情報取得 | Categories | LEFT JOIN | カテゴリ名の取得 |

### テーブル別操作詳細

#### Products (+ Suppliers, Categories JOIN)

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | ProductID | 全件 | ProductIdにマッピング |
| SELECT | ProductName | 全件 | ソートキーとしても使用 |
| SELECT | UnitPrice | 全件 | |
| SELECT | SupplierID | 全件 | |
| SELECT | Supplier.CompanyName | LEFT JOIN | SupplierCompanyNameにマッピング |
| SELECT | CategoryID | 全件 | |
| SELECT | Category.CategoryName | LEFT JOIN | CategoryNameにマッピング |
| SELECT | Discontinued | 全件 | |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 500 | InternalServerError | データベースエラー等 | システム管理者へ連絡 |

### リトライ仕様

データベース接続エラー等の一時的な障害に対しては、クライアント側でのリトライを推奨。

## トランザクション仕様

読み取り専用クエリのため、明示的なトランザクション管理は不要。Entity Frameworkのデフォルトの分離レベルが適用される。

## パフォーマンス要件

- レスポンス時間：1秒以内
- ProjectTo<>()によるクエリ最適化（必要なカラムのみSELECT）
- 関連テーブルはLEFT JOINで効率的に取得

## セキュリティ考慮事項

- [AllowAnonymous]属性により認証不要でアクセス可能
- 商品情報は公開情報のため、認証なしでのアクセスを許可
- SQLインジェクション対策はEntity Frameworkが自動で実施

## 備考

- MediatRによるCQRSパターンを採用
- AutoMapperによるオブジェクトマッピング
- CreateEnabledは現在ハードコードされているが、将来的にユーザー権限に基づいて設定予定（TODOコメント）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

DTOとViewModelの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ProductDto.cs | `Src/Application/Products/Queries/GetProductsList/ProductDto.cs` | 商品一覧用DTO。関連エンティティのプロパティ含む |
| 1-2 | ProductsListVm.cs | `Src/Application/Products/Queries/GetProductsList/ProductsListVm.cs` | DTOリストとCreateEnabledフラグ |
| 1-3 | Product.cs | `Src/Domain/Entities/Product.cs` | ドメインエンティティ |

**読解のコツ**:
- ProductDto.csの**25-30行目**にAutoMapperのマッピング設定がある
- Supplier.CompanyName、Category.CategoryNameのマッピングを確認

#### Step 2: エントリーポイントを理解する

APIコントローラーがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | RESTful APIコントローラー |

**主要処理フロー**:
1. **17-24行目**: GetAll()メソッドでHTTP GETリクエストを処理
2. **18行目**: [AllowAnonymous]属性で認証不要を設定
3. **21行目**: Mediator.Send()でGetProductsListQueryを発行
4. **23行目**: Ok(vm)でHTTP 200レスポンス返却

#### Step 3: クエリハンドラを理解する

MediatRのRequestHandlerがビジネスロジックを実行する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GetProductsListQuery.cs | `Src/Application/Products/Queries/GetProductsList/GetProductsListQuery.cs` | クエリ定義（パラメータなし） |
| 3-2 | GetProductsListQueryHandler.cs | `Src/Application/Products/Queries/GetProductsList/GetProductsListQueryHandler.cs` | クエリハンドラ実装 |

**主要処理フロー**:
- **23-37行目**: Handle()メソッドの実装
- **25-28行目**: INorthwindDbContext.ProductsからProjectTo<>()でマッピングしながら取得
- **27行目**: OrderBy(p => p.ProductName)でソート
- **30-34行目**: ProductsListVmにラップして返却
- **33行目**: CreateEnabled = true（TODOコメントあり）

### プログラム呼び出し階層図

```
Angular SPA (ProductsClient.getAll())
    │
    ▼
GET /api/Products
    │
    └─ ProductsController.GetAll()
           │
           └─ Mediator.Send(GetProductsListQuery)
                  │
                  └─ GetProductsListQueryHandler.Handle()
                         │
                         ├─ INorthwindDbContext.Products
                         │      └─ DbSet<Product>
                         │
                         ├─ ProjectTo<ProductDto>()
                         │      └─ AutoMapper マッピング
                         │            ├─ Supplier.CompanyName → SupplierCompanyName
                         │            └─ Category.CategoryName → CategoryName
                         │
                         ├─ OrderBy(ProductName)
                         │
                         └─ ProductsListVm
                                └─ CreateEnabled = true
```

### データフロー図

```
[入力]                    [処理]                              [出力]

HTTP GET Request  ───▶  ProductsController               ───▶  HTTP 200 Response
                  ───▶  GetProductsListQueryHandler      ───▶  ProductsListVm
Products Table    ───▶  ProjectTo<ProductDto>            ───▶  JSON
Suppliers Table   ───▶  LEFT JOIN                        ───▶  SupplierCompanyName
Categories Table  ───▶  LEFT JOIN                        ───▶  CategoryName
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ProductsController.cs | `Src/WebUI/Controllers/ProductsController.cs` | ソース | APIコントローラー |
| GetProductsListQuery.cs | `Src/Application/Products/Queries/GetProductsList/GetProductsListQuery.cs` | ソース | クエリ定義 |
| GetProductsListQueryHandler.cs | `Src/Application/Products/Queries/GetProductsList/GetProductsListQueryHandler.cs` | ソース | クエリハンドラ |
| ProductDto.cs | `Src/Application/Products/Queries/GetProductsList/ProductDto.cs` | ソース | DTO定義 |
| ProductsListVm.cs | `Src/Application/Products/Queries/GetProductsList/ProductsListVm.cs` | ソース | ViewModel定義 |
| Product.cs | `Src/Domain/Entities/Product.cs` | ソース | ドメインエンティティ |
| INorthwindDbContext.cs | `Src/Application/Common/Interfaces/INorthwindDbContext.cs` | ソース | DbContextインターフェース |
| ProductConfiguration.cs | `Src/Persistence/Configurations/ProductConfiguration.cs` | ソース | EF Coreエンティティ設定 |
