# 通知設計書 2-ユーザー登録確認メール

## 概要

本ドキュメントは、NorthwindTradersシステムにおけるユーザー登録確認メール機能の設計仕様を定義する。新規ユーザーがアカウント登録を完了した際に、メールアドレスの所有権を確認するための確認リンク付きメールを送信する。

### 本通知の処理概要

ユーザー登録確認メールは、新規ユーザーがRegister画面でアカウント登録を完了した直後に、ASP.NET Core IdentityのIEmailSenderインターフェースを通じて送信されるメール通知である。

**業務上の目的・背景**：メールアドレスの所有権確認は、セキュリティ上重要なプロセスである。不正なメールアドレスでの登録を防止し、ユーザーが正当なアドレスの所有者であることを検証することで、なりすましや不正アカウント作成を防ぐ。また、パスワードリセットや重要な通知の送信先として有効なメールアドレスを確保する役割も果たす。

**通知の送信タイミング**：RegisterModel.OnPostAsyncメソッドでUserManagerによるユーザー作成（CreateAsync）が成功した直後に送信される。具体的には、UserManager.GenerateEmailConfirmationTokenAsyncでトークンを生成し、確認用URLを構築した後、IEmailSender.SendEmailAsyncが呼び出される。

**通知の受信者**：登録フォームで入力されたメールアドレス（Input.Email）が受信者となる。これは同時にユーザーのユーザー名としても使用される。

**通知内容の概要**：メールには確認用のリンクが含まれる。リンクをクリックすると/Account/ConfirmEmailページに遷移し、メールアドレスの確認が完了する。

**期待されるアクション**：受信者はメール内の確認リンクをクリックし、メールアドレスの所有権を証明する。確認が完了すると、アカウントが有効化され、システムへのログインが可能となる（RequireConfirmedAccountが有効な場合）。

## 通知種別

メール（Email）

ASP.NET Core IdentityのIEmailSenderインターフェースを使用した標準的なメール送信機能。

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（async/await） |
| 優先度 | 高（ユーザー登録フローのクリティカルパス） |
| リトライ | IEmailSender実装依存 |

### 送信先決定ロジック

送信先はユーザー登録フォームで入力されたメールアドレス（Input.Email）に固定される。
- RegisterModel.csの行92でInput.Emailが直接SendEmailAsyncに渡される
- 追加の送信先決定ロジックは存在しない

## 通知テンプレート

### メール通知の場合

| 項目 | 内容 |
|-----|------|
| 送信元アドレス | IEmailSender実装依存 |
| 送信元名称 | IEmailSender実装依存 |
| 件名 | Confirm your email |
| 形式 | HTML |

### 本文テンプレート

```html
Please confirm your account by <a href='{callbackUrl}'>clicking here</a>.
```

**注記**: callbackUrlはHtmlEncoder.Default.Encodeでエンコードされ、XSS対策が施されている。

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | 添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| callbackUrl | メール確認用URL | Url.Page + userId + code | Yes |
| userId | ユーザーID | ApplicationUser.Id | Yes |
| code | 確認トークン（Base64Urlエンコード済） | UserManager.GenerateEmailConfirmationTokenAsync | Yes |

### callbackUrl構築詳細

```csharp
var callbackUrl = Url.Page(
    "/Account/ConfirmEmail",
    pageHandler: null,
    values: new { area = "Identity", userId = user.Id, code = code },
    protocol: Request.Scheme);
```

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作 | ユーザー登録フォーム送信 | UserManager.CreateAsync成功時 | Register.cshtml.csのOnPostAsync |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ユーザー作成失敗 | UserManager.CreateAsyncがSucceeded=falseの場合、メール送信は実行されない |
| ModelState無効 | フォームバリデーションエラー時はメール送信に到達しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Register画面: フォーム送信] --> B[ModelState検証]
    B -->|無効| C[フォーム再表示]
    B -->|有効| D[ApplicationUser作成]
    D --> E[UserManager.CreateAsync]
    E --> F{作成成功?}
    F -->|No| G[エラー表示]
    F -->|Yes| H[確認トークン生成]
    H --> I[callbackUrl構築]
    I --> J[IEmailSender.SendEmailAsync]
    J --> K{RequireConfirmedAccount?}
    K -->|Yes| L[RegisterConfirmation画面へ]
    K -->|No| M[自動サインイン]
    L --> N[終了]
    M --> N
    C --> N
    G --> N
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | メール送信処理自体ではDB参照なし（トークン生成はIdentity内部処理） |

### テーブル別参照項目詳細

メール送信処理では直接のデータベース参照は行わない。UserManagerがIdentity内部でトークン管理を行う。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| AspNetUsers | INSERT | 新規ユーザーレコード作成 |

#### AspNetUsersテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| INSERT | Id | GUID生成 | 自動生成 |
| INSERT | UserName | Input.Email | メールアドレス |
| INSERT | Email | Input.Email | メールアドレス |
| INSERT | PasswordHash | ハッシュ化されたパスワード | 自動生成 |
| INSERT | EmailConfirmed | false | 未確認状態 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ユーザー作成失敗 | パスワード要件不足、重複ユーザー等 | result.Errorsをフォームに表示 |
| メール送信失敗 | IEmailSender実装の送信エラー | 例外がスローされる（ハンドリングなし） |
| トークン生成失敗 | Identity内部エラー | 例外がスローされる |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | IEmailSender実装依存 |
| リトライ間隔 | IEmailSender実装依存 |
| リトライ対象エラー | IEmailSender実装依存 |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | IEmailSender実装依存 |
| 1日あたり上限 | IEmailSender実装依存 |

### 配信時間帯

制限なし（24時間送信可能）

## セキュリティ考慮事項

- **トークンのエンコード**: EmailConfirmationTokenはWebEncoders.Base64UrlEncodeでエンコードされ、URL安全な形式で送信される
- **URLエンコード**: callbackUrlはHtmlEncoder.Default.Encodeで処理され、XSS攻撃を防止
- **トークン有効期限**: ASP.NET Core Identityのデフォルト設定に依存（通常24時間）
- **ワンタイム使用**: 確認トークンは一度使用すると無効化される
- **HTTPS強制**: Request.Schemeを使用するため、HTTPSで運用時は確認URLもHTTPSとなる

## 備考

- IEmailSenderの具体的な実装はDIコンテナで提供される必要がある
- 開発環境ではコンソール出力やモック実装が使用される場合がある
- RequireConfirmedAccountがtrueの場合、メール確認完了までログインが制限される
- 確認トークンの再送信はLogin画面のOnPostSendVerificationEmailAsyncで対応（別通知として定義）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ユーザー登録で使用されるデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | IdentityUserを継承したユーザーエンティティ |
| 1-2 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | InputModelクラス（行47-64）でフォーム入力構造を理解 |

**読解のコツ**: ASP.NET Core IdentityのIdentityUserは多くのプロパティを継承で持つ。ApplicationUserは拡張用のクラスだが、現在は空の継承のみ。

#### Step 2: エントリーポイントを理解する

ユーザー登録処理の開始点を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | OnPostAsyncメソッド（行72-113）が処理開始点 |

**主要処理フロー**:
1. **行74**: returnUrlの初期化
2. **行76**: ModelState検証
3. **行78**: ApplicationUserインスタンス作成
4. **行79**: UserManager.CreateAsyncでユーザー作成
5. **行84**: GenerateEmailConfirmationTokenAsyncでトークン生成
6. **行85**: Base64Urlエンコード
7. **行86-90**: callbackUrl構築
8. **行92-93**: SendEmailAsync実行

#### Step 3: トークン生成とURL構築を理解する

確認トークンの生成とURLの構築プロセスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | 行84-90でトークン生成からURL構築までの流れ |

**主要処理フロー**:
- **行84**: `_userManager.GenerateEmailConfirmationTokenAsync(user)` - トークン生成
- **行85**: `WebEncoders.Base64UrlEncode` - URLセーフなエンコード
- **行86-90**: `Url.Page` - 確認ページURLの構築

#### Step 4: メール送信を理解する

IEmailSenderを通じたメール送信の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | 行92-93でSendEmailAsync呼び出し |

**主要処理フロー**:
- **行92-93**: `_emailSender.SendEmailAsync(Input.Email, "Confirm your email", ...)` - メール送信

#### Step 5: 依存性注入を理解する

RegisterModelのコンストラクタで注入される依存関係を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | コンストラクタ（行28-38）で注入されるサービス |

### プログラム呼び出し階層図

```
RegisterModel.OnPostAsync (HTTP POST)
    │
    ├─ ModelState.IsValid チェック
    │
    ├─ new ApplicationUser { ... }
    │
    ├─ UserManager<ApplicationUser>.CreateAsync(user, password)
    │      │
    │      └─ AspNetUsers テーブルへINSERT
    │
    ├─ UserManager<ApplicationUser>.GenerateEmailConfirmationTokenAsync(user)
    │
    ├─ WebEncoders.Base64UrlEncode(token)
    │
    ├─ Url.Page("/Account/ConfirmEmail", ...)
    │
    └─ IEmailSender.SendEmailAsync(email, subject, htmlMessage)
           │
           └─ 実装依存（外部メールサービス等）
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Register Form
  │
  ├─ Email ─────────────▶ ApplicationUser作成
  │                              │
  ├─ Password ──────────▶ PasswordHash生成
  │                              │
  └─ ConfirmPassword ──▶ 検証のみ
                                 │
                                 ▼
                        UserManager.CreateAsync
                                 │
                                 ▼
                        TokenGeneration ─────────▶ 確認トークン
                                                       │
                                                       ▼
                                               URL構築（エンコード）
                                                       │
                                                       ▼
                                               IEmailSender ───▶ 確認メール送信
                                                                      │
                                                                      ▼
                                                                 ユーザーの
                                                                 メールボックス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Register.cshtml.cs | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml.cs` | ソース | ユーザー登録ページのコードビハインド |
| Register.cshtml | `Src/WebUI/Areas/Identity/Pages/Account/Register.cshtml` | テンプレート | ユーザー登録フォームのRazor View |
| ApplicationUser.cs | `Src/Infrastructure/Identity/ApplicationUser.cs` | ソース | ユーザーエンティティ定義 |
| DependencyInjection.cs | `Src/Infrastructure/DependencyInjection.cs` | ソース | DIコンテナ設定（Identity設定含む） |
