# アーキテクチャ構成図

## システム全体構成図

```mermaid
graph TB
    subgraph "User Interface"
        CLI["CLI<br/>(cli.zig)"]
    end

    subgraph "JavaScript Runtime Layer"
        VM["VirtualMachine<br/>(VirtualMachine.zig)"]
        EL["EventLoop<br/>(event_loop.zig)"]
        JSC["JavaScriptCore<br/>(WebKit)"]
        GO["JSGlobalObject"]
    end

    subgraph "JavaScript Bindings Layer"
        Bindings["C++ Bindings<br/>(bindings/)"]
        WebCore["WebCore APIs<br/>(webcore/)"]
        NodeAPI["Node.js APIs<br/>(node/)"]
        BunAPI["Bun APIs<br/>(api/)"]
    end

    subgraph "Built-in Modules"
        NodeModules["node:* modules<br/>(js/node/)"]
        BunModules["bun:* modules<br/>(js/bun/)"]
        InternalModules["Internal modules<br/>(js/internal/)"]
    end

    subgraph "Tooling Layer"
        Bundler["Bundler<br/>(bundler/)"]
        PackageManager["Package Manager<br/>(install/)"]
        Transpiler["Transpiler<br/>(transpiler.zig)"]
        TestRunner["Test Runner<br/>(bun:test)"]
    end

    subgraph "Core Services"
        Parser["JS Parser<br/>(js_parser.zig)"]
        Lexer["JS Lexer<br/>(js_lexer.zig)"]
        Resolver["Module Resolver<br/>(resolver/)"]
        HTTP["HTTP Client<br/>(http.zig)"]
        Server["HTTP Server<br/>(api/server.zig)"]
    end

    subgraph "Core Runtime"
        Core["Core Runtime<br/>(bun.zig)"]
        Sys["System Abstraction<br/>(sys.zig)"]
        Alloc["Memory Allocators<br/>(allocators/)"]
        FS["File System<br/>(fs.zig)"]
    end

    subgraph "External Systems"
        NPM["npm Registry"]
        FileSystem["File System"]
        Network["Network"]
        DB["Databases<br/>(SQLite, PostgreSQL, MySQL)"]
    end

    CLI --> VM
    CLI --> Bundler
    CLI --> PackageManager
    CLI --> TestRunner

    VM --> EL
    VM --> JSC
    VM --> GO
    VM --> Transpiler

    EL --> Bindings

    JSC --> Bindings
    GO --> WebCore
    GO --> NodeAPI
    GO --> BunAPI

    WebCore --> NodeModules
    NodeAPI --> NodeModules
    BunAPI --> BunModules
    BunAPI --> InternalModules

    Bundler --> Parser
    Bundler --> Resolver
    Transpiler --> Parser
    Transpiler --> Lexer

    PackageManager --> HTTP
    PackageManager --> NPM

    Server --> EL
    HTTP --> Network

    Parser --> Core
    Resolver --> FS
    FS --> FileSystem

    Core --> Sys
    Core --> Alloc
    Sys --> FileSystem
    Sys --> Network

    BunAPI --> DB
```

## レイヤー構成図

```mermaid
graph TB
    subgraph "Layer 1: CLI/Entry"
        L1["main.zig<br/>cli.zig<br/>cli/*_command.zig"]
    end

    subgraph "Layer 2: JavaScript Runtime"
        L2["VirtualMachine.zig<br/>event_loop.zig<br/>JSGlobalObject"]
    end

    subgraph "Layer 3: JavaScript Bindings"
        L3["bindings/*.cpp<br/>bindings/*.zig<br/>*.classes.ts"]
    end

    subgraph "Layer 4: Built-in Modules"
        L4["js/node/*<br/>js/bun/*<br/>js/builtins/*"]
    end

    subgraph "Layer 5: Tooling"
        L5["bundler/*<br/>install/*<br/>transpiler.zig"]
    end

    subgraph "Layer 6: Core Services"
        L6["js_parser.zig<br/>resolver/*<br/>http.zig"]
    end

    subgraph "Layer 7: Core Runtime"
        L7["bun.zig<br/>sys.zig<br/>allocators/*"]
    end

    L1 --> L2
    L1 --> L5
    L2 --> L3
    L3 --> L4
    L2 --> L6
    L5 --> L6
    L6 --> L7
    L4 --> L7
```

## コンポーネント依存関係図

```mermaid
graph LR
    subgraph "Runtime Components"
        VM[VirtualMachine]
        EL[EventLoop]
        ML[ModuleLoader]
        TPC[TranspilerCache]
    end

    subgraph "Parser Components"
        JP[js_parser]
        JL[js_lexer]
        JPr[js_printer]
        AST[ast]
    end

    subgraph "Bundler Components"
        BV2[bundle_v2]
        LC[LinkerContext]
        PT[ParseTask]
        CH[Chunk]
    end

    subgraph "Package Manager Components"
        PM[PackageManager]
        LF[lockfile]
        NPM[npm]
        DI[dependency]
    end

    VM --> EL
    VM --> ML
    VM --> TPC

    ML --> JP
    TPC --> JP

    JP --> JL
    JP --> AST
    JP --> JPr

    BV2 --> LC
    BV2 --> PT
    LC --> CH
    PT --> JP

    PM --> LF
    PM --> NPM
    PM --> DI
```

## データフロー図

### JavaScript実行フロー

```mermaid
sequenceDiagram
    participant User
    participant CLI
    participant VM as VirtualMachine
    participant EL as EventLoop
    participant Trans as Transpiler
    participant JSC as JavaScriptCore

    User->>CLI: bun run script.ts
    CLI->>VM: initialize()
    VM->>Trans: transpile(script.ts)
    Trans-->>VM: JavaScript code
    VM->>JSC: evaluate(code)
    JSC->>EL: schedule tasks
    loop Event Loop
        EL->>EL: process microtasks
        EL->>EL: process task queue
        EL->>JSC: run callbacks
    end
    JSC-->>VM: result
    VM-->>CLI: exit code
```

### HTTPサーバー リクエストフロー

```mermaid
sequenceDiagram
    participant Client
    participant uWS as uWebSockets
    participant EL as EventLoop
    participant Handler as JS Handler
    participant Response

    Client->>uWS: HTTP Request
    uWS->>EL: connection event
    EL->>Handler: invoke callback
    Handler->>Handler: process request
    Handler->>Response: create Response
    Response->>uWS: send response
    uWS->>Client: HTTP Response
```

### パッケージインストールフロー

```mermaid
sequenceDiagram
    participant User
    participant CLI
    participant PM as PackageManager
    participant LF as Lockfile
    participant NPM as npm Registry
    participant FS as FileSystem

    User->>CLI: bun install
    CLI->>PM: install()
    PM->>LF: read(bun.lock)
    PM->>PM: resolve dependencies
    PM->>NPM: fetch metadata
    NPM-->>PM: package info
    PM->>NPM: download tarballs
    NPM-->>PM: package files
    PM->>FS: extract to node_modules
    PM->>LF: write(bun.lock)
    PM-->>CLI: success
```

## モジュール構成図

```mermaid
graph TB
    subgraph "Entry Points"
        main[main.zig]
        bunzig[bun.zig]
    end

    subgraph "CLI Commands"
        cli[cli.zig]
        run[run_command.zig]
        build[build_command.zig]
        install[install_command.zig]
        test[test - bun:test]
    end

    subgraph "JavaScript Runtime"
        vm[VirtualMachine.zig]
        eventloop[event_loop.zig]
        console[ConsoleObject.zig]
        module[ModuleLoader.zig]
    end

    subgraph "JS/TS Processing"
        parser[js_parser.zig]
        lexer[js_lexer.zig]
        printer[js_printer.zig]
        transpiler[transpiler.zig]
    end

    subgraph "Bundler"
        bundlev2[bundle_v2.zig]
        linker[LinkerContext.zig]
        chunk[Chunk.zig]
    end

    subgraph "Package Manager"
        pkgmgr[PackageManager.zig]
        lockfile[lockfile.zig]
        npm[npm.zig]
    end

    subgraph "Network"
        http[http.zig]
        server[server.zig]
        websocket[WebSocket]
    end

    subgraph "Core"
        sys[sys.zig]
        fs[fs.zig]
        allocators[allocators/]
    end

    main --> bunzig
    bunzig --> cli
    cli --> run
    cli --> build
    cli --> install
    cli --> test

    run --> vm
    vm --> eventloop
    vm --> console
    vm --> module
    module --> transpiler
    transpiler --> parser
    parser --> lexer
    transpiler --> printer

    build --> bundlev2
    bundlev2 --> linker
    linker --> chunk
    bundlev2 --> parser

    install --> pkgmgr
    pkgmgr --> lockfile
    pkgmgr --> npm

    server --> eventloop
    http --> sys

    vm --> sys
    bundlev2 --> fs
    pkgmgr --> fs
    fs --> sys
    sys --> allocators
```

## 外部システム連携図

```mermaid
graph LR
    subgraph "Bun Runtime"
        Core[Bun Core]
        HTTP[HTTP Client]
        Server[HTTP Server]
        SQL[SQL Drivers]
        Shell[Shell]
    end

    subgraph "JavaScript Engine"
        JSC[JavaScriptCore<br/>WebKit]
    end

    subgraph "System Libraries"
        libuv[libuv<br/>Windows only]
        BoringSSL[BoringSSL]
        mimalloc[mimalloc]
        zlib[zlib]
        sqlite[SQLite]
    end

    subgraph "External Services"
        NPM[npm Registry]
        GitHub[GitHub<br/>git dependencies]
        PG[PostgreSQL]
        MySQL[MySQL]
    end

    subgraph "Operating System"
        FS[File System]
        Net[Network Stack]
        Proc[Process Management]
    end

    Core --> JSC
    Core --> mimalloc
    Core --> FS
    Core --> Proc

    HTTP --> BoringSSL
    HTTP --> Net
    HTTP --> NPM
    HTTP --> GitHub

    Server --> BoringSSL
    Server --> Net

    SQL --> sqlite
    SQL --> PG
    SQL --> MySQL

    Shell --> Proc
    Shell --> FS

    Core -.->|Windows| libuv
    HTTP --> zlib
```
