# アーキテクチャ設計書

## 概要

本ドキュメントは、Bunプロジェクトのアーキテクチャ設計について記述する。Bunは、速度を重視した設計のオールインワンJavaScriptランタイム&ツールキットであり、バンドラー、テストランナー、Node.js互換パッケージマネージャーを含む。主にZigで記述され、JavaScriptCore（WebKit）との統合のためにC++も使用している。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| JavaScriptCore (WebKit) | C++バインディング | JavaScript実行エンジン |
| npmレジストリ | HTTP API | パッケージのダウンロード・公開 |
| ファイルシステム | システムコール | ファイルI/O、モジュール解決 |
| ネットワーク | TCP/UDP/HTTP | fetch API、WebSocket、HTTPサーバー |
| SQLite | ネイティブライブラリ | bun:sqliteモジュール |
| PostgreSQL/MySQL | ネイティブプロトコル | データベース接続 |
| libuv | C FFI | Windows環境でのイベントループ |

## レイヤー構成

### アーキテクチャスタイル

モジュラーモノリス + ネイティブランタイムアーキテクチャ

Bunは、複数の機能ドメイン（ランタイム、バンドラー、パッケージマネージャー、テストランナー）を単一のバイナリに統合したモジュラーモノリス構造を採用している。各機能は独立したモジュールとして実装されているが、共通の基盤（Zigランタイム、メモリアロケータ、イベントループ）を共有することで、高いパフォーマンスと一貫した動作を実現している。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| CLI/Entry | コマンドライン引数解析、コマンドディスパッチ | `src/cli.zig`, `src/main.zig` |
| JavaScript Runtime | JS実行環境、グローバルオブジェクト管理 | `src/bun.js/VirtualMachine.zig`, `src/bun.js/event_loop.zig` |
| JavaScript Bindings | JSCとZig/C++の連携 | `src/bun.js/bindings/` |
| Built-in Modules | Node.js互換モジュール、Bun固有API | `src/js/node/`, `src/js/bun/` |
| Bundler | コードバンドリング、Tree-shaking | `src/bundler/` |
| Package Manager | 依存関係解決、パッケージインストール | `src/install/` |
| Transpiler | TypeScript/JSX変換 | `src/transpiler.zig`, `src/js_parser.zig` |
| HTTP/Network | HTTPクライアント/サーバー、WebSocket | `src/http.zig`, `src/bun.js/api/server.zig` |
| File System | ファイルI/O、パス解決 | `src/fs.zig`, `src/resolver/` |
| Core Runtime | メモリ管理、システムコール抽象化 | `src/bun.zig`, `src/sys.zig`, `src/allocators/` |

### レイヤー間の依存関係ルール

- **依存の方向性**: 上位レイヤーから下位レイヤーへの一方向依存
- **許可される参照**:
  - CLI -> JavaScript Runtime, Bundler, Package Manager
  - JavaScript Runtime -> Core Runtime, JavaScript Bindings
  - Built-in Modules -> JavaScript Bindings, Core Runtime
- **禁止される参照**:
  - Core Runtime -> CLI, JavaScript Runtime（循環依存の防止）
  - Built-in Modules -> Bundler, Package Manager（独立性の維持）

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Runtime Core | JavaScript実行、イベントループ | `src/bun.js/`, `src/bun.zig` |
| JavaScript Parsing | JS/TS/JSXパース、AST構築 | `src/js_parser.zig`, `src/js_lexer.zig`, `src/ast/` |
| Module Resolution | モジュールパス解決、インポート処理 | `src/resolver/` |
| Bundling | コードバンドリング、最適化 | `src/bundler/` |
| Package Management | npm互換パッケージ管理 | `src/install/` |
| HTTP/Networking | ネットワーク通信 | `src/http.zig`, `src/http/` |
| Shell | クロスプラットフォームシェル | `src/shell/` |
| CSS Processing | CSSパース、バンドリング | `src/css/` |
| Database | SQL系データベース連携 | `src/sql/` |
| Server-Side Rendering | Bake SSRフレームワーク | `src/bake/` |

### パッケージ構造

```
src/
├── bun.zig                 # コアランタイム、モジュールエクスポート
├── main.zig                # エントリポイント
├── cli.zig                 # CLIコマンド処理
├── cli/                    # 各CLIコマンド実装
│   ├── build_command.zig
│   ├── install_command.zig
│   ├── run_command.zig
│   └── ...
├── bun.js/                 # JavaScriptランタイム
│   ├── VirtualMachine.zig  # VM管理
│   ├── event_loop.zig      # イベントループ
│   ├── bindings/           # JSC C++バインディング
│   ├── api/                # Bun API実装
│   │   ├── server.zig      # HTTPサーバー
│   │   ├── FFI.zig         # 外部関数インターフェース
│   │   └── ...
│   ├── node/               # Node.js互換API
│   └── webcore/            # Web API実装
├── js/                     # JavaScriptビルトインモジュール
│   ├── node/               # node:* モジュール
│   ├── bun/                # bun:* モジュール
│   ├── builtins/           # コアビルトイン関数
│   ├── internal/           # 内部モジュール
│   └── thirdparty/         # サードパーティモジュール置換
├── bundler/                # バンドラー
│   ├── bundle_v2.zig
│   ├── LinkerContext.zig
│   └── ParseTask.zig
├── install/                # パッケージマネージャー
│   ├── PackageManager.zig
│   ├── lockfile.zig
│   └── npm.zig
├── transpiler.zig          # トランスパイラー
├── js_parser.zig           # JSパーサー
├── js_lexer.zig            # JS字句解析
├── js_printer.zig          # JSコード生成
├── resolver/               # モジュール解決
├── http.zig                # HTTPクライアント
├── http/                   # HTTP関連モジュール
├── shell/                  # シェル実装
├── css/                    # CSSプロセッサ
├── sql/                    # データベース連携
│   ├── postgres/
│   └── mysql/
├── bake/                   # SSRフレームワーク
├── allocators/             # カスタムアロケータ
├── sys.zig                 # システムコール抽象化
└── codegen/                # コード生成スクリプト
```

### コンポーネント依存関係

主要コンポーネント間の依存関係:

1. **VirtualMachine** -> EventLoop, JSGlobalObject, Transpiler, ModuleLoader
2. **EventLoop** -> uws (libuv wrapper), Task Queue, Timer
3. **Bundler** -> Parser, Linker, Resolver, Transpiler
4. **PackageManager** -> Lockfile, NPM Registry Client, Dependency Resolver
5. **HTTPServer** -> EventLoop, WebSocket, TLS
6. **Transpiler** -> Parser, Printer, SourceMap

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| RDB | SQLite | 組み込み | bun:sqliteモジュール |
| RDB | PostgreSQL | ネイティブプロトコル | src/sql/postgres/ |
| RDB | MySQL | ネイティブプロトコル | src/sql/mysql/ |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| mimalloc | 組み込み | メモリアロケーション | N/A |
| RuntimeTranspilerCache | 内部実装 | トランスパイル結果キャッシュ | セッション |
| HTTPキャッシュ | 内部実装 | HTTPレスポンスキャッシュ | 設定依存 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 内部TaskQueue | 内部実装 | イベントループタスク管理 |
| ImmediateTaskQueue | 内部実装 | setImmediate用キュー |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| N/A | - | - |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| JavaScriptCore | WebKit | JavaScript実行エンジン |
| BoringSSL | 組み込み | TLS/暗号化処理 |
| zlib | 組み込み | 圧縮/解凍 |
| Brotli | 組み込み | Brotli圧縮 |
| picohttp | 組み込み | HTTPパース |
| uSockets/uWebSockets | 組み込み | イベントループ、WebSocket |

## データフロー

### リクエスト処理の流れ

#### JavaScript実行フロー

1. CLIがコマンドを解析し、適切なコマンドハンドラを呼び出す
2. VirtualMachineが初期化され、JSGlobalObjectが作成される
3. EventLoopが開始され、エントリポイントのJSファイルがロードされる
4. Transpilerがソースコードをトランスパイル（必要に応じて）
5. JavaScriptCoreがコードを実行
6. イベントループがmicrotaskとタスクキューを処理
7. 全タスク完了後、VMがシャットダウン

#### HTTPサーバーリクエストフロー

1. `Bun.serve()`でHTTPサーバーを作成
2. uWebSocketsがリスニングソケットをバインド
3. クライアント接続時、EventLoopがコールバックをスケジュール
4. リクエストハンドラ（JavaScript関数）が呼び出される
5. Responseオブジェクトが作成され、クライアントに送信
6. コネクションがkeep-aliveまたはクローズ

### 非同期処理の流れ

1. JavaScript側でPromise/async操作を開始
2. Zigコードが非同期操作を開始（I/O、タイマー等）
3. EventLoopが完了を監視
4. 完了時、対応するPromiseが解決
5. microtaskキューがドレインされ、then/catchハンドラが実行

### データ永続化の流れ

#### パッケージマネージャー（bun install）

1. package.jsonを解析
2. 依存関係ツリーを構築
3. lockfile（bun.lock）を読み込み/生成
4. npmレジストリからパッケージメタデータを取得
5. tarballをダウンロード、展開
6. node_modules/に配置
7. lockfileを更新、保存

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| NPM認証トークン | `src/install/npm.zig` | npmレジストリ認証 |
| 環境変数 | `src/env_loader.zig` | APIキー等の読み込み |
| TLS証明書検証 | `src/http.zig` | HTTPSクライアント |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | `src/output.zig`, `src/logger.zig` | stdout/stderr |
| デバッグログ | `Output.scoped()` マクロ | stdout（デバッグビルド時） |
| エラーログ | logger.Log構造体 | stderr |

### エラーハンドリング

Zigのエラー型システムを活用した明示的なエラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| JSError | JavaScript例外としてスロー | スタックトレース付きエラー |
| OOM (Out of Memory) | panic または graceful degradation | プロセス終了またはエラーメッセージ |
| I/Oエラー | エラー型を返却、呼び出し元で処理 | ユーザーフレンドリーなエラーメッセージ |
| パースエラー | ロケーション付きエラー報告 | 行番号・列番号付きエラー |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| SQLite | ネイティブトランザクション | SERIALIZABLE |
| パッケージインストール | アトミックなファイル操作 | N/A |
| lockfile更新 | 一時ファイル経由のアトミック書き込み | N/A |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| ゼロコスト抽象化 | 全体 | Zigのcomptime機能による最適化 |
| 明示的なメモリ管理 | Core Runtime | カスタムアロケータ、defer によるリソース解放 |
| フェイルファースト | エラーハンドリング | アサーション、明示的なエラー型 |
| 単一責任原則 | モジュール分割 | 機能ごとの独立したファイル |
| 依存性逆転 | JSバインディング | インターフェース経由のJSC連携 |

### コーディング規約

#### Zig

- プライベートフィールドは `#` プレフィックスを使用
- decl literalsを推奨
- `@import`はファイル末尾または構造体の末尾に配置
- 関数内でのインライン`@import`は禁止

#### C++

- clang-formatによる自動フォーマット
- JavaScriptCoreのコーディング規約に準拠

#### TypeScript/JavaScript (Built-in Modules)

- `.$call`/`.$apply` を使用（tamper-proof）
- `$` プレフィックスでプライベートAPI、JSC intrinsicsにアクセス
- `require()`は文字列リテラルのみ

## 備考

### ビルドシステム

- Zigビルドシステム（`build.zig`）とCMakeのハイブリッド
- コード生成スクリプト（`src/codegen/`）によるバインディング自動生成
- `.classes.ts`ファイルからZig/C++バインディングを生成

### パフォーマンス最適化

- mimalloc によるメモリアロケーション最適化
- SIMD命令の活用（文字列処理等）
- JavaScriptCoreのバイトコードキャッシュ
- トランスパイル結果のキャッシュ

### クロスプラットフォーム対応

- macOS, Linux, Windows対応
- プラットフォーム固有コードは `src/darwin.zig`, `src/linux.zig`, `src/windows.zig` に分離
- `src/sys.zig` による抽象化レイヤー
