# Bun コードリーディングガイドライン

## はじめに

このガイドラインは、Bunのコードベースを効率的に理解するための手引きです。
Zig、C++、TypeScriptに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

**Bunとは:**
Bun は、速度を重視したオールインワン JavaScript ランタイム & ツールキットです。バンドラー、テストランナー、Node.js 互換パッケージマネージャを備えており、主に Zig で記述され、JavaScriptCore エンジン（WebKit）との統合には C++ が使用されています。

---

## 1. 言語基礎

> このセクションでは、Bunで使用されている主要言語の基本構文と概念を解説します。

### 1.1 Zigプログラム構造

Zigは、Bunのコアランタイムを構成する主要言語です。シンプルかつ安全なシステムプログラミングを重視しています。

```zig
// ファイル: src/bun.zig:7-9
const bun = @This();

pub const Environment = @import("./env.zig");
pub const env_var = @import("./env_var.zig");
```

**解説:**
- `@This()` はモジュール自身を参照する特殊構文です
- `pub const` は公開定数を宣言します
- `@import` は他のZigファイルを読み込みます

### 1.2 データ型と変数

**Zigの基本データ型:**

```zig
// ファイル: src/resolver/resolver.zig:45-46
pub threadlocal var extension_path: [512]u8 = undefined;
pub threadlocal var tsconfig_match_full_buf: bun.PathBuffer = undefined;
```

**解説:**
- `threadlocal` はスレッドローカル変数を宣言します
- `[512]u8` は512バイトの配列型です
- `undefined` は初期化前の未定義状態を示します

**TypeScriptの型定義:**

```typescript
// ファイル: src/js/node/fs.ts:2
import type { Dirent as DirentType, PathLike, Stats as StatsType } from "fs";
```

### 1.3 制御構造

**Zigの条件分岐とエラーハンドリング:**

```zig
// ファイル: src/bun.zig:112-114
if (std.math.isNan(value)) {
    return 0;
}
```

**Zigのswitch文:**

```zig
// ファイル: src/bun.js/api/server.zig:41-46
pub fn memoryCost(this: AnyRoute) usize {
    return switch (this) {
        .static => |static_route| static_route.memoryCost(),
        .file => |file_route| file_route.memoryCost(),
        .html => |html_bundle_route| html_bundle_route.data.memoryCost(),
        .framework_router => @sizeOf(bun.bake.Framework.FileSystemRouterType),
    };
}
```

### 1.4 関数/メソッド定義

**Zig関数の定義:**

```zig
// ファイル: src/resolver/resolver.zig:6-11
pub fn isPackagePath(path: string) bool {
    return !std.fs.path.isAbsolute(path) and @call(bun.callmod_inline, isPackagePathNotAbsolute, .{path});
}
```

**TypeScript関数（ビルトイン用）:**

```typescript
// ファイル: src/js/node/fs.ts:22-28
function ensureCallback(callback) {
  if (!$isCallable(callback)) {
    throw $ERR_INVALID_ARG_TYPE("cb", "function", callback);
  }
  return callback;
}
```

### 1.5 モジュール/インポート

**Zigのインポート方式:**

```zig
// ファイル: src/bun.js/api/crypto.zig:5-18
pub const PasswordObject = @import("./crypto/PasswordObject.zig").PasswordObject;
pub const CryptoHasher = @import("./crypto/CryptoHasher.zig").CryptoHasher;
pub const HMAC = @import("./crypto/HMAC.zig");
```

**TypeScriptのインポート（ビルトイン用特殊構文）:**

```typescript
// ファイル: src/js/node/fs.ts:3-6
const EventEmitter = require("node:events");
const promises = require("node:fs/promises");
const types = require("node:util/types");
const { validateFunction, validateInteger } = require("internal/validators");
```

**重要:** ビルトインモジュールでは通常のES Module構文ではなく、`require()` を使用する必要があります。

---

## 2. プロジェクト固有の概念

> このセクションでは、Bunプロジェクト特有の概念を解説します。

### 2.1 JavaScriptCore（JSC）統合

BunはWebKitのJavaScriptCoreエンジンを使用しています。C++で記述されたバインディングがZigコードとJSCの間を橋渡しします。

```cpp
// ファイル: src/bun.js/bindings/ZigGlobalObject.cpp:1-10
#include "root.h"
#include "ZigGlobalObject.h"
#include "JavaScriptCore/ArgList.h"
#include "JavaScriptCore/JSCellButterfly.h"
```

### 2.2 `$` プレフィックスと特殊構文

ビルトインJavaScriptでは、`$` プレフィックスを使用してプライベートAPIにアクセスします。

```typescript
// ファイル: src/js/builtins/ReadableStreamInternals.ts:29-34
export function readableStreamReaderGenericInitialize(reader, stream) {
  $putByIdDirectPrivate(reader, "ownerReadableStream", stream);
  $putByIdDirectPrivate(stream, "reader", reader);
  if ($getByIdDirectPrivate(stream, "state") === $streamReadable)
    $putByIdDirectPrivate(reader, "closedPromiseCapability", $newPromiseCapability(Promise));
}
```

**主な`$`関数:**
- `$isCallable(fn)` - 関数かどうかをチェック
- `$putByIdDirectPrivate(obj, key, value)` - プライベートプロパティを設定
- `$getByIdDirectPrivate(obj, key)` - プライベートプロパティを取得
- `$assert(condition)` - アサーション（リリースビルドでは削除）
- `$debug(...)` - デバッグログ（リリースビルドでは削除）

**重要:** `.call` や `.apply` の代わりに `.$call` と `.$apply` を使用してください。ユーザーによる改ざんを防ぐためです。

### 2.3 コード生成システム

Bunは `.classes.ts` ファイルからC++とZigのバインディングを自動生成します。

```typescript
// ファイル例: src/bun.js/api/crypto.classes.ts, server.classes.ts など
```

主なコード生成スクリプト:
- `src/codegen/generate-classes.ts` - `.classes.ts` からバインディング生成
- `src/codegen/bundle-modules.ts` - ビルトインモジュールのバンドル
- `src/codegen/bundle-functions.ts` - グローバル関数のバンドル

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `*.zig` | Zigソースファイル | `bun.zig`, `cli.zig` |
| `*.cpp` | C++ソースファイル | `ZigGlobalObject.cpp` |
| `*.ts` | TypeScriptソースファイル | `fs.ts`, `crypto.ts` |
| `*.classes.ts` | クラス定義ファイル（コード生成用） | `server.classes.ts` |
| `*.test.ts` | テストファイル | `serve.test.ts` |
| `PascalCase.zig` | 主要モジュール/クラス | `PackageManager.zig` |
| `snake_case.zig` | ユーティリティ/補助モジュール | `bundle_v2.zig` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `JS` | JavaScriptに公開されるクラス | `JSBundler`, `JSTranspiler` |
| `Zig` | Zigから呼び出される関数 | `ZigGlobalObject` |
| `pub` | 公開メンバー（Zig） | `pub fn start()` |
| `_` | プライベート/内部使用 | `_lazyGlob` |
| `$` | ビルトイン専用プライベートAPI | `$isCallable` |
| `threadlocal` | スレッドローカル変数 | `threadlocal var buf` |

### 3.3 ディレクトリ構成規則

| ディレクトリパターン | 説明 |
|---------------------|------|
| `src/bun.js/api/` | Bun固有のJavaScript API実装 |
| `src/bun.js/node/` | Node.js互換モジュール |
| `src/bun.js/bindings/` | C++ JavaScriptCoreバインディング |
| `src/js/node/` | Node.jsビルトインモジュール（TypeScript） |
| `src/js/bun/` | Bunビルトインモジュール（TypeScript） |
| `src/install/` | パッケージマネージャ |
| `src/bundler/` | バンドラー |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
bun-main/
├── src/                          # メインソースコード
│   ├── bun.zig                   # Zigモジュールルート
│   ├── cli.zig                   # CLIコマンドオーケストレーション
│   ├── js_parser.zig             # JavaScriptパーサー
│   ├── js_lexer.zig              # JavaScriptレキサー
│   ├── resolver/                 # モジュール解決システム
│   │   └── resolver.zig          # メインリゾルバー
│   ├── bun.js/                   # JavaScriptランタイム
│   │   ├── api/                  # Bun固有API実装
│   │   │   ├── server.zig        # HTTPサーバー
│   │   │   ├── crypto.zig        # 暗号化API
│   │   │   └── *.classes.ts      # クラス定義
│   │   ├── bindings/             # C++ JSCバインディング
│   │   │   └── ZigGlobalObject.cpp
│   │   └── node/                 # Node.js互換レイヤー
│   ├── js/                       # ビルトインJavaScriptモジュール
│   │   ├── node/                 # node:fs, node:path など
│   │   ├── bun/                  # bun:ffi, bun:sqlite など
│   │   ├── builtins/             # コアビルトイン（streams, console）
│   │   └── internal/             # 内部モジュール
│   ├── bundler/                  # バンドラー実装
│   │   ├── bundle_v2.zig
│   │   └── LinkerContext.zig
│   ├── install/                  # パッケージマネージャ
│   │   ├── PackageManager.zig
│   │   └── lockfile/             # ロックファイル処理
│   ├── codegen/                  # コード生成スクリプト
│   │   ├── generate-classes.ts
│   │   └── bundle-modules.ts
│   ├── shell/                    # クロスプラットフォームシェル
│   ├── css/                      # CSSパーサー/プロセッサー
│   └── http/                     # HTTPクライアント
├── test/                         # テストディレクトリ
│   ├── js/bun/                   # Bun固有APIテスト
│   ├── js/node/                  # Node.js互換テスト
│   ├── js/web/                   # Web APIテスト
│   ├── cli/                      # CLIコマンドテスト
│   ├── bundler/                  # バンドラーテスト
│   └── regression/issue/         # 回帰テスト
├── packages/                     # NPMパッケージ
│   └── bun-types/                # TypeScript型定義
└── scripts/                      # ビルド/ユーティリティスクリプト
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | コアランタイム実装 | `bun.zig`, `cli.zig` |
| `src/bun.js/api/` | Bun固有JavaScript API | `server.zig`, `crypto.zig` |
| `src/bun.js/bindings/` | JSCバインディング（C++） | `ZigGlobalObject.cpp` |
| `src/js/node/` | Node.jsビルトイン | `fs.ts`, `path.ts` |
| `src/bundler/` | JavaScriptバンドラー | `bundle_v2.zig` |
| `src/install/` | パッケージマネージャ | `PackageManager.zig` |
| `test/` | テストスイート | `*.test.ts` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Bunは多層アーキテクチャを採用しており、パフォーマンスと互換性の両立を実現しています。

```
┌─────────────────────────────────────────────────────────────────┐
│                    JavaScript/TypeScript                         │
│                    (ユーザーコード)                              │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│              Built-in JavaScript Modules                         │
│     src/js/node/  src/js/bun/  src/js/builtins/                 │
│              (TypeScript with $ syntax)                          │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                 C++ Bindings Layer                               │
│              src/bun.js/bindings/                               │
│         (JavaScriptCore ←→ Zig Bridge)                          │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                   Zig Core Runtime                               │
│        src/bun.js/api/  src/resolver/  src/bundler/             │
│     (Performance-critical implementations)                       │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│              JavaScriptCore Engine (WebKit)                      │
│                   (JavaScript VM)                                │
└─────────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| JavaScript層 | ユーザー向けAPI提供 | `src/js/node/fs.ts` |
| C++バインディング層 | JSC↔Zig連携 | `ZigGlobalObject.cpp` |
| Zigコア層 | 高性能実装 | `server.zig`, `resolver.zig` |
| JSCエンジン層 | JavaScript実行 | WebKit依存 |

### 5.3 データフロー

1. **ユーザーコード実行**: JavaScriptCore がユーザーコードを解析・実行
2. **ビルトイン呼び出し**: `require("node:fs")` などでビルトインモジュールをロード
3. **ネイティブ処理**: C++バインディング経由でZig実装を呼び出し
4. **結果返却**: Zig→C++→JavaScript の順で結果を返却

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

**メインエントリーポイント: `src/bun.zig`**

```zig
// ファイル: src/bun.zig:1-7
//! This is the root source file of Bun's zig module. It can be imported using
//! `@import("bun")`, and should be able to reach all code via `.` syntax.
//!
//! Prefer adding new code into a separate file and adding an import, or putting
//! code in the relevant namespace.

const bun = @This();
```

**CLIエントリーポイント: `src/cli.zig`**

```zig
// ファイル: src/cli.zig:11-25
pub const Cli = struct {
    pub fn start(allocator: std.mem.Allocator) void {
        is_main_thread = true;
        start_time = std.time.nanoTimestamp();
        log_ = logger.Log.init(allocator);
        var log = &log_;
        Command.start(allocator, log) catch |err| {
            log.print(Output.errorWriter()) catch {};
            bun.crash_handler.handleRootError(err, @errorReturnTrace());
        };
    }
};
```

### 6.2 HTTPサーバー

**場所:** `src/bun.js/api/server.zig`

```zig
// ファイル: src/bun.js/api/server.zig:4-6
pub const WebSocketServerContext = @import("./server/WebSocketServerContext.zig");
pub const HTTPStatusText = @import("./server/HTTPStatusText.zig");
pub const HTMLBundle = @import("./server/HTMLBundle.zig");
```

サーバー実装はモジュール化されており、各機能は個別のファイルに分離されています。

### 6.3 パッケージマネージャ

**場所:** `src/install/PackageManager.zig`

```zig
// ファイル: src/install/PackageManager.zig:1-20
cache_directory_: ?std.fs.Dir = null,
cache_directory_path: stringZ = "",
root_dir: *Fs.FileSystem.DirEntry,
allocator: std.mem.Allocator,
log: *logger.Log,
resolve_tasks: ResolveTaskQueue = .{},
```

### 6.4 JavaScriptパーサー

**場所:** `src/js_parser.zig`

```zig
// ファイル: src/js_parser.zig:1-5
//! ** IMPORTANT **
//! ** When making changes to the JavaScript Parser that impact runtime behavior or fix bugs **
//! ** you must also increment the `expected_version` in RuntimeTranspilerCache.zig **
//! ** IMPORTANT **
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| `comptime` | コンパイル時計算 | 高 | 各種Zigファイル |
| `threadlocal` | スレッドローカルバッファ | 高 | `resolver.zig` |
| `$ prefix` | ビルトインプライベートAPI | 高 | `src/js/builtins/` |
| クラス生成 | `.classes.ts`からの自動生成 | 中 | `*.classes.ts` |
| エラーハンドリング | `JSError`型 | 高 | 各種API |

### 各パターンの詳細

#### パターン1: `threadlocal` バッファ

**目的:** 高頻度で使用される一時バッファをスレッドごとに再利用

**実装例:**
```zig
// ファイル: src/resolver/resolver.zig:45-51
pub threadlocal var extension_path: [512]u8 = undefined;
pub threadlocal var tsconfig_match_full_buf: bun.PathBuffer = undefined;
pub threadlocal var esm_subpath: [512]u8 = undefined;
```

**解説:** アロケーション回数を削減し、パフォーマンスを向上させます。

#### パターン2: `$` プライベートAPI

**目的:** ユーザーコードから改ざんされない安全なAPI呼び出し

**実装例:**
```typescript
// ファイル: src/js/builtins/ReadableStreamInternals.ts:30-31
$putByIdDirectPrivate(reader, "ownerReadableStream", stream);
$putByIdDirectPrivate(stream, "reader", reader);
```

**解説:** `$` プレフィックス関数はJSCの内部APIを直接呼び出します。

#### パターン3: エラーハンドリング（JSError）

**目的:** JavaScript例外とZigエラーの統合管理

**実装例:**
```zig
// ファイル: src/bun.zig:156-175
pub const JSError = error{
    JSError,
    OutOfMemory,
    JSTerminated,
};
```

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: `bun run` コマンド実行フロー

**概要:** ユーザーが `bun run script.js` を実行した際の処理フロー

**処理フロー:**
```
CLI起動 → コマンド解析 → RunCommand → JSランタイム初期化 → スクリプト実行
```

**詳細な追跡:**

1. **CLI起動** (`src/cli.zig:11-25`)
   ```zig
   pub fn start(allocator: std.mem.Allocator) void {
       is_main_thread = true;
       start_time = std.time.nanoTimestamp();
       Command.start(allocator, log) catch |err| {
           // エラーハンドリング
       };
   }
   ```

2. **コマンドディスパッチ** (`src/cli.zig:79`)
   ```zig
   pub const RunCommand = @import("./cli/run_command.zig").RunCommand;
   ```

3. **JavaScript実行** (JSCエンジン)
   - ZigGlobalObject経由でJSCを初期化
   - ユーザースクリプトをパース・実行

#### 例2: ファイルシステム操作フロー

**概要:** `fs.readFile()` 呼び出しの処理フロー

**処理フロー:**
```
JavaScript API → TypeScriptビルトイン → Zigバインディング → システムコール
```

**詳細な追跡:**

1. **ビルトインモジュール** (`src/js/node/fs.ts:1-14`)
   ```typescript
   // Hardcoded module "node:fs"
   const { fs } = promises.$data;
   ```

2. **ネイティブバインディング呼び出し**
   - `fs` オブジェクトはZigで実装されたネイティブバインディング
   - システムコールを実行して結果を返却

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `CLAUDE.md` - 開発ガイドライン
3. このコードリーディングガイドライン
4. `docs/code-to-docs/アーキテクチャ設計書/` - アーキテクチャ詳細

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/` - 機能一覧
2. `docs/code-to-docs/機能設計書/` - 詳細設計
3. 関連ソースコード（`src/`配下）

#### 改修作業を行う場合
1. 対象機能の設計書
2. 関連テストファイル（`test/`配下）
3. `test/CLAUDE.md` - テスト記述ガイドライン

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| CLAUDE.md | 開発ガイドライン | 開発開始時 |
| test/CLAUDE.md | テスト記述ガイド | テスト作成時 |
| src/js/CLAUDE.md | ビルトイン記述ガイド | ビルトイン開発時 |
| package.json | プロジェクト設定 | ビルドコマンド確認時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: Zigの `@import` がどこを参照しているかわからない
A: `@import("bun")` は `src/bun.zig` を参照します。相対パスの場合は現在のファイルからの相対位置です。

#### Q: `$` プレフィックスの関数が見つからない
A: `$` プレフィックス関数はビルド時に `@` に変換され、JSCの内部APIを呼び出します。定義はJSCソースコード内にあります。

#### Q: `.classes.ts` ファイルの内容がどこで使われるか
A: `src/codegen/generate-classes.ts` によってC++とZigのバインディングコードが自動生成されます。

#### Q: テストの実行方法がわからない
A: 必ず `bun bd test <file>` を使用してください。`bun test` では変更が反映されません。

### デバッグのヒント

1. **デバッグビルド**: `bun bd` でデバッグビルドを使用
2. **ログ出力**: `BUN_DEBUG_QUIET_LOGS=1` で余計なログを抑制
3. **特定スコープのログ**: `BUN_DEBUG_<scopeName>=1` で特定のログを有効化
4. **JSC例外チェック**: `BUN_JSC_validateExceptionChecks=1` で例外スコープを検証

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| JSC | JavaScriptCore - WebKitのJavaScriptエンジン |
| Zig | Bunのコアを構成するシステムプログラミング言語 |
| ビルトイン | 組み込みモジュール（node:fs など） |
| バインディング | 異なる言語間の橋渡しコード |
| トランスパイラ | ソースコード変換器 |
| バンドラー | 複数ファイルを1つにまとめるツール |

### B. 主要ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/bun.zig` | Zigモジュールルート | 基本型定義、ユーティリティ |
| `src/cli.zig` | CLIエントリー | コマンドディスパッチ |
| `src/bun.js/api/server.zig` | HTTPサーバー | Webサーバー実装 |
| `src/install/PackageManager.zig` | パッケージマネージャ | 依存関係管理 |
| `src/js_parser.zig` | JSパーサー | JavaScript/TypeScript解析 |
| `test/harness.ts` | テストハーネス | テストユーティリティ |

### C. 参考資料

- [Zig公式ドキュメント](https://ziglang.org/documentation/)
- [JavaScriptCore API](https://developer.apple.com/documentation/javascriptcore)
- [Bun公式ドキュメント](https://bun.sh/docs)
- [WebKit JavaScriptCore](https://trac.webkit.org/wiki/JavaScriptCore)
