# セキュリティ設計書

## 概要

本ドキュメントは、Bun JavaScript ランタイムのセキュリティ設計について記述したものです。Bunは、JavaScriptCore エンジンをベースとした高性能なJavaScript/TypeScriptランタイムであり、パッケージマネージャー、バンドラー、テストランナーを統合した開発ツールキットです。本設計書では、暗号化機能、TLS/SSL通信、パスワードハッシュ、入力検証、シークレット管理などのセキュリティ機能について説明します。

## 認証設計

### 認証方式

Bunはランタイム環境として、アプリケーションレベルの認証機能を直接提供するのではなく、開発者が認証システムを構築するための暗号化プリミティブとAPIを提供します。

| 機能 | 説明 | 実装ファイル |
| --- | --- | --- |
| パスワードハッシュ | Argon2id/Argon2i/Argon2d/bcrypt対応 | `src/bun.js/api/crypto/PasswordObject.zig` |
| HMAC認証 | メッセージ認証コード生成 | `src/bun.js/api/crypto/HMAC.zig` |
| 署名検証 | 各種署名アルゴリズム対応 | `src/bun.js/bindings/ncrypto.cpp` |

### セッション管理

Bunはセッション管理を直接提供せず、ユーザーアプリケーション層での実装に委ねています。ただし、安全なセッション管理に必要な暗号化プリミティブを提供します。

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッションID生成 | `crypto.randomUUID()`/`crypto.randomUUIDv7()` | 暗号学的に安全な乱数生成 |
| タイミング攻撃対策 | `crypto.timingSafeEqual()` | 定数時間比較関数 |

## 認可設計

### 権限体系

Bunランタイムは、OSレベルのファイルシステムおよびネットワーク権限に依存します。特別な権限システムは実装されておらず、実行ユーザーの権限で動作します。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 実行ユーザー | OS権限に依存 | ファイルシステム、ネットワークアクセス |
| シークレットアクセス | キーチェーン権限 | macOSキーチェーンAPI経由 |

### アクセス制御

Bunは以下のアクセス制御メカニズムを提供します。

1. **シークレット管理**: `Bun.secrets` APIを通じて、OSのキーチェーン/資格情報ストアへの安全なアクセスを提供
   - 実装: `src/bun.js/bindings/JSSecrets.cpp`
   - エラーコード: `ERR_SECRETS_ACCESS_DENIED`, `ERR_SECRETS_AUTH_FAILED` 等

2. **TLS証明書検証**: サーバー/クライアント証明書の検証オプション
   - `request_cert`: 証明書要求フラグ
   - `reject_unauthorized`: 未検証証明書の拒否

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | BoringSSL (OpenSSL互換ライブラリ) によるTLS実装 |
| TLSバージョン | TLS 1.2/1.3 サポート |
| 証明書管理 | システムCA証明書、バンドルCA証明書、カスタムCA証明書のサポート |
| 暗号スイート | カスタマイズ可能な暗号スイート設定 |

### TLS/SSL設定

TLS設定は `SSLConfig` 構造体で管理されます。

| 設定項目 | 説明 | デフォルト値 |
| --- | --- | --- |
| `key_file_name` | 秘密鍵ファイルパス | なし |
| `cert_file_name` | 証明書ファイルパス | なし |
| `ca_file_name` | CA証明書ファイルパス | システムCA |
| `passphrase` | 秘密鍵のパスフレーズ | なし |
| `ssl_ciphers` | 暗号スイート | デフォルト |
| `reject_unauthorized` | 証明書検証の厳格モード | 0 (無効) |
| `request_cert` | クライアント証明書要求 | 0 (無効) |
| `low_memory_mode` | 低メモリモード | false |

実装ファイル: `src/bun.js/api/server/SSLConfig.zig`

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS 1.2/1.3 (BoringSSL) |
| ハッシュ関数 | MD4, MD5, SHA1, SHA224, SHA256, SHA384, SHA512, SHA512-224, SHA512-256, SHA3-224, SHA3-256, SHA3-384, SHA3-512, SHAKE128, SHAKE256, BLAKE2b256, BLAKE2b512, BLAKE2s256, RIPEMD160 |
| HMAC | 上記ハッシュ関数を使用したHMAC |
| パスワードハッシュ | Argon2id (推奨), Argon2i, Argon2d, bcrypt |

### 暗号化アルゴリズム実装

**ハッシュ関数** (`src/bun.js/api/crypto/EVP.zig`):

```
対応アルゴリズム:
- blake2b256, blake2b512, blake2s256
- md4, md5
- ripemd160
- sha1, sha224, sha256, sha384, sha512
- sha512-224, sha512-256
- sha3-224, sha3-256, sha3-384, sha3-512
- shake128, shake256
```

**HMAC** (`src/bun.js/api/crypto/HMAC.zig`):
- BoringSSL HMAC_CTX を使用
- EVPアルゴリズムと組み合わせて使用

**パスワードハッシュ** (`src/bun.js/api/crypto/PasswordObject.zig`):

| アルゴリズム | パラメータ | デフォルト値 |
| --- | --- | --- |
| Argon2id | `timeCost`, `memoryCost` | interactive_2id パラメータ |
| Argon2i | `timeCost`, `memoryCost` | interactive_2id パラメータ |
| Argon2d | `timeCost`, `memoryCost` | interactive_2id パラメータ |
| bcrypt | `cost` | 10 (4-31の範囲) |

### 機密情報管理

1. **シークレットストレージ** (`src/bun.js/bindings/JSSecrets.cpp`):
   - macOS: キーチェーンAPI経由でのシークレット管理
   - セキュリティエラーの適切なハンドリング (アクセス拒否、認証失敗、ユーザーキャンセル等)
   - メモリ上のシークレットは使用後にゼロクリア

2. **センシティブデータの解放** (`src/bun.js/api/server/SSLConfig.zig`):
   - `bun.freeSensitive()` によるセキュアなメモリ解放
   - パスフレーズ、秘密鍵等の機密データのゼロ化

3. **乱数生成** (`src/bun.js/webcore/Crypto.zig`):
   - `crypto.getRandomValues()`: TypedArrayへの暗号学的乱数充填
   - `crypto.randomUUID()`: UUID v4生成
   - `crypto.randomUUIDv7()`: タイムスタンプベースUUID v7生成
   - 512バイト以下はキャッシュから、それ以上は `bun.csprng()` を使用

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | アプリケーション層での対策が必要 (Bunは特別な対策を提供しない) |
| SQLインジェクション | プリペアドステートメントのサポート (`bun:sqlite`) |
| CSRF | アプリケーション層での対策が必要 |
| パストラバーサル | ファイルシステムAPIでの検証 |
| 入力値検証 | NodeValidator による型・範囲検証 |

### 入力検証機能

`NodeValidator.cpp` で提供される検証関数:

| 関数 | 説明 |
| --- | --- |
| `validateInteger()` | 整数値の範囲検証 |
| `validateNumber()` | 数値の範囲検証 |
| `validateString()` | 文字列型検証 |

これらの関数は Node.js 互換APIの入力検証に使用され、適切なエラーコード (`ERR_INVALID_ARG_TYPE`, `ERR_OUT_OF_RANGE`) を返します。

### CORS設定

`src/http/FetchRequestMode.zig` で定義されるFetchリクエストモード:

| モード | 説明 |
| --- | --- |
| `same-origin` | 同一オリジンのみ |
| `no-cors` | CORS制限なし |
| `cors` | CORS有効 |
| `navigate` | ナビゲーション |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| セキュリティスキャン結果 | パッケージ脆弱性、アドバイザリ | 実行時のみ |
| TLSエラー | 証明書検証エラー、ハンドシェイクエラー | 実行時のみ |

### セキュリティスキャナー

`src/install/PackageManager/security_scanner.zig` で実装されるパッケージセキュリティスキャン機能:

| 項目 | 説明 |
| --- | --- |
| `SecurityAdvisoryLevel` | `fatal` (致命的), `warn` (警告) |
| `SecurityScanResults` | スキャン結果、検出件数、スキャン時間 |

スキャン結果には以下の情報が含まれます:
- パッケージ名
- 脆弱性URL
- 説明
- 影響を受けるパッケージパス

## TLSソケット機能

`src/bun.js/api/bun/socket/tls_socket_functions.zig` で提供されるTLSソケット関連機能:

| 関数 | 説明 |
| --- | --- |
| `getServername()` | SNIサーバー名の取得 |
| `setServername()` | SNIサーバー名の設定 |
| `getPeerCertificate()` | ピア証明書の取得 |
| `getPeerX509Certificate()` | X509形式ピア証明書の取得 |
| `getX509Certificate()` | 自己証明書の取得 |
| `getTLSVersion()` | TLSバージョンの取得 |
| `setMaxSendFragment()` | 最大送信フラグメントサイズの設定 |
| `getSharedSigalgs()` | 共有署名アルゴリズムの取得 |
| `getTLSFinishedMessage()` | TLS Finishedメッセージの取得 |

## X509証明書管理

`src/bun.js/bindings/NodeTLS.cpp` で提供されるCA証明書管理:

| 関数 | 説明 |
| --- | --- |
| `getBundledRootCertificates()` | バンドルされたルートCA証明書の取得 |
| `getExtraCACertificates()` | 追加CA証明書の取得 |
| `getSystemCACertificates()` | システムCA証明書の取得 |
| `getDefaultCiphers()` | デフォルト暗号スイートの取得 |
| `setDefaultCiphers()` | デフォルト暗号スイートの設定 |

## 備考

1. **BoringSSL**: BunはOpenSSLのフォークであるBoringSSLを使用しており、Googleが維持管理する安定したTLS/暗号化実装を提供します。

2. **Node.js互換性**: BunはNode.js cryptoモジュールとの高い互換性を維持しており、既存のNode.jsアプリケーションのセキュリティコードの多くがそのまま動作します。

3. **WebCrypto API**: Web標準のSubtleCrypto APIをサポートしており、ブラウザ互換のコードを実行可能です。

4. **メモリセキュリティ**: Zigのメモリ安全性機能を活用し、バッファオーバーフローやuse-after-freeなどの脆弱性を防止しています。

5. **タイミング攻撃対策**: `timingSafeEqual()` 関数により、比較操作に対するタイミング攻撃を防止する定数時間比較が可能です。

6. **シークレット管理**: プラットフォーム固有のセキュアストレージ (macOSキーチェーン等) を活用したシークレット管理APIを提供し、機密情報をメモリ上に長時間保持しないよう設計されています。
