# 機能設計書 103-node:process

## 概要

本ドキュメントは、BunにおけるNode.js互換の`node:process`モジュールおよび`globalThis.process`オブジェクトの機能設計を記述したものである。このモジュールは、現在のプロセスに関する情報取得と制御を提供する。

### 本機能の処理概要

`process`オブジェクトは、Node.jsアプリケーションの実行環境に関する情報と制御機能を提供する。環境変数、コマンドライン引数、標準入出力、プロセスの終了制御、システム情報など、プロセス全体に関わる機能を集約している。

**業務上の目的・背景**：アプリケーションが動作する環境情報の取得や、プロセスのライフサイクル管理は、多くのNode.jsアプリケーションで必要とされる基本機能である。特に、環境変数による設定管理、コマンドライン引数の処理、シグナルハンドリングなどは、サーバーアプリケーションやCLIツールで頻繁に使用される。BunではNode.js互換のprocess APIを提供し、既存アプリケーションの移行を支援する。

**機能の利用シーン**：
- 環境変数の読み取り・設定（`process.env`）
- コマンドライン引数の取得（`process.argv`）
- プロセスの終了（`process.exit()`）
- 作業ディレクトリの取得・変更（`process.cwd()`, `process.chdir()`）
- メモリ使用量の確認（`process.memoryUsage()`）
- 標準入出力の操作（`process.stdin`, `process.stdout`, `process.stderr`）
- イベントリスナーの登録（`process.on('exit')`, `process.on('SIGINT')`等）

**主要な処理内容**：
1. 環境情報の提供（`arch`, `platform`, `version`, `versions`）
2. 環境変数へのアクセス（`process.env`）
3. コマンドライン引数の取得（`argv`, `execArgv`）
4. 作業ディレクトリの操作（`cwd()`, `chdir()`）
5. プロセス終了制御（`exit()`, `exitCode`）
6. リソース使用量の取得（`memoryUsage()`, `cpuUsage()`, `resourceUsage()`）
7. イベントエミッタとしての機能（`on`, `emit`, `once`等）

**関連システム・外部連携**：
- オペレーティングシステムのプロセス管理API
- 環境変数（シェル/OS提供）
- シグナルハンドリング（POSIX）

**権限による制御**：
- `setuid()`, `setgid()`等の権限変更関数はPOSIX環境でのみ利用可能
- 一部の操作には適切な権限が必要

## 関連画面

本機能はバックエンドモジュールであり、直接的な関連画面は存在しない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | - |

## 機能種別

プロセス管理 / 環境情報取得 / イベントエミッタ

## 入力仕様

### 入力パラメータ

#### `process.exit([code])`

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| code | number | No | 終了コード（デフォルト: 0） | 0-255の整数 |

#### `process.chdir(directory)`

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| directory | string | Yes | 新しい作業ディレクトリパス | 非空文字列 |

#### `process.kill(pid, [signal])`

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pid | number | Yes | プロセスID | 正の整数 |
| signal | string \| number | No | シグナル（デフォルト: 'SIGTERM'） | 有効なシグナル名/番号 |

#### `process.setTitle(title)`

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| title | string | Yes | 新しいプロセスタイトル | - |

### 入力データソース

- 関数呼び出しの引数
- `process.env`: オペレーティングシステムの環境変数
- `process.argv`: コマンドライン引数（Bunランタイムから提供）

## 出力仕様

### 出力データ

#### 主要なプロパティ

| プロパティ名 | 型 | 説明 |
|-------------|-----|------|
| arch | string | CPUアーキテクチャ（'x64', 'arm64'等） |
| platform | string | プラットフォーム（'darwin', 'linux', 'win32'） |
| version | string | Node.js互換バージョン文字列 |
| versions | object | 各依存ライブラリのバージョン |
| pid | number | プロセスID |
| ppid | number | 親プロセスID |
| env | object | 環境変数のオブジェクト |
| argv | string[] | コマンドライン引数配列 |
| execArgv | string[] | Bun/Node固有の実行引数 |
| execPath | string | Bunの実行ファイルパス |
| cwd() | string | 現在の作業ディレクトリ |
| title | string | プロセスタイトル |

#### `process.memoryUsage()`の戻り値

| 項目名 | 型 | 説明 |
|--------|-----|------|
| rss | number | 常駐セットサイズ（バイト） |
| heapTotal | number | V8ヒープ総サイズ |
| heapUsed | number | V8ヒープ使用量 |
| external | number | 外部メモリ使用量 |
| arrayBuffers | number | ArrayBufferのメモリ使用量 |

#### `process.cpuUsage([previousValue])`の戻り値

| 項目名 | 型 | 説明 |
|--------|-----|------|
| user | number | ユーザーCPU時間（マイクロ秒） |
| system | number | システムCPU時間（マイクロ秒） |

### 出力先

- 関数の戻り値
- プロパティへの直接アクセス
- イベントリスナーへのコールバック

## 処理フロー

### 処理シーケンス

```
1. プロセス初期化
   └─ Zigネイティブコードでprocessオブジェクトを初期化
2. プロパティアクセス
   └─ 遅延評価でプロパティを生成
   └─ キャッシュ済みの場合は即座に返却
3. メソッド呼び出し
   └─ 引数の検証
   └─ ネイティブAPIの呼び出し
   └─ 結果の変換・返却
4. イベントエミッタ機能
   └─ EventEmitterを継承
   └─ 'exit', 'beforeExit', シグナル等のイベント
```

### フローチャート

```mermaid
flowchart TD
    A[process参照] --> B{初期化済み?}
    B -->|No| C[globalThis.processを初期化]
    B -->|Yes| D[プロパティ/メソッドアクセス]
    C --> D
    D --> E{アクセス種別}
    E -->|プロパティ読み取り| F[ゲッター呼び出し]
    E -->|メソッド呼び出し| G[引数検証]
    F --> H[値を返却]
    G --> I[ネイティブAPI呼び出し]
    I --> J{成功?}
    J -->|Yes| K[結果を返却]
    J -->|No| L[エラーをスロー]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-301 | argv構成 | argv[0]=bunパス、argv[1]=スクリプトパス、以降=ユーザー引数 | 常時 |
| BR-302 | 終了コード範囲 | exitCodeは0-255の範囲 | process.exit()呼び出し時 |
| BR-303 | プラットフォーム文字列 | darwin/linux/win32のいずれか | process.platform |
| BR-304 | アーキテクチャ文字列 | x64/arm64のいずれか | process.arch |
| BR-305 | タイトル設定 | 設定時にmutexでスレッドセーフに処理 | process.title設定時 |
| BR-306 | NODE_NO_WARNINGS | 警告を抑制 | 環境変数設定時 |

### 計算ロジック

**メモリ使用量計算**：
- JavaScriptCoreのヒープ情報を取得
- OS固有のAPIでRSSを取得（macOS: mach_task_info、Linux: /proc/self/statm）

**CPU使用量計算**：
- getrusage()システムコールを使用
- previousValueとの差分を計算可能

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INVALID_ARG_TYPE | TypeError | 引数の型が不正 | 正しい型を指定 |
| ERR_INVALID_ARG_VALUE | RangeError | 引数の値が不正 | 有効な値を指定 |
| ENOENT | SystemError | ディレクトリが存在しない | パスを確認 |
| EACCES | SystemError | 権限不足 | 適切な権限で実行 |
| ESRCH | SystemError | プロセスが存在しない | PIDを確認 |
| EPERM | SystemError | 操作が許可されていない | 権限を確認 |

### リトライ仕様

本機能はリトライを行わない。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- `process.env`へのアクセスは初回アクセス時に環境変数をキャッシュ
- `process.memoryUsage()`はシステムコールを伴うため、頻繁な呼び出しは避ける
- タイトル設定はmutexを使用するため、高頻度での変更は避ける

## セキュリティ考慮事項

- `process.env`には機密情報が含まれる可能性があるため、ログ出力時は注意
- `setuid()`, `setgid()`は特権操作であり、慎重に使用
- `process.kill()`で他プロセスへのシグナル送信は権限が必要

## 備考

- `process`はEventEmitterを継承しており、イベント機能を使用可能
- Workerスレッドでは一部のプロパティが異なる値を返す
- スタンドアロン実行ファイル（`bun build --compile`）では`execArgv`の構成が異なる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、processオブジェクトの基本構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BunProcess.cpp | `src/bun.js/bindings/BunProcess.cpp` | C++でのprocessオブジェクト定義 |

**読解のコツ**:
- **169-178行目**: `constructArch`でアーキテクチャ文字列を生成
- **180-191行目**: `constructPlatform`でプラットフォーム文字列を生成
- **193-200行目**: `constructVersions`でversionsオブジェクトを構築

#### Step 2: Zigネイティブ実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node_process.zig | `src/bun.js/node/node_process.zig` | プロセス操作のZig実装 |

**主要処理フロー**:
1. **1-13行目**: C++からの関数エクスポート定義
2. **17-37行目**: `getTitle`/`setTitle`でプロセスタイトルの取得・設定
3. **39-50行目**: `createArgv0`/`getExecPath`でargv[0]と実行パスを取得
4. **52-159行目**: `createExecArgv`でexecArgv配列を構築
5. **161-223行目**: `createArgv`でargv配列を構築
6. **243-296行目**: `getCwd`/`setCwd`で作業ディレクトリ操作
7. **299-310行目**: `exit`でプロセス終了処理

#### Step 3: プロセス管理の低レベル実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | process.zig | `src/bun.js/api/bun/process.zig` | プロセスの低レベル管理 |

**主要処理フロー**:
- **1-28行目**: `win_rusage`構造体でWindowsのリソース使用量情報を定義
- **41-71行目**: `uv_getrusage`でWindowsプロセスのリソース情報を取得
- **75-134行目**: `ProcessExitHandler`でプロセス終了時のコールバック管理
- **137-200行目**: `Process`構造体でプロセス状態を管理

#### Step 4: 環境変数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | JSEnvironmentVariableMap.cpp | `src/bun.js/bindings/JSEnvironmentVariableMap.cpp` | process.envの実装 |

**読解のコツ**:
- `process.env`はProxyオブジェクトとして実装
- 読み取り時はOS環境変数から取得
- 書き込み時はOSの環境変数も更新（Windows: SetEnvironmentVariableW）

### プログラム呼び出し階層図

```
JavaScript
    │
    ├─ process.cwd()
    │      └─ node_process.zig: getCwd()
    │             └─ bun.api.node.path.getCwd()
    │
    ├─ process.chdir(dir)
    │      └─ node_process.zig: setCwd()
    │             └─ Syscall.chdir()
    │
    ├─ process.exit(code)
    │      └─ node_process.zig: exit()
    │             └─ vm.onExit() / vm.globalExit()
    │
    ├─ process.argv
    │      └─ node_process.zig: createArgv()
    │             └─ bun.argv からJS配列を構築
    │
    ├─ process.env
    │      └─ JSEnvironmentVariableMap (C++)
    │             └─ getenv() / setenv()
    │
    └─ process.memoryUsage()
           └─ BunProcess.cpp
                  └─ JavaScriptCore heap info + OS API
```

### データフロー図

```
[システム]                  [処理]                        [出力]

環境変数 ─────────────────▶ process.env ────────────────▶ { KEY: "value" }
                                 │
コマンドライン引数 ──────────▶ createArgv() ──────────────▶ ["bun", "script.js", ...]
                                 │
OS プロセス情報 ─────────────▶ getpid(), getcwd() ────────▶ pid, cwd
                                 │
メモリ情報 ──────────────────▶ memoryUsage() ─────────────▶ { rss, heapTotal, ... }
                                 │
CPU情報 ─────────────────────▶ cpuUsage() ────────────────▶ { user, system }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| BunProcess.cpp | `src/bun.js/bindings/BunProcess.cpp` | ソース | processオブジェクトのC++実装 |
| BunProcess.h | `src/bun.js/bindings/BunProcess.h` | ヘッダー | processオブジェクトの宣言 |
| node_process.zig | `src/bun.js/node/node_process.zig` | ソース | processのZig実装 |
| process.zig | `src/bun.js/api/bun/process.zig` | ソース | プロセス管理の低レベル実装 |
| JSEnvironmentVariableMap.cpp | `src/bun.js/bindings/JSEnvironmentVariableMap.cpp` | ソース | process.envの実装 |
| JSNextTickQueue.h | `src/bun.js/bindings/JSNextTickQueue.h` | ヘッダー | process.nextTickの実装 |
