# 機能設計書 113-クラッシュレポート

## 概要

本ドキュメントは、Bunのクラッシュハンドラーおよびクラッシュレポート機能の設計を記述する。プログラムクラッシュ時にスタックトレースを収集し、問題解決に役立つ情報を生成する。

### 本機能の処理概要

Bunのクラッシュハンドラーは、panic、セグメンテーションフォルト、不正命令などのクラッシュイベントを検出し、スタックトレースを含む診断情報を生成する。デバッグビルドではソースコードにマッピングされたバックトレースを表示し、リリースビルドでは「trace string」と呼ばれる圧縮エンコードされたURLを生成して、bun.reportサービスでのリモートデマングルを可能にする。

**業務上の目的・背景**：ランタイムのクラッシュは開発者にとって深刻な問題である。Bunは高性能を追求するため、Zig言語で実装されており、メモリ安全性に関する問題が発生する可能性がある。クラッシュ時に詳細な診断情報を提供することで、問題の迅速な特定と修正を支援する。リリースビルドではデバッグシンボルを含まないため、trace stringを通じてbun.reportサービスにキャッシュされたシンボル情報を活用する。

**機能の利用シーン**：
- @panic()によるプログラム異常終了時
- セグメンテーションフォルト発生時
- 不正命令実行時
- バスエラー発生時（POSIX）
- 浮動小数点エラー発生時（POSIX）
- スタックオーバーフロー発生時（Windows）
- メモリ不足発生時
- ネイティブプラグインクラッシュ時

**主要な処理内容**：
1. クラッシュ理由（CrashReason）の分類と記録
2. スタックトレースの収集
3. 現在実行中のアクション（Action）の記録
4. trace stringの生成（Bunバージョン、プラットフォーム、アドレス情報を含む）
5. bun.reportへのレポートURL生成
6. デバッグビルドでの即時スタックトレース表示
7. 自動リスタート（--auto-reload時）の制御

**関連システム・外部連携**：
- bun.report（リモートシンボル解決サービス）
- Zig標準ライブラリのdebug機能
- OS固有のシグナルハンドラー

**権限による制御**：
- BUN_ENABLE_CRASH_REPORTING環境変数でレポート送信制御
- BUN_CRASH_REPORT_URL環境変数でレポートURL変更可能

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はランタイム内部機能であり、コンソール出力を行う |

## 機能種別

エラーハンドリング / 診断情報生成 / レポート送信

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| reason | CrashReason | Yes | クラッシュの理由 | 有効なCrashReason |
| error_return_trace | ?*std.builtin.StackTrace | No | エラーリターントレース | - |
| begin_addr | ?usize | No | バックトレース開始アドレス | - |

### 入力データソース

- OSシグナル（SIGSEGV, SIGILL, SIGBUS, SIGFPE等）
- @panic()呼び出し
- エラーリターントレース

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| trace_string | []const u8 | 圧縮エンコードされたクラッシュ情報URL |
| stack_trace | - | コンソール出力されるスタックトレース |
| crash_reason | - | クラッシュ理由のフォーマット済み文字列 |
| action_context | - | クラッシュ時のアクションコンテキスト |

### 出力先

- 標準エラー出力（stderr）
- bun.report（HTTPリクエスト、オプション）

## 処理フロー

### 処理シーケンス

```
1. クラッシュ検出
   └─ シグナルハンドラーまたは@panic()から呼び出し

2. panicステージ管理
   └─ panic_stageを0→1に遷移
   └─ panickingカウンタをインクリメント

3. クラッシュ前ハンドラー実行
   └─ before_crash_handlersに登録されたコールバック実行

4. メタデータ収集
   └─ Bunバージョン、プラットフォーム情報
   └─ current_action（実行中の処理）

5. スタックトレース収集
   └─ error_return_traceまたはcaptureStackTrace
   └─ Linux glibcではbacktrace()も使用

6. 出力生成
   └─ デバッグビルド: 即時スタックトレース表示
   └─ リリースビルド: trace string URL生成

7. 他スレッドのパニック待機
   └─ waitForOtherThreadToFinishPanicking

8. レポート送信（オプション）
   └─ report関数によるHTTPリクエスト

9. 後処理
   └─ セグフォルトハンドラーリセット
   └─ 自動リスタート（設定時）
   └─ プロセス終了
```

### フローチャート

```mermaid
flowchart TD
    A[クラッシュ発生] --> B{panic_stage確認}
    B -->|0| C[panic_stage=1]
    B -->|1| D[panic_stage=2, リセット]
    B -->|2| E[panic_stage=3, abort]
    C --> F[before_crash_handlers実行]
    F --> G[メタデータ収集]
    G --> H{デバッグビルド?}
    H -->|Yes| I[即時スタックトレース表示]
    H -->|No| J{メッセージ出力済み?}
    J -->|No| K[クラッシュメッセージ出力]
    J -->|Yes| L[追加情報出力]
    I --> M[trace string生成]
    K --> M
    L --> M
    M --> N[他スレッド待機]
    N --> O{レポート有効?}
    O -->|Yes| P[HTTPレポート送信]
    O -->|No| Q[スキップ]
    P --> R[セグフォルトハンドラーリセット]
    Q --> R
    R --> S{自動リスタート?}
    S -->|Yes| T[プロセスリロード]
    S -->|No| U[abort]
    D --> V[ハンドラーリセット]
    V --> E
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | クラッシュ理由分類 | panic, unreachable, segfault, illegal_instruction等を分類 | 全クラッシュ |
| BR-002 | アクション記録 | parse, visit, print, bundle_generate_chunk等の実行中アクションを記録 | current_action設定時 |
| BR-003 | ネイティブプラグイン識別 | inside_native_pluginがセットされている場合は専用メッセージ | プラグイン実行中 |
| BR-004 | libuv関数クラッシュ識別 | unsupported_uv_functionがセットされている場合は専用メッセージ | NAPI実行中 |
| BR-005 | レポート抑制 | suppress_reporting=trueの場合はレポート送信をスキップ | suppressReporting()呼び出し時 |
| BR-006 | 複数スレッドパニック | 最初のスレッドのみがメッセージを出力、他は待機 | マルチスレッド環境 |

### 計算ロジック

trace stringのエンコード形式：
- Bunバージョンとコミット情報
- プラットフォーム識別子
- ASLRを除去したアドレスリスト
- パニックメッセージ（該当時）

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 二重パニック | パニックハンドラー内でパニック発生 | panic_stage進行、最終的にabort |
| - | メモリ不足 | スタックトレース収集中のメモリ不足 | スタックフォールバックバッファ使用 |
| - | レポート送信失敗 | HTTPリクエスト失敗 | 無視して処理継続 |

### リトライ仕様

リトライは行わない。クラッシュハンドラーは一度きりの処理。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- クラッシュハンドラーは可能な限り高速に実行
- スタックトレース収集は最大20フレーム
- 二重パニック時は即座にabort

## セキュリティ考慮事項

- trace stringにはソースコード情報は含まれない（アドレスのみ）
- BUN_ENABLE_CRASH_REPORTINGでレポート送信を制御可能
- suppressReporting()でプログラム的にレポート抑制可能

## 備考

- bun.reportはオープンソース（https://github.com/oven-sh/bun.report）
- デバッグビルドでは--debug-crash-handler-use-trace-stringでリリースビルド動作をテスト可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

クラッシュ理由とアクションの分類を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | crash_handler.zig | `src/crash_handler.zig` | CrashReason、Action union定義 |

**読解のコツ**: Zigのunion(enum)は判別共用体であり、各バリアントがクラッシュの種類を表す。

**主要処理フロー**:
- **66-103行目**: CrashReason union - クラッシュ理由の分類（panic, unreachable, segfault等）
- **106-164行目**: Action union - 実行中アクションの種類（parse, visit, print等）
- **30-58行目**: グローバル状態変数（panic_stage, panicking, current_action等）

#### Step 2: エントリーポイントを理解する

クラッシュハンドラーのメイン処理を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | crash_handler.zig | `src/crash_handler.zig` | crashHandler関数 |

**主要処理フロー**:
- **198-500行目**: crashHandler - メインクラッシュハンドラー
  - panic_stageによる二重パニック制御
  - before_crash_handlersの実行
  - メッセージ出力とスタックトレース収集
  - trace string生成
  - 自動リスタート制御

#### Step 3: スタックトレース収集を理解する

バックトレースの収集方法を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | crash_handler.zig | `src/crash_handler.zig` | captureLibcBacktrace、スタックトレース関連処理 |

**主要処理フロー**:
- **166-195行目**: captureLibcBacktrace - Linux glibcのbacktrace()使用
- **339-368行目**: スタックトレース収集ロジック（error_return_traceまたはcaptureStackTrace）

#### Step 4: trace string生成を理解する

リリースビルドでのURL生成を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | crash_handler.zig | `src/crash_handler.zig` | TraceString構造体（ファイル後半） |

**主要処理フロー**:
- **375-379行目**: trace_str_buf生成（デバッグビルド用）
- **440-444行目**: trace_str_buf生成（リリースビルド用）
- TraceString構造体はbun.reportのデコーダーと同期が必要

#### Step 5: レポート送信を理解する

HTTPによるレポート送信処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | crash_handler.zig | `src/crash_handler.zig` | report関数、isReportingEnabled |

**主要処理フロー**:
- **462行目**: report(trace_str_buf.slice())呼び出し
- **26-27行目**: default_report_base_url定義
- **63行目**: suppress_reporting変数

### プログラム呼び出し階層図

```
[シグナルハンドラー / @panic()]
    │
    └─ crashHandler
           │
           ├─ panic_stage管理
           │      └─ 0→1→(2→3→abort)
           │
           ├─ before_crash_handlers実行
           │
           ├─ メタデータ収集
           │      ├─ printMetadata
           │      └─ current_action出力
           │
           ├─ スタックトレース収集
           │      ├─ error_return_trace使用
           │      ├─ std.debug.captureStackTrace
           │      └─ captureLibcBacktrace (Linux)
           │
           ├─ 出力生成
           │      ├─ dumpStackTrace (デバッグ)
           │      └─ TraceString生成 (リリース)
           │
           ├─ waitForOtherThreadToFinishPanicking
           │
           ├─ report (オプション)
           │
           ├─ resetSegfaultHandler
           │
           └─ bun.reloadProcess (自動リスタート時)
                  または abort
```

### データフロー図

```
[入力]                        [処理]                           [出力]

シグナル           ───▶  crashHandler         ───▶  stderr出力
  │                         │
  ├─ SIGSEGV               ├─ CrashReason分類
  ├─ SIGILL                ├─ Action記録
  ├─ SIGBUS                ├─ スタックトレース収集
  └─ @panic()              └─ trace string生成
                                   │
                                   ▼
                            ┌─────────────────┐
                            │ 出力データ      │
                            │ ├─ エラーメッセージ
                            │ ├─ スタックトレース
                            │ └─ trace string URL
                            └─────────────────┘
                                   │
                                   ▼
                            [bun.report] (オプション)
                                   │
                                   └─ シンボル解決済みレポート
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| crash_handler.zig | `src/crash_handler.zig` | ソース | クラッシュハンドラーメイン実装 |
| Output.zig | `src/Output.zig` | ソース | 出力ユーティリティ |
| bun.zig | `src/bun.zig` | ソース | 共通定義（reloadProcess等） |
| Futex.zig | `src/Futex.zig` | ソース | スレッド同期 |
| std.debug | Zig標準ライブラリ | 外部 | スタックトレース収集 |
