# 機能設計書 115-メモリアロケータ

## 概要

本ドキュメントは、Bunのメモリアロケーション機能の設計を記述する。高速で効率的なメモリ管理を実現するためのカスタムアロケータ群を提供する。

### 本機能の処理概要

Bunは高性能なJavaScriptランタイムとして、メモリ割り当てのパフォーマンスが全体の速度に大きく影響する。標準のmallocの代わりに、mimalloc（Microsoft製の高速アロケータ）をベースにしたカスタムアロケータ群を実装し、様々なユースケースに最適化されたメモリ管理を提供する。

**業務上の目的・背景**：JavaScriptランタイムは大量のメモリ割り当て・解放を行うため、アロケータの性能がボトルネックになりやすい。Bunはmimallocを採用し、さらにZig言語の型安全なアロケータインターフェースでラップすることで、高速かつ安全なメモリ管理を実現する。スレッドローカルヒープを活用することで、マルチスレッド環境でのロック競合を最小化する。

**機能の利用シーン**：
- JavaScriptオブジェクトのメモリ割り当て
- パーサー・バンドラーの一時バッファ
- HTTPリクエスト/レスポンスのバッファ
- ファイルI/Oバッファ
- スレッドプール内のワーカーでのメモリ割り当て

**主要な処理内容**：
1. mimallocのZigバインディング提供
2. MimallocArenaによるスレッドローカルヒープ管理
3. Zig標準Allocatorインターフェースへの適合
4. デバッグビルドでのスレッド所有権検証
5. メモリ統計情報の出力

**関連システム・外部連携**：
- mimalloc（外部ライブラリ）
- Zig標準ライブラリのAllocatorインターフェース
- JavaScriptCoreのメモリ管理

**権限による制御**：特になし（ランタイム内部機能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はランタイム内部機能であり、直接的な画面は存在しない |

## 機能種別

メモリ管理 / アロケーション / パフォーマンス最適化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| size | usize | Yes | 割り当てサイズ | - |
| alignment | std.mem.Alignment | No | アライメント要件 | 16バイト以下または明示的指定 |
| count | usize | No | 要素数（calloc用） | - |

### 入力データソース

- Zig/C++コードからの割り当て要求

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ptr | ?*anyopaque | 割り当てられたメモリへのポインタ（失敗時null） |
| usable_size | usize | 実際に使用可能なサイズ |

### 出力先

- メモリアドレス

## 処理フロー

### 処理シーケンス

```
1. MimallocArena初期化（init）
   └─ mi_heap_new()で新規ヒープ作成
   └─ スレッドID記録（デバッグビルド）

2. メモリ割り当て（allocator）
   └─ スレッド所有権検証（デバッグビルド）
   └─ アライメント判定
   └─ mi_heap_malloc / mi_heap_malloc_aligned呼び出し

3. メモリ解放（free）
   └─ mi_free / mi_free_size / mi_free_size_aligned呼び出し

4. リサイズ（resize/remap）
   └─ mi_expand / mi_heap_realloc_aligned呼び出し

5. ヒープ解放（deinit）
   └─ mi_heap_destroy呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[割り当て要求] --> B{アライメント > 16?}
    B -->|Yes| C[mi_heap_malloc_aligned]
    B -->|No| D[mi_heap_malloc]
    C --> E{成功?}
    D --> E
    E -->|Yes| F[ポインタ返却]
    E -->|No| G[null返却]

    H[解放要求] --> I{デバッグビルド?}
    I -->|Yes| J[mi_is_in_heap_region検証]
    I -->|No| K[mi_free]
    J --> K

    L[リサイズ要求] --> M[mi_expand]
    M --> N{成功?}
    N -->|Yes| O[true返却]
    N -->|No| P[false返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | アライメント閾値 | 16バイト以下はデフォルトアライメント使用 | mustUseAlignedAlloc |
| BR-002 | スレッド所有権 | ヒープは作成スレッドでのみ使用可能 | デバッグビルド |
| BR-003 | Borrowedパターン | 借用アロケータで所有権を明確化 | MimallocArena.borrow() |
| BR-004 | デフォルトヒープ | mi_heap_main()でスレッドローカルデフォルトヒープ取得 | Default/Borrowed.getDefault() |

### 計算ロジック

mustUseAlignedAlloc判定：
```zig
pub fn mustUseAlignedAlloc(alignment: std.mem.Alignment) bool {
    return alignment.toByteUnits() > MI_MAX_ALIGN_SIZE; // 16
}
```

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| null | 割り当て失敗 | メモリ不足 | null返却、呼び出し側でbun.outOfMemory()等 |
| panic | スレッド違反 | 別スレッドからのアクセス（デバッグ） | パニック、開発時の修正 |

### リトライ仕様

リトライは行わない。割り当て失敗時は即座にnull返却。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- mimallocによる高速割り当て（mallocより高速）
- スレッドローカルヒープでロック競合を回避
- 小さいアロケーション（<128ワード）は特に高速

## セキュリティ考慮事項

- メモリリージョン外へのアクセス検出（デバッグビルド）
- スレッド所有権検証でデータ競合を防止

## 備考

- mimalloc v3のTHeap APIも提供
- ASANビルドではmimallocを無効化してシステムアロケータを使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: mimallocバインディングを理解する

mimallocのextern宣言を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | mimalloc.zig | `src/allocators/mimalloc.zig` | extern関数宣言 |

**読解のコツ**: Zigのextern宣言は、C言語の関数シグネチャをそのままZigに翻訳している。

**主要処理フロー**:
- **1-41行目**: 基本アロケーション関数（mi_malloc, mi_free, mi_realloc等）
- **42-66行目**: Heap opaque型とメソッド
- **67-144行目**: ヒープ関連extern関数
- **145-189行目**: Option列挙型
- **232-234行目**: mustUseAlignedAlloc関数

#### Step 2: MimallocArenaを理解する

Zigアロケータインターフェースへの適合を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MimallocArena.zig | `src/allocators/MimallocArena.zig` | MimallocArena構造体 |

**主要処理フロー**:
- **1-9行目**: Self構造体定義（#heap、thread_id）
- **13-18行目**: Default構造体 - ゼロサイズスレッドローカル
- **25-80行目**: Borrowed構造体 - 借用パターン
- **84-91行目**: allocator/borrow関数
- **124-127行目**: deinit - mi_heap_destroy
- **129-134行目**: init - mi_heap_new
- **144-154行目**: assertThreadOwnership - スレッド所有権検証
- **199-204行目**: c_allocator_vtable - Allocator VTable

#### Step 3: その他のアロケータを理解する

特殊用途のアロケータを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | NullableAllocator.zig | `src/allocators/NullableAllocator.zig` | null許容アロケータ |
| 3-2 | BufferFallbackAllocator.zig | `src/allocators/BufferFallbackAllocator.zig` | バッファフォールバック |
| 3-3 | MaxHeapAllocator.zig | `src/allocators/MaxHeapAllocator.zig` | 最大ヒープ制限付き |

### プログラム呼び出し階層図

```
アプリケーションコード
    │
    └─ MimallocArena.allocator()
           │
           └─ std.mem.Allocator
                  │
                  ├─ vtable_alloc
                  │      │
                  │      └─ Borrowed.alignedAlloc
                  │             │
                  │             ├─ mustUseAlignedAlloc判定
                  │             │
                  │             ├─ mi_heap_malloc_aligned
                  │             │
                  │             └─ mi_heap_malloc
                  │
                  ├─ vtable_resize
                  │      └─ mi_expand
                  │
                  ├─ vtable_remap
                  │      └─ mi_heap_realloc_aligned
                  │
                  └─ vtable_free
                         │
                         ├─ mi_free (リリース)
                         │
                         └─ mi_free_size_aligned (デバッグ)

MimallocArena.init
    └─ mi_heap_new → ヒープ作成

MimallocArena.deinit
    └─ mi_heap_destroy → ヒープ破棄

Borrowed.getDefault
    └─ mi_heap_main → スレッドローカルヒープ
```

### データフロー図

```
[入力]                        [処理]                           [出力]

割り当て要求        ───▶  MimallocArena       ───▶  メモリポインタ
  │                         │
  ├─ size                  ├─ assertThreadOwnership (デバッグ)
  └─ alignment             ├─ mustUseAlignedAlloc判定
                            └─ mi_heap_malloc(_aligned)
                                   │
                                   ▼
                            ┌─────────────────┐
                            │ mimalloc Heap   │
                            │ ├─ ページ管理    │
                            │ ├─ セグメント管理│
                            │ └─ アリーナ管理  │
                            └─────────────────┘
                                   │
                                   ▼
                            メモリブロック

解放要求            ───▶  vtable_free         ───▶  ヒープに返却
  │                         │
  └─ ptr                   └─ mi_free
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| mimalloc.zig | `src/allocators/mimalloc.zig` | ソース | mimallocのZigバインディング |
| MimallocArena.zig | `src/allocators/MimallocArena.zig` | ソース | スレッドローカルヒープアロケータ |
| NullableAllocator.zig | `src/allocators/NullableAllocator.zig` | ソース | null許容アロケータ |
| BufferFallbackAllocator.zig | `src/allocators/BufferFallbackAllocator.zig` | ソース | バッファフォールバックアロケータ |
| MaxHeapAllocator.zig | `src/allocators/MaxHeapAllocator.zig` | ソース | 最大ヒープ制限付きアロケータ |
| LinuxMemFdAllocator.zig | `src/allocators/LinuxMemFdAllocator.zig` | ソース | Linux memfdアロケータ |
| allocation_scope.zig | `src/allocators/allocation_scope.zig` | ソース | アロケーションスコープ |
| basic.zig | `src/allocators/basic.zig` | ソース | 基本アロケータユーティリティ |
| fallback.zig | `src/allocators/fallback.zig` | ソース | フォールバックアロケータ |
| maybe_owned.zig | `src/allocators/maybe_owned.zig` | ソース | 所有権オプショナルアロケータ |
