# 機能設計書 116-JavaScriptCoreバインディング

## 概要

本ドキュメントは、BunとJavaScriptCore（WebKit）を接続するバインディング層の設計を記述する。ZigコードからJavaScript値の操作、グローバルオブジェクトへのアクセス、VMの制御を可能にする。

### 本機能の処理概要

JavaScriptCoreバインディングは、BunのZigコードとWebKitのJavaScriptCoreエンジンの間の橋渡しを行う。JSValue、JSGlobalObject、VMなどの主要なJavaScriptCore概念をZig言語で型安全に扱えるようにし、JavaScript関数の呼び出し、プロパティアクセス、型変換などの操作を提供する。

**業務上の目的・背景**：BunはWebKitのJavaScriptCoreエンジンを使用してJavaScriptを実行するが、BunのコアランタイムはZigで実装されている。このバインディング層により、Zig実装のBun APIがJavaScriptと相互運用できるようになる。型安全なインターフェースにより、メモリ安全性とパフォーマンスを両立する。

**機能の利用シーン**：
- Node.js互換APIの実装（fs、http等）
- Bun固有API（Bun.file、Bun.serve等）の実装
- JavaScript例外のハンドリング
- Promise/async/awaitのサポート
- オブジェクトプロパティの読み書き
- JavaScript関数の呼び出し
- 型変換（JavaScript ⇔ Zig）

**主要な処理内容**：
1. JSValue - JavaScript値のABI互換表現
2. JSGlobalObject - グローバルオブジェクトとエラー処理
3. VM - JavaScriptCore仮想マシン制御
4. JSPromise - Promiseの作成と解決
5. JSObject - オブジェクトプロパティ操作
6. ZigString - 文字列のゼロコピー受け渡し
7. Exception - 例外のZig側ハンドリング

**関連システム・外部連携**：
- JavaScriptCore（WebKit）
- C++バインディングコード（src/bun.js/bindings/*.cpp）
- コード生成システム（*.classes.ts）

**権限による制御**：特になし（ランタイム内部機能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はランタイム内部機能であり、直接的な画面は存在しない |

## 機能種別

言語間バインディング / 型変換 / API提供

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| JSValue | enum(i64) | Yes | JavaScript値（64ビットエンコード） | - |
| JSGlobalObject | opaque | Yes | グローバルオブジェクトポインタ | - |
| VM | opaque | Yes | JavaScriptCore VMポインタ | - |

### 入力データソース

- JavaScriptコードからの値
- C++バインディングからのコールバック

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| JSValue | enum(i64) | JavaScript値 |
| JSError | error | JavaScript例外発生 |
| Zig型 | various | 変換されたZig値 |

### 出力先

- JavaScript側への返却値
- Zig側での処理結果

## 処理フロー

### 処理シーケンス

```
1. JSValue操作
   └─ 型判定（isNumber, isString, isObject等）
   └─ 値取得（toInt32, toString等）
   └─ 型変換（to, coerce）

2. JSGlobalObject操作
   └─ エラー生成（createErrorInstance, throw等）
   └─ マイクロタスク管理（queueMicrotask）
   └─ VMアクセス

3. VM操作
   └─ GC制御（runGC, collectAsync）
   └─ 実行制御（setExecutionTimeLimit）
   └─ トラップ通知（notifyNeedTermination）

4. 関数呼び出し
   └─ JSValue.call
   └─ fromJSHostCall（例外チェック付き）
```

### フローチャート

```mermaid
flowchart TD
    A[Zigコード] --> B{JSValue操作}
    B -->|型判定| C[isNumber/isString/isObject]
    B -->|値取得| D[toInt32/toString/asDouble]
    B -->|関数呼び出し| E[call]

    C --> F{型一致?}
    F -->|Yes| G[値取得]
    F -->|No| H[型変換/エラー]

    E --> I[fromJSHostCall]
    I --> J{例外発生?}
    J -->|Yes| K[JSError返却]
    J -->|No| L[結果返却]

    M[JSGlobalObject] --> N{エラー処理}
    N -->|生成| O[createErrorInstance]
    N -->|スロー| P[throwValue]

    Q[VM] --> R{制御操作}
    R -->|GC| S[runGC/collectAsync]
    R -->|終了| T[notifyNeedTermination]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | JSValue ABI互換 | enum(i64)でEncodedJSValueと互換 | 全JSValue操作 |
| BR-002 | 例外伝播 | JSError型で例外を伝播、hasException()で確認 | JS呼び出し時 |
| BR-003 | ゼロ値の意味 | JSValue.zero = 0は例外発生を示す（将来廃止予定） | 互換性維持 |
| BR-004 | マイクロタスクドレイン | JS関数呼び出し後はdrainMicrotasks必須 | イベントループ |
| BR-005 | fromJSHostCall | 例外チェック付きextern呼び出しパターン | 全extern呼び出し |
| BR-006 | TopExceptionScope | 例外スコープの明示的管理 | 複雑な例外処理 |

### 計算ロジック

JSValue型判定（ビット演算）：
- isInt32: 特定ビットパターンでint32を識別
- isDouble: NaNボクシングで64ビット浮動小数点を識別
- isCell: ポインタとしてオブジェクトを識別

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| JSError | JavaScript例外 | JS関数呼び出しで例外発生 | error.JSError返却 |
| JSTerminated | 終了要求 | VM終了要求時 | error.JSTerminated返却 |
| OutOfMemory | メモリ不足 | メモリ割り当て失敗 | throwOutOfMemory |

### リトライ仕様

リトライは行わない。例外発生時は即座にエラー返却。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- JSValueはenum(i64)でインライン格納（ヒープ割り当て不要）
- extern関数はC++とゼロコスト相互運用
- ZigStringはゼロコピーで文字列受け渡し

## セキュリティ考慮事項

- 型安全なインターフェースでメモリ安全性を保証
- 例外の適切なハンドリングで情報漏洩を防止
- TopExceptionScopeで例外スコープを明示管理

## 備考

- *.classes.tsからZig/C++バインディングを自動生成
- extern関数名は`JSC__`または`Bun__`プレフィックス

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: JSValueを理解する

JavaScript値の基本表現を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | JSValue.zig | `src/bun.js/bindings/JSValue.zig` | JSValue enum定義 |

**読解のコツ**: Zigのenum(i64)はJavaScriptCoreのEncodedJSValueとABI互換。特殊値（undefined, null, true, false）は固定エンコード。

**主要処理フロー**:
- **1-23行目**: JSValue enum定義（js_undefined, null, true, false, zero）
- **32-34行目**: cast - ポインタからJSValueへの変換
- **70-74行目**: isFalsey/isTruthy - 真偽値判定
- **119-121行目**: toNumber - 数値変換
- **124-137行目**: isInteger - 整数判定
- **151-183行目**: coerce - 型強制変換
- **187-209行目**: to - ゼロコスト型変換
- **249-270行目**: call - JavaScript関数呼び出し

#### Step 2: JSGlobalObjectを理解する

グローバルオブジェクトとエラー処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | JSGlobalObject.zig | `src/bun.js/bindings/JSGlobalObject.zig` | JSGlobalObject opaque定義 |

**主要処理フロー**:
- **1行目**: JSGlobalObject = opaque定義
- **6-14行目**: throwStackOverflow/throwOutOfMemory
- **47-60行目**: throwInvalidArguments/throwMissingArgumentsValue
- **261-282行目**: createErrorInstance - エラーインスタンス生成
- **372-376行目**: throw - フォーマット付きエラースロー
- **404-410行目**: queueMicrotask - マイクロタスク登録
- **422-424行目**: throwValue - 値をエラーとしてスロー
- **483-489行目**: hasException/clearException

#### Step 3: VMを理解する

JavaScriptCore VMの制御を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | VM.zig | `src/bun.js/bindings/VM.zig` | VM opaque定義 |

**主要処理フロー**:
- **1-5行目**: VM opaque定義、HeapType enum
- **7-15行目**: create/deinit - VM作成・破棄
- **42-52行目**: getAPILock/Lock - APIロック管理
- **79-92行目**: runGC/collectAsync - GC制御
- **94-112行目**: setExecutionTimeLimit/clearExecutionTimeLimit
- **117-143行目**: notifyNeed*Trap - VMトラップ通知
- **158-167行目**: throwError - エラースロー
- **174-179行目**: drainMicrotasks - マイクロタスクドレイン

#### Step 4: その他の主要型を理解する

Promise、Object、String等の型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | JSPromise.zig | `src/bun.js/bindings/JSPromise.zig` | Promise操作 |
| 4-2 | JSObject.zig | `src/bun.js/bindings/JSObject.zig` | オブジェクト操作 |
| 4-3 | ZigString.zig | `src/bun.js/bindings/ZigString.zig` | 文字列操作 |
| 4-4 | Exception.zig | `src/bun.js/bindings/Exception.zig` | 例外型 |

### プログラム呼び出し階層図

```
Bun API (Zig)
    │
    ├─ JSValue操作
    │      │
    │      ├─ 型判定
    │      │      ├─ isNumber() → ビット演算
    │      │      ├─ isString() → jsType()
    │      │      └─ isObject() → jsType()
    │      │
    │      ├─ 値取得
    │      │      ├─ toInt32() → JSC__JSValue__toInt32
    │      │      ├─ toString() → JSC__JSValue__toString
    │      │      └─ asDouble() → ビット操作
    │      │
    │      └─ 関数呼び出し
    │             └─ call() → Bun__JSValue__call
    │                    └─ fromJSHostCall (例外チェック)
    │
    ├─ JSGlobalObject操作
    │      │
    │      ├─ エラー生成
    │      │      ├─ createErrorInstance()
    │      │      └─ throw()
    │      │
    │      └─ マイクロタスク
    │             └─ queueMicrotask()
    │
    └─ VM操作
           │
           ├─ GC制御
           │      ├─ runGC()
           │      └─ collectAsync()
           │
           └─ 終了制御
                  └─ notifyNeedTermination()
```

### データフロー図

```
[JavaScript]                  [バインディング]              [Zig]

JSValue         ───▶    JSValue enum(i64)    ───▶  Zig型
  │                         │                       │
  ├─ number                ├─ isNumber()           ├─ i32
  ├─ string                ├─ isString()           ├─ []const u8
  ├─ object                ├─ isObject()           ├─ struct
  └─ function              └─ isCallable()         └─ callback

JSGlobalObject  ───▶    JSGlobalObject opaque ───▶  エラー処理
  │                         │                       │
  └─ Error                 ├─ createErrorInstance() ├─ JSError
                            └─ throwValue()         └─ error返却

VM              ───▶    VM opaque             ───▶  制御
  │                         │                       │
  ├─ GC                    └─ runGC()              └─ メモリ解放
  └─ microtasks            └─ drainMicrotasks()    └─ タスク実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| JSValue.zig | `src/bun.js/bindings/JSValue.zig` | ソース | JavaScript値のZig表現 |
| JSGlobalObject.zig | `src/bun.js/bindings/JSGlobalObject.zig` | ソース | グローバルオブジェクト |
| VM.zig | `src/bun.js/bindings/VM.zig` | ソース | JavaScriptCore VM |
| JSPromise.zig | `src/bun.js/bindings/JSPromise.zig` | ソース | Promise操作 |
| JSObject.zig | `src/bun.js/bindings/JSObject.zig` | ソース | オブジェクト操作 |
| JSArray.zig | `src/bun.js/bindings/JSArray.zig` | ソース | 配列操作 |
| JSFunction.zig | `src/bun.js/bindings/JSFunction.zig` | ソース | 関数操作 |
| JSString.zig | `src/bun.js/bindings/JSString.zig` | ソース | 文字列操作 |
| ZigString.zig | `src/bun.js/bindings/ZigString.zig` | ソース | Zig文字列変換 |
| Exception.zig | `src/bun.js/bindings/Exception.zig` | ソース | 例外型 |
| ZigException.zig | `src/bun.js/bindings/ZigException.zig` | ソース | Zig例外 |
| TopExceptionScope.zig | `src/bun.js/bindings/TopExceptionScope.zig` | ソース | 例外スコープ |
| ErrorCode.zig | `src/bun.js/bindings/ErrorCode.zig` | ソース | エラーコード |
| CallFrame.zig | `src/bun.js/bindings/CallFrame.zig` | ソース | コールフレーム |
| JSCell.zig | `src/bun.js/bindings/JSCell.zig` | ソース | JSCセル |
| JSType.zig | `src/bun.js/bindings/JSType.zig` | ソース | JS型列挙 |
| codegen.zig | `src/bun.js/bindings/codegen.zig` | ソース | コード生成ヘルパー |
