# 機能設計書 42-グローバルキャッシュ

## 概要

本ドキュメントは、Bunパッケージマネージャーのグローバルキャッシュ管理機能に関する設計書である。パッケージのキャッシュ保存、ハードリンクによる高速インストール、キャッシュのクリーンアップについて詳述する。

### 本機能の処理概要

グローバルキャッシュ機能は、ダウンロードしたnpmパッケージをシステム全体で共有可能な場所に保存し、ハードリンクを使用して各プロジェクトのnode_modulesに高速にインストールする機能である。

**業務上の目的・背景**：従来のパッケージマネージャーでは、各プロジェクトごとにパッケージをダウンロード・展開するため、ディスク容量とインストール時間が大きくなる。グローバルキャッシュ機能により、同一パッケージを複数プロジェクトで共有でき、ディスク使用量の削減と2回目以降のインストール高速化を実現する。Bunの「爆速インストール」の中核を担う機能である。

**機能の利用シーン**：
- `bun install` 実行時のパッケージ取得とキャッシュ
- 同一パッケージの別プロジェクトへのインストール時
- `bun pm cache` によるキャッシュ管理操作
- `bun link` によるローカルパッケージのグローバル登録

**主要な処理内容**：
1. パッケージtarballのダウンロードと展開
2. キャッシュディレクトリへのファイル保存
3. ハードリンクによるnode_modulesへのファイル配置
4. キャッシュの整合性検証（integrityチェック）
5. キャッシュディレクトリの容量管理とクリーンアップ
6. グローバルリンクディレクトリの管理

**関連システム・外部連携**：
- ファイルシステム（ハードリンク、シンボリックリンク）
- npmレジストリ（tarballダウンロード）
- bunfig.toml（キャッシュパス設定）

**権限による制御**：ファイルシステムの権限に依存。キャッシュディレクトリへの書き込み権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 6 | install | 主機能 | ハードリンクを使用した高速キャッシュ管理 |
| 12 | link | 補助機能 | リンク先のグローバルキャッシュへの登録 |
| 17 | pm | 補助機能 | キャッシュ情報の表示やクリア |

## 機能種別

データ管理（ファイルシステム操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| package_name | string | Yes | パッケージ名 | 有効なnpmパッケージ名 |
| version | string | Yes | バージョン | セマンティックバージョン形式 |
| tarball_data | bytes | Yes | パッケージアーカイブデータ | gzip圧縮tarball |
| integrity | string | No | SHA512ハッシュ | sha512-で始まる文字列 |
| cache_dir | path | No | キャッシュディレクトリパス | 有効なディレクトリパス |

### 入力データソース

- npmレジストリからのtarballダウンロード
- ローカルファイル（file:プロトコル）
- Gitリポジトリ
- bunfig.toml（キャッシュ設定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cache_path | path | キャッシュに保存されたパッケージのパス |
| installed_files | array | インストールされたファイルのリスト |
| hardlink_count | number | 作成されたハードリンク数 |
| disk_usage | number | 使用ディスク容量（バイト） |

### 出力先

- グローバルキャッシュディレクトリ（~/.bun/install/cache）
- node_modules（ハードリンクまたはコピー）
- グローバルリンクディレクトリ

## 処理フロー

### 処理シーケンス

```
1. キャッシュ存在確認
   └─ パッケージ名+バージョンでキャッシュを検索
2. キャッシュミスの場合
   └─ tarballをダウンロード、展開、キャッシュに保存
3. キャッシュヒットの場合
   └─ integrityチェックで整合性を検証
4. インストール先の準備
   └─ node_modules配下にディレクトリ構造を作成
5. ファイルの配置
   └─ ハードリンク作成（失敗時はコピーにフォールバック）
6. バイナリリンクの設定
   └─ package.jsonのbinフィールドに基づくリンク作成
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{キャッシュにパッケージあり?}
    B -->|Yes| C{integrityチェック}
    B -->|No| D[tarballをダウンロード]
    C -->|OK| E[キャッシュからインストール]
    C -->|NG| D
    D --> F[tarballを展開]
    F --> G[キャッシュに保存]
    G --> E
    E --> H{ハードリンク可能?}
    H -->|Yes| I[ハードリンクを作成]
    H -->|No| J[ファイルをコピー]
    I --> K[バイナリリンク設定]
    J --> K
    K --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-42-01 | キャッシュパス | デフォルトは~/.bun/install/cache | 環境変数/設定で変更可能 |
| BR-42-02 | ハードリンク優先 | 可能な限りハードリンクを使用 | 同一ファイルシステム上 |
| BR-42-03 | コピーフォールバック | ハードリンク不可時はコピー | 別ファイルシステム/権限不足時 |
| BR-42-04 | integrity検証 | SHA512ハッシュで整合性確認 | lockfileにintegrityがある場合 |
| BR-42-05 | パッチ対応キャッシュ | パッチ適用版は別キャッシュに保存 | patchedDependencies使用時 |

### 計算ロジック

キャッシュパスの生成:
```
cache_path = cache_dir + "/" + package_name + "/" + version
patched_cache_path = cache_path + "+" + patch_hash
```

## データベース操作仕様

本機能はデータベースを使用しない。ファイルシステムベースのキャッシュを使用。

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ENOSPC | ディスク容量不足 | キャッシュ保存時 | 古いキャッシュをクリア |
| EACCES | 権限エラー | キャッシュディレクトリ書き込み不可 | 権限を確認/変更 |
| EXDEV | クロスデバイス | 異なるファイルシステム間のハードリンク | コピーにフォールバック |
| ENOENT | ファイル未発見 | キャッシュ破損 | 再ダウンロード |
| IntegrityMismatch | 整合性エラー | ハッシュ不一致 | 再ダウンロード |

### リトライ仕様

- ハードリンク失敗時: コピー操作にフォールバック（リトライなし）
- ダウンロード失敗時: ネットワーク層のリトライポリシーに従う

## トランザクション仕様

アトミック性確保のため、一時ディレクトリに展開後にリネームでキャッシュに移動。

## パフォーマンス要件

- キャッシュヒット時のインストール: 10ms/パッケージ以下
- ハードリンク作成: 1ms/ファイル以下
- 並列展開対応（マルチスレッド）

## セキュリティ考慮事項

- integrityチェックによるパッケージ改ざん検知
- キャッシュディレクトリの適切な権限設定（0755）
- シンボリックリンク攻撃への対策

## 備考

- Windowsではジャンクションを使用
- macOSではAPFS クローンを活用可能
- `bun pm cache rm` でキャッシュ削除可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

キャッシュ管理で使用される主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PackageManager.zig | `src/install/PackageManager.zig` | cache_directory_、global_link_dirフィールドの役割 |
| 1-2 | install.zig | `src/install/install.zig` | Store構造体でキャッシュストアを理解 |

**読解のコツ**: PackageManager構造体の先頭部分にキャッシュ関連のフィールドが定義されている。

#### Step 2: キャッシュディレクトリの決定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PackageManager.zig | `src/install/PackageManager.zig` | getCacheDirectory関数 |

**主要処理フロー**:
1. **1行目**: cache_directory_フィールドにキャッシュディレクトリを保持
2. **94-96行目**: global_link_dir、global_dirでグローバルリンクを管理

#### Step 3: パッケージインストール処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PackageInstall.zig | `src/install/PackageInstall.zig` | PackageInstall構造体とinstall関数 |
| 3-2 | extract_tarball.zig | `src/install/extract_tarball.zig` | tarballの展開処理 |

**主要処理フロー**:
- tarballダウンロード → 展開 → キャッシュ保存 → ハードリンク作成

#### Step 4: ハードリンク作成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | PackageInstall.zig | `src/install/PackageInstall.zig` | copyfile操作とハードリンクフォールバック |

### プログラム呼び出し階層図

```
PackageManager
    │
    ├─ getCacheDirectory()
    │      └─ キャッシュパスの決定
    │
    ├─ PackageInstall.install()
    │      │
    │      ├─ キャッシュ存在確認
    │      │
    │      ├─ ExtractTarball (キャッシュミス時)
    │      │      └─ tarball展開 → キャッシュ保存
    │      │
    │      └─ copyfile/hardlink (インストール)
    │             └─ node_modulesへの配置
    │
    └─ Bin.link()
           └─ バイナリリンクの作成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

tarball URL ─────────▶ NetworkTask ──────────────▶ tarball data
                              │
                              ▼
                      ExtractTarball ─────────────▶ 展開ファイル
                              │
                              ▼
                      キャッシュ保存 ───────────────▶ ~/.bun/install/cache/
                              │
                              ▼
                      ハードリンク/コピー ──────────▶ node_modules/
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PackageManager.zig | `src/install/PackageManager.zig` | ソース | キャッシュディレクトリ管理の中核 |
| PackageInstall.zig | `src/install/PackageInstall.zig` | ソース | パッケージインストール処理 |
| extract_tarball.zig | `src/install/extract_tarball.zig` | ソース | tarball展開処理 |
| install.zig | `src/install/install.zig` | ソース | インストール機能の統合 |
| bin.zig | `src/install/bin.zig` | ソース | バイナリリンク処理 |
| integrity.zig | `src/install/integrity.zig` | ソース | 整合性検証 |
