# 機能設計書 45-パッチ管理

## 概要

本ドキュメントは、Bunパッケージマネージャーのパッチ管理機能に関する設計書である。依存パッケージへのローカル修正（パッチ）の作成・適用・管理について詳述する。

### 本機能の処理概要

パッチ管理機能は、npmパッケージに対するローカル修正をdiffファイル（.patch）として保存し、インストール時に自動適用する機能である。pnpm patchおよびpatch-package互換の機能を提供する。

**業務上の目的・背景**：開発中、依存パッケージにバグや機能不足がある場合、上流へのPR/issue対応を待てないことがある。パッチ管理機能により、パッケージをフォークせずにローカル修正を維持でき、上流が修正された際にはパッチを削除するだけで済む。チーム全体でパッチを共有でき、再現性のある開発環境を構築できる。

**機能の利用シーン**：
- `bun patch <package>` でパッチ編集セッション開始
- パッケージのソースコードを編集
- `bun patch-commit <path>` でパッチを保存
- `bun install` 時にパッチを自動適用

**主要な処理内容**：
1. パッチ対象パッケージの一時ディレクトリへの展開
2. ユーザーによる編集の追跡
3. 編集前後の差分をdiff形式で生成
4. patches/ディレクトリへのパッチファイル保存
5. package.jsonのpatchedDependencies更新
6. インストール時のパッチ適用

**関連システム・外部連携**：
- ファイルシステム（一時ディレクトリ、patches/）
- diff/patchユーティリティ互換形式
- package.json（patchedDependencies）
- pnpm patch互換

**権限による制御**：ファイルシステムの書き込み権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 6 | install | 補助機能 | インストール時のパッチ自動適用 |

## 機能種別

データ管理（ファイル差分管理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| package_name | string | Yes | パッチ対象パッケージ名 | インストール済みパッケージ |
| version | string | No | 対象バージョン | セマンティックバージョン |
| patch_file | path | No | パッチファイルパス | 有効なdiff形式 |

### 入力データソース

- package.json（patchedDependencies）
- patches/ディレクトリ内の.patchファイル
- node_modules内のパッケージ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| patch_file | path | 生成されたパッチファイル |
| patch_hash | string | パッチのハッシュ値 |
| modified_files | array | 変更されたファイル一覧 |

### 出力先

- patches/<package-name>@<version>.patch
- package.json（patchedDependenciesセクション）
- キャッシュ（パッチ適用済みパッケージ）

## 処理フロー

### 処理シーケンス

```
1. パッチ開始（bun patch）
   └─ パッケージを一時ディレクトリに展開
2. 編集
   └─ ユーザーがファイルを編集
3. パッチコミット（bun patch-commit）
   └─ 差分を計算、パッチファイル生成
4. package.json更新
   └─ patchedDependenciesに追加
5. インストール時適用
   └─ パッチをパッケージに適用
```

### フローチャート

```mermaid
flowchart TD
    A[bun patch開始] --> B[パッケージをキャッシュから取得]
    B --> C[一時ディレクトリに展開]
    C --> D[オリジナルのコピーを保持]
    D --> E[ユーザーに編集パス通知]
    E --> F[ユーザーが編集]
    F --> G[bun patch-commit]
    G --> H[差分を計算]
    H --> I{差分あり?}
    I -->|No| J[パッチなし終了]
    I -->|Yes| K[パッチファイル生成]
    K --> L[patches/に保存]
    L --> M[package.json更新]
    M --> N[パッチハッシュ計算]
    N --> O[lockfile更新]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-01 | パッチファイル形式 | unified diff形式 | パッチ生成時 |
| BR-45-02 | パッチパス | patches/<pkg>@<version>.patch | デフォルト |
| BR-45-03 | ハッシュ更新 | パッチ変更時にハッシュ再計算 | patch-commit時 |
| BR-45-04 | キャッシュ分離 | パッチ適用版は別キャッシュ | パッチ有効時 |
| BR-45-05 | バージョン固定 | パッチはバージョン固定で管理 | 推奨 |

### 計算ロジック

パッチハッシュの計算:
```
patch_hash = sha256(patch_file_content)
cache_key = package_name + "@" + version + "+" + patch_hash
```

## データベース操作仕様

本機能はデータベースを使用しない。

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| PackageNotFound | パッケージ未発見 | 指定パッケージが未インストール | パッケージをインストール |
| PatchApplyFailed | 適用失敗 | パッチがコンフリクト | パッチを再作成 |
| InvalidPatchFormat | 形式エラー | パッチ形式が不正 | 正しいdiff形式で作成 |
| NoDiffFound | 差分なし | 編集がない | 何もせず終了 |
| TempDirError | 一時Dir エラー | 一時ディレクトリ作成失敗 | 権限を確認 |

### リトライ仕様

パッチ適用失敗時は自動リトライしない。エラーメッセージを表示して手動対応を促す。

## トランザクション仕様

パッチ適用はアトミック。適用失敗時はパッケージを元の状態に戻す。

## パフォーマンス要件

- パッチ生成: 1秒以内（通常のパッチサイズ）
- パッチ適用: 100ms/パッケージ以下
- ハッシュ計算: 10ms以内

## セキュリティ考慮事項

- パッチファイルは平文で保存（レビュー可能）
- 悪意のあるパッチへの対策（手動レビュー推奨）
- パッチ適用時のファイルパス検証

## 備考

- pnpm patch互換
- patch-package形式をサポート
- `bun pm patch-remove <pkg>` でパッチ削除可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

パッチ管理で使用される主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | patch_install.zig | `src/install/patch_install.zig` | PatchTask構造体 |
| 1-2 | PackageManager.zig | `src/install/PackageManager.zig` | patch関連フィールド |

**読解のコツ**: PatchTaskが非同期のパッチ適用処理を管理。

#### Step 2: パッチコマンドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PackageManager.zig | `src/install/PackageManager.zig` | Subcommand.patch、@"patch-commit" |
| 2-2 | patch_install.zig | `src/install/patch_install.zig` | applyPatch関数 |

**主要処理フロー**:
- **147-148行目（PackageManager.zig）**: Subcommand に patch と patch-commit を定義

#### Step 3: パッチ適用処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PackageManager.zig | `src/install/PackageManager.zig` | patch_apply_batch、patch_task_fifo |
| 3-2 | patch_install.zig | `src/install/patch_install.zig` | パッチの実行ロジック |

**主要処理フロー**:
- **61-63行目（PackageManager.zig）**: patch_apply_batch、patch_calc_hash_batch、patch_task_fifoでパッチタスク管理
- **64行目**: patch_task_queue でパッチタスクのキュー管理
- **69行目**: pending_pre_calc_hashes でハッシュ計算の追跡

#### Step 4: pnpm互換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | pnpm.zig | `src/install/pnpm.zig` | patchedDependenciesの処理 |

**主要処理フロー**:
- **22-23行目**: patch_hash_idxでパッチハッシュのパースを理解
- **136-137行目**: found_patchesでパッチ情報を管理
- **196-222行目**: patchedDependenciesの解析ロジック

### プログラム呼び出し階層図

```
PackageManager
    │
    ├─ Subcommand.patch
    │      └─ パッチ編集セッション開始
    │
    ├─ Subcommand.@"patch-commit"
    │      └─ パッチファイル生成とコミット
    │
    ├─ patch_apply_batch
    │      └─ パッチ適用のバッチ処理
    │
    ├─ patch_calc_hash_batch
    │      └─ パッチハッシュ計算のバッチ処理
    │
    ├─ PatchTask
    │      └─ 個別パッチの非同期適用
    │
    └─ patched_dependencies_to_remove
           └─ 削除対象パッチの管理
```

### データフロー図

```
[入力]                    [処理]                         [出力]

bun patch <pkg> ────────▶ キャッシュから展開 ──────────▶ 一時ディレクトリ
                              │
                              ▼
ユーザー編集 ───────────▶ ファイル変更追跡 ──────────▶ 変更ファイル一覧
                              │
                              ▼
bun patch-commit ───────▶ diff生成 ────────────────▶ .patchファイル
                              │
                              ▼
                      ハッシュ計算 ────────────────▶ patch_hash
                              │
                              ▼
                      package.json更新 ────────────▶ patchedDependencies
                              │
                              ▼
bun install ────────────▶ パッチ適用 ──────────────▶ node_modules/
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| patch_install.zig | `src/install/patch_install.zig` | ソース | パッチ適用の中核ロジック |
| PackageManager.zig | `src/install/PackageManager.zig` | ソース | パッチタスク管理 |
| pnpm.zig | `src/install/pnpm.zig` | ソース | pnpmパッチ形式互換 |
| lockfile.zig | `src/install/lockfile.zig` | ソース | パッチ情報の永続化 |
| install.zig | `src/install/install.zig` | ソース | patched_dependenciesフラグ |
| extract_tarball.zig | `src/install/extract_tarball.zig` | ソース | パッケージ展開 |
