# 機能設計書 51-コードカバレッジ

## 概要

本ドキュメントは、Bunテストランナーにおけるコードカバレッジ機能の設計仕様を記述する。

### 本機能の処理概要

コードカバレッジ機能は、テスト実行時にソースコードの実行状況を計測し、どの行が実行されたかをレポートとして出力する。JavaScriptCore（JSC）エンジンの基本ブロックカバレッジ機能を活用し、行単位・関数単位・ステートメント単位でのカバレッジ情報を収集する。

**業務上の目的・背景**：ソフトウェア品質保証において、テストがコードのどの部分をカバーしているかを把握することは重要である。未テスト領域の特定、テストの網羅性評価、リグレッションリスクの可視化を実現する。

**機能の利用シーン**：CI/CDパイプラインでのテスト品質検証、開発中のテストカバレッジ確認、コードレビュー時の品質評価、カバレッジ閾値によるビルドゲート制御。

**主要な処理内容**：
1. `--coverage`オプションの解析とカバレッジモードの有効化
2. JSCエンジンからの基本ブロック実行情報の取得
3. ソースマップを使用した元ファイル行番号へのマッピング
4. テキスト形式またはLCOV形式でのレポート生成
5. カバレッジ閾値との比較による成功/失敗判定

**関連システム・外部連携**：JavaScriptCoreエンジン（基本ブロックカバレッジAPI）、ソースマップ解析、LCOV形式レポート（外部ツール連携用）。

**権限による制御**：特になし。全ユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | test | 主画面 | `--coverage`オプションによるカバレッジレポート生成 |

## 機能種別

計算処理 / レポート出力

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| --coverage | フラグ | No | カバレッジ収集を有効化 | - |
| --coverage-dir | 文字列 | No | カバレッジ出力ディレクトリ | 有効なパス |
| --coverage-reporter | 文字列 | No | レポート形式（text, lcov） | text または lcov |

### 入力データソース

- JSCエンジンからの基本ブロック実行情報
- テスト対象ソースファイル
- ソースマップファイル（存在する場合）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_url | string | ソースファイルパス |
| executable_lines | Bitset | 実行可能行のビットセット |
| lines_which_have_executed | Bitset | 実行された行のビットセット |
| line_hits | u32[] | 各行の実行回数 |
| functions | Block[] | 関数の開始行・終了行 |
| functions_which_have_executed | Bitset | 実行された関数 |

### 出力先

- 標準出力（テキストレポート）
- ファイル出力（LCOVレポート）
- カバレッジディレクトリ（指定時）

## 処理フロー

### 処理シーケンス

```
1. テストコマンド実行時にcoverageオプション解析
   └─ Arguments.zigでオプションをパース
2. JSCエンジンでカバレッジ収集モードを有効化
   └─ 基本ブロック実行追跡を開始
3. テスト実行
   └─ 各ファイルのカバレッジ情報を蓄積
4. ByteRangeMapping生成
   └─ ソースコードのバイトオフセットと行番号のマッピング
5. 基本ブロック情報からReport生成
   └─ generateReportFromBlocksで行カバレッジを計算
6. ソースマップ適用（存在する場合）
   └─ 元ファイルの行番号に変換
7. レポート出力
   └─ Text形式またはLCOV形式で出力
```

### フローチャート

```mermaid
flowchart TD
    A[テスト開始] --> B{--coverage指定?}
    B -->|No| C[通常テスト実行]
    B -->|Yes| D[カバレッジモード有効化]
    D --> E[JSC基本ブロック追跡開始]
    E --> F[テスト実行]
    F --> G[基本ブロック情報取得]
    G --> H[ByteRangeMapping生成]
    H --> I{ソースマップあり?}
    I -->|Yes| J[行番号マッピング変換]
    I -->|No| K[直接行計算]
    J --> L[Report生成]
    K --> L
    L --> M{レポート形式}
    M -->|text| N[テキストレポート出力]
    M -->|lcov| O[LCOVレポート出力]
    N --> P[終了]
    O --> P
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-51-1 | カバレッジ計算 | 実行行数/実行可能行数でカバレッジ率を算出 | 常時 |
| BR-51-2 | 閾値判定 | デフォルト90%未満でfail表示 | カバレッジ有効時 |
| BR-51-3 | 未実行関数処理 | 未実行関数内の行は実行可能として計上 | 関数カバレッジ計算時 |

### 計算ロジック

```
行カバレッジ率 = executable_lines ∩ lines_which_have_executed / executable_lines
関数カバレッジ率 = functions_which_have_executed.count() / functions.len
ステートメントカバレッジ率 = stmts_which_have_executed.count() / stmts.len
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | OutOfMemory | メモリ不足 | カバレッジ収集を中止 |
| - | SourceMapError | ソースマップ解析失敗 | ソースマップなしで継続 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし（データベース操作なし）。

## パフォーマンス要件

- カバレッジ収集によるテスト実行時間のオーバーヘッド：10-20%以内
- 大規模プロジェクト（1000ファイル以上）でもレポート生成は10秒以内

## セキュリティ考慮事項

- カバレッジレポートにソースコードパスが含まれるため、公開環境での出力に注意
- LCOV出力ファイルのパーミッション設定

## 備考

- JSCは分岐カバレッジ（ブランチカバレッジ）を直接サポートしていないため、現時点では未対応
- 関数名の取得もJSC側で未サポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、カバレッジレポートのデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CodeCoverage.zig | `src/sourcemap/CodeCoverage.zig` | Report構造体、Bitsetによる行管理 |

**読解のコツ**: Zigの`packed struct`とビットセット操作を理解すること。`Report`構造体が全体の中心となる。

**主要処理フロー**:
- **17-26行目**: Report構造体の定義。executable_lines、lines_which_have_executed等のフィールド
- **28-41行目**: linesCoverageFraction関数。行カバレッジ率の計算ロジック
- **53-59行目**: functionCoverageFraction関数。関数カバレッジ率の計算

#### Step 2: エントリーポイントを理解する

テストコマンドからカバレッジ機能がどのように呼び出されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | test_command.zig | `src/cli/test_command.zig` | --coverageオプションの処理 |
| 2-2 | Arguments.zig | `src/cli/Arguments.zig` | コマンドライン引数パース |

**主要処理フロー**:
- test_command.zigでcoverageオプションが有効な場合、カバレッジ収集モードに入る

#### Step 3: カバレッジ生成処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CodeCoverage.zig | `src/sourcemap/CodeCoverage.zig` | ByteRangeMapping、Report.generate |

**主要処理フロー**:
- **358-406行目**: ByteRangeMapping構造体。ソースファイルとバイトオフセットのマッピング
- **374-391行目**: generate関数。バイト範囲マッピングの生成
- **320-348行目**: Report.generate関数。JSCからカバレッジ情報を取得

#### Step 4: レポート出力を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CodeCoverage.zig | `src/sourcemap/CodeCoverage.zig` | Text出力、LCOV出力 |

**主要処理フロー**:
- **61-208行目**: Text.writeFormat関数。テキスト形式レポートの出力
- **210-262行目**: Lcov.writeFormat関数。LCOV形式レポートの出力

### プログラム呼び出し階層図

```
test_command.zig (テストコマンド実行)
    │
    ├─ Arguments.zig (--coverage オプション解析)
    │
    └─ CodeCoverage.zig
           ├─ ByteRangeMapping.generate
           │      └─ LineOffsetTable生成
           │
           ├─ Report.generate
           │      └─ CodeCoverage__withBlocksAndFunctions (JSC連携)
           │
           └─ Report出力
                  ├─ Text.writeFormat
                  └─ Lcov.writeFormat
```

### データフロー図

```
[入力]                    [処理]                           [出力]

JSC基本ブロック情報 ──▶ ByteRangeMapping.generate ──▶ 行マッピングデータ
                              │
ソースファイル ─────────▶ LineOffsetTable生成 ──────▶ 行オフセットテーブル
                              │
ソースマップ ──────────▶ マッピング変換 ───────────▶ 元ファイル行番号
                              │
                              ▼
                       Report.generate
                              │
                              ├──▶ テキストレポート (stdout)
                              └──▶ LCOVレポート (ファイル)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CodeCoverage.zig | `src/sourcemap/CodeCoverage.zig` | ソース | カバレッジ計算・レポート生成の中核 |
| test_command.zig | `src/cli/test_command.zig` | ソース | テストコマンド実装、カバレッジオプション処理 |
| Arguments.zig | `src/cli/Arguments.zig` | ソース | CLI引数パース |
| sourcemap.zig | `src/sourcemap/sourcemap.zig` | ソース | ソースマップ解析 |
| bunfig.zig | `src/bunfig.zig` | ソース | 設定ファイル読み込み（coverage設定含む） |
