# 機能設計書 55-Bun.serve

## 概要

本ドキュメントは、Bunの高性能HTTPサーバー機能（Bun.serve）の設計仕様を記述する。

### 本機能の処理概要

Bun.serveは、uWebSockets（uws）をベースとした高性能HTTPサーバーを提供する。HTTP/1.1、HTTPS（TLS）、WebSocketをサポートし、シンプルなAPIで高スループットなWebサーバーを構築できる。

**業務上の目的・背景**：Node.jsのhttpモジュールやExpressと比較して、大幅に高速なHTTPサーバーを提供する。フレームワークなしでも高性能なAPIサーバーやWebアプリケーションを構築可能。

**機能の利用シーン**：APIサーバー構築、静的ファイル配信、WebSocketサーバー、リバースプロキシ、開発サーバー。

**主要な処理内容**：
1. サーバー設定オプションの解析（port、hostname、TLS等）
2. uWebSocketsアプリケーションの初期化
3. ルーティングとハンドラーの登録
4. リクエスト/レスポンス処理
5. WebSocketアップグレード対応

**関連システム・外部連携**：uWebSockets（C++ライブラリ）、OpenSSL（TLS）、ファイルシステム。

**権限による制御**：特権ポート（1024未満）へのバインドには管理者権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | CLIベースのサーバー起動 |

## 機能種別

サーバー機能 / ネットワーク

## 入力仕様

### 入力パラメータ（Bun.serve オプション）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| fetch | function | Yes | リクエストハンドラー関数 | 必須 |
| port | number | No | ポート番号（デフォルト: 3000） | 0-65535 |
| hostname | string | No | ホスト名（デフォルト: "0.0.0.0"） | 有効なホスト名/IP |
| development | boolean | No | 開発モード | - |
| tls | object | No | TLS設定（証明書、鍵等） | - |
| websocket | object | No | WebSocket設定 | - |
| unix | string | No | Unixソケットパス | 有効なパス |
| idleTimeout | number | No | アイドルタイムアウト（秒） | 0-255 |
| maxRequestBodySize | number | No | 最大リクエストボディサイズ | - |
| reusePort | boolean | No | ポート再利用 | - |

### 入力データソース

- HTTPリクエスト（Method、URL、Headers、Body）
- WebSocket接続リクエスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Server | object | サーバーインスタンス |
| Response | Response | HTTPレスポンス |

### サーバーインスタンスのプロパティ

```typescript
interface Server {
  port: number;
  hostname: string;
  development: boolean;
  pendingRequests: number;
  url: URL;

  stop(closeActiveConnections?: boolean): void;
  fetch(request: Request): Promise<Response>;
  upgrade(req: Request, options?: object): boolean;
  publish(topic: string, message: string | ArrayBuffer, compress?: boolean): number;
  reload(options: object): void;
  timeout(request: Request, seconds: number): void;
}
```

## 処理フロー

### 処理シーケンス

```
1. Bun.serve()呼び出し
   └─ ServerConfigの解析
2. uws.NewApp()でアプリケーション作成
   └─ SSL有効時はHTTPSアプリ
3. ルートハンドラー登録
   └─ fetch関数をコールバックとして設定
4. listen()でソケットバインド
   └─ ポート/Unixソケットに接続待機
5. リクエスト受信
   └─ RequestContext生成
6. fetch()ハンドラー呼び出し
   └─ Responseを返却
7. レスポンス送信
   └─ ストリーミング対応
```

### フローチャート

```mermaid
flowchart TD
    A[Bun.serve呼び出し] --> B[ServerConfig解析]
    B --> C{TLS有効?}
    C -->|Yes| D[HTTPSアプリ作成]
    C -->|No| E[HTTPアプリ作成]
    D --> F[ルートハンドラー登録]
    E --> F
    F --> G[listen開始]
    G --> H{リクエスト受信}
    H --> I[RequestContext生成]
    I --> J{WebSocket Upgrade?}
    J -->|Yes| K[WebSocket処理]
    J -->|No| L[fetch呼び出し]
    L --> M{Response種別}
    M -->|通常| N[レスポンス送信]
    M -->|ストリーム| O[ストリーミング送信]
    K --> H
    N --> H
    O --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-1 | fetch必須 | fetchハンドラーは必須パラメータ | 常時 |
| BR-55-2 | デフォルトポート | port未指定時は3000 | port未指定時 |
| BR-55-3 | Unixソケット排他 | unixオプション指定時はport/hostnameは無視 | unix指定時 |
| BR-55-4 | WebSocket要upgrade | WebSocket使用時はwebsocketオプション必須 | WebSocket使用時 |

### サーバータイプ

```
Server種別 = protocol_enum × development_kind
  where protocol_enum ∈ {http, https}
        development_kind ∈ {debug, production}
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| EADDRINUSE | PortInUse | ポートが使用中 | 別のポートを指定 |
| EACCES | PermissionDenied | 特権ポートへの権限なし | 管理者権限で実行 |
| - | TLSError | 証明書/鍵の読み込み失敗 | 設定を確認 |

### リトライ仕様

リトライは行わない。サーバー起動失敗時は即座にエラーを返す。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- リクエスト処理：マイクロ秒オーダー
- 同時接続：数万接続をサポート
- メモリ効率：リクエストプールによる再利用

## セキュリティ考慮事項

- TLS 1.2/1.3のサポート
- HTTP Strict Transport Security（HSTS）対応
- クライアントエラーハンドリング（on_clienterror）

## 備考

- uWebSocketsはC++で実装された高性能HTTPライブラリ
- development=trueで詳細なエラー出力
- inspector_server_idでデバッガー連携

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、サーバーの設定とインスタンス構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | server.zig | `src/bun.js/api/server.zig` | NewServer関数、ThisServer構造体 |
| 1-2 | ServerConfig.zig | `src/bun.js/api/server/ServerConfig.zig` | サーバー設定構造体 |

**読解のコツ**: `NewServer`はジェネリック関数で、protocol_enum（http/https）とdevelopment_kind（debug/production）の組み合わせで4種類のサーバー型を生成する。

**主要処理フロー**:
- **508-528行目**: NewServer関数。protocol_enumとdevelopment_kindでサーバー型を決定
- **533-566行目**: ThisServer構造体。app、listener、config等のフィールド

#### Step 2: エントリーポイントを理解する

Bun.serve()がどのように呼び出されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BunObject.zig | `src/bun.js/api/BunObject.zig` | serve関数の登録 |

**主要処理フロー**:
- **33行目**: `pub const serve = toJSCallback(Bun.serve);`

#### Step 3: リクエスト処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | server.zig | `src/bun.js/api/server.zig` | RequestContext、fetch処理 |

**主要処理フロー**:
- **530行目**: `RequestContext = NewRequestContext(ssl_enabled, debug_mode, @This())`
- **652-678行目**: timeout関数。リクエストタイムアウト設定

#### Step 4: WebSocket処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | server.zig | `src/bun.js/api/server.zig` | onUpgrade関数 |
| 4-2 | ServerWebSocket.zig | `src/bun.js/api/server/ServerWebSocket.zig` | WebSocket実装 |

**主要処理フロー**:
- **746-999行目**: onUpgrade関数。WebSocketアップグレード処理
- **700-744行目**: publish関数。WebSocketメッセージブロードキャスト

### プログラム呼び出し階層図

```
Bun.serve() (BunObject.zig)
    │
    ├─ ServerConfig解析
    │      └─ ServerConfig.zig
    │
    ├─ NewServer (server.zig)
    │      ├─ uws.NewApp (HTTP/HTTPS)
    │      ├─ ルートハンドラー登録
    │      └─ listen開始
    │
    ├─ リクエスト受信
    │      └─ RequestContext生成
    │             └─ fetch呼び出し
    │                    └─ Response返却
    │
    └─ WebSocket処理
           └─ ServerWebSocket.zig
                  ├─ onOpen
                  ├─ onMessage
                  └─ onClose
```

### データフロー図

```
[入力]                    [処理]                           [出力]

Bun.serve(options) ──▶ ServerConfig解析 ──────▶ ThisServer生成
                              │
                              ▼
                        uws.NewApp
                              │
                              ▼
HTTPリクエスト ────────▶ RequestContext ──────────▶ Response送信
                              │
                              └─ fetch(request)
                                     │
                                     ▼
                              Response/ストリーム

WebSocket接続 ─────────▶ onUpgrade ───────────▶ ServerWebSocket
                                                    │
                                                    ├─ onMessage
                                                    └─ publish
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| server.zig | `src/bun.js/api/server.zig` | ソース | HTTPサーバーの中核実装 |
| ServerConfig.zig | `src/bun.js/api/server/ServerConfig.zig` | ソース | サーバー設定構造体 |
| ServerWebSocket.zig | `src/bun.js/api/server/ServerWebSocket.zig` | ソース | WebSocket実装 |
| StaticRoute.zig | `src/bun.js/api/server/StaticRoute.zig` | ソース | 静的ルート |
| FileRoute.zig | `src/bun.js/api/server/FileRoute.zig` | ソース | ファイルルート |
| NodeHTTPResponse.zig | `src/bun.js/api/server/NodeHTTPResponse.zig` | ソース | Node.js互換Response |
| BunObject.zig | `src/bun.js/api/BunObject.zig` | ソース | Bun.serve登録 |
