# 機能設計書 57-Bun.file

## 概要

本ドキュメントは、BunのファイルI/O操作API（Bun.file）の設計仕様を記述する。

### 本機能の処理概要

Bun.fileは、ファイルシステムへのアクセスを提供するBlobライクなAPIである。ファイルを遅延読み込みし、必要な時にのみディスクアクセスを行う。Blobインターフェースを実装しており、text()、arrayBuffer()、stream()等のメソッドが利用可能。

**業務上の目的・背景**：ファイル操作を簡潔かつ効率的に行えるAPIを提供する。Node.jsのfs APIと比較して、よりモダンでPromiseベースのインターフェースを持つ。

**機能の利用シーン**：設定ファイルの読み込み、ログファイルの出力、静的アセットの配信、ファイルのストリーミング処理。

**主要な処理内容**：
1. Bun.file()でBlobオブジェクト生成
2. Storeでファイル参照を管理
3. 遅延読み込み（必要時に実際のI/O実行）
4. text/arrayBuffer/stream等のメソッドでデータ取得
5. writerでの書き込みストリーム生成

**関連システム・外部連携**：ファイルシステム、libuv（Windows）、io_uring（Linux）。

**権限による制御**：ファイルシステムのパーミッションに従う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | プログラマティックなファイル操作 |

## 機能種別

ファイルI/O

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | string/URL/number | Yes | ファイルパスまたはファイルディスクリプタ | 有効なパス |
| options | object | No | オプション | - |
| options.type | string | No | MIMEタイプ | 有効なMIMEタイプ |

### 入力データソース

- ファイルシステム
- ファイルディスクリプタ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| BunFile | Blob | ファイル参照を持つBlobオブジェクト |

### BunFileプロパティ・メソッド

```typescript
interface BunFile extends Blob {
  name: string;
  lastModified: number;
  exists(): Promise<boolean>;
  text(): Promise<string>;
  json(): Promise<any>;
  arrayBuffer(): Promise<ArrayBuffer>;
  bytes(): Promise<Uint8Array>;
  stream(): ReadableStream;
  writer(options?: object): FileSink;
}
```

## 処理フロー

### 処理シーケンス

```
1. Bun.file()呼び出し
   └─ constructBunFile関数
2. Blob構造体生成
   └─ Store.File参照を設定
3. 遅延評価
   └─ サイズ等は必要時に取得
4. データ取得（text/arrayBuffer等）
   └─ ReadFile.createで非同期読み込み
5. ストリーム取得
   └─ stream()でReadableStream生成
```

### フローチャート

```mermaid
flowchart TD
    A[Bun.file呼び出し] --> B[Blob生成]
    B --> C[Store.File参照設定]
    C --> D{操作種別}
    D -->|exists| E[ファイル存在確認]
    D -->|text/json/arrayBuffer| F[ReadFile.create]
    D -->|stream| G[ReadableStream生成]
    D -->|writer| H[FileSink生成]
    F --> I[Promise返却]
    G --> J[ストリーム返却]
    H --> K[FileSink返却]
    E --> L[boolean返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-57-1 | 遅延読み込み | ファイル内容は実際に必要になるまで読み込まない | 常時 |
| BR-57-2 | MIMEタイプ推測 | 拡張子からMIMEタイプを自動推測 | type未指定時 |
| BR-57-3 | サイズ遅延取得 | ファイルサイズは必要時にstat()で取得 | size参照時 |
| BR-57-4 | S3対応 | s3://スキームでS3オブジェクトにアクセス可能 | S3パス指定時 |

### Store種別

```
Store.data = union {
  file: File,     // ローカルファイル
  bytes: Bytes,   // インメモリ
  s3: S3,         // S3オブジェクト
}
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ENOENT | FileNotFound | ファイルが存在しない | パスを確認 |
| EACCES | PermissionDenied | 読み取り権限なし | 権限を確認 |
| EISDIR | IsDirectory | ディレクトリを開こうとした | パスを確認 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 遅延読み込みによりメモリ効率を最適化
- io_uring（Linux）/libuv（Windows）による非同期I/O
- ファイルコピーは最適化パスを使用

## セキュリティ考慮事項

- ファイルシステムのパーミッションチェック
- パストラバーサル対策

## 備考

- BunFileはBlobのサブタイプ
- Store.Fileはファイルパス、FD、オフセット、サイズを保持
- 最大ファイルサイズは9ペタバイト（u52）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、BlobとStore構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Blob.zig | `src/bun.js/webcore/Blob.zig` | Blob構造体、Store参照 |
| 1-2 | Store.zig | `src/bun.js/webcore/blob/Store.zig` | Store.File、Store.data union |

**読解のコツ**: BlobはStoreへの参照を持ち、Storeの種別（file/bytes/s3）によって動作が変わる。

**主要処理フロー**:
- **1-10行目** (Blob.zig): ファイル概要コメント
- **26-52行目** (Blob.zig): Blob構造体フィールド（size, offset, store, content_type等）
- **60-61行目** (Blob.zig): SizeType = u52、最大9ペタバイト

#### Step 2: ファイル読み込みを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Blob.zig | `src/bun.js/webcore/Blob.zig` | doReadFile関数 |
| 2-2 | read_file.zig | `src/bun.js/webcore/blob/read_file.zig` | ReadFile.create |

**主要処理フロー**:
- **125-170行目** (Blob.zig): doReadFile関数。非同期ファイル読み込みのPromise生成

#### Step 3: BunObject登録を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | BunObject.zig | `src/bun.js/api/BunObject.zig` | file関数の登録 |

**主要処理フロー**:
- **20行目**: `pub const file = toJSCallback(WebCore.Blob.constructBunFile);`

### プログラム呼び出し階層図

```
Bun.file(path) (BunObject.zig)
    │
    └─ WebCore.Blob.constructBunFile
           │
           ├─ Blob生成
           │      └─ Store.File参照設定
           │
           └─ Blobメソッド
                  ├─ text() → doReadFile → ReadFile.create → Promise
                  ├─ arrayBuffer() → doReadFile → Promise
                  ├─ stream() → ReadableStream生成
                  ├─ exists() → ファイル存在確認 → Promise<boolean>
                  └─ writer() → FileSink生成
```

### データフロー図

```
[入力]                    [処理]                           [出力]

ファイルパス ──────────▶ Blob生成 ──────────────▶ BunFile
                              │
                              ▼
                        Store.File設定
                              │
text()呼び出し ────────▶ doReadFile ─────────────▶ Promise<string>
                              │
                              ▼
                        ReadFile.create
                              │
                              ▼
stream()呼び出し ──────▶ ReadableStream生成 ────▶ ReadableStream
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Blob.zig | `src/bun.js/webcore/Blob.zig` | ソース | Blob構造体、ファイル読み込み |
| Store.zig | `src/bun.js/webcore/blob/Store.zig` | ソース | ストレージ管理 |
| read_file.zig | `src/bun.js/webcore/blob/read_file.zig` | ソース | ファイル読み込み実装 |
| write_file.zig | `src/bun.js/webcore/blob/write_file.zig` | ソース | ファイル書き込み実装 |
| copy_file.zig | `src/bun.js/webcore/blob/copy_file.zig` | ソース | ファイルコピー最適化 |
| BunObject.zig | `src/bun.js/api/BunObject.zig` | ソース | Bun.file登録 |
