# 機能設計書 58-Bun.write

## 概要

本ドキュメントは、Bunのファイル書き込みAPI（Bun.write）の設計仕様を記述する。

### 本機能の処理概要

Bun.writeは、ファイルへのデータ書き込みを簡潔に行えるAPIである。文字列、Blob、ArrayBuffer等の様々なデータ型を受け付け、効率的にファイルに書き込む。ファイルコピーの最適化パスも提供する。

**業務上の目的・背景**：ファイル書き込みをワンライナーで実行できるシンプルなAPIを提供する。Node.jsのfs.writeFileと比較して、より直感的でPromiseベースのインターフェースを持つ。

**機能の利用シーン**：ログファイルの出力、設定ファイルの保存、データのエクスポート、ファイルのコピー。

**主要な処理内容**：
1. Bun.write()呼び出し
2. 書き込みデータの種別判定（string/Blob/ArrayBuffer等）
3. 書き込み先の判定（ファイルパス/BunFile）
4. 最適化パスの選択（ファイルコピー等）
5. 非同期書き込み実行

**関連システム・外部連携**：ファイルシステム、io_uring（Linux）、libuv（Windows）。

**権限による制御**：ファイルシステムのパーミッションに従う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | プログラマティックなファイル書き込み |

## 機能種別

ファイルI/O

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| destination | string/BunFile/number | Yes | 書き込み先 | 有効なパス/FD |
| data | string/Blob/ArrayBuffer/Response | Yes | 書き込みデータ | - |

### サポートされるデータ型

- string
- Blob/BunFile
- ArrayBuffer/TypedArray
- Response
- ReadableStream

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| bytesWritten | Promise\<number\> | 書き込まれたバイト数 |

### 戻り値

```typescript
Bun.write(destination, data): Promise<number>
```

## 処理フロー

### 処理シーケンス

```
1. Bun.write()呼び出し
   └─ writeFile関数
2. 書き込み先の判定
   └─ ファイルパス/BunFile/FD
3. データ種別の判定
   └─ string/Blob/ArrayBuffer等
4. 最適化パスの選択
   └─ ファイルtoファイルならcopy_file使用
5. 非同期書き込み実行
   └─ WriteFile.create
6. Promise解決
   └─ 書き込みバイト数を返却
```

### フローチャート

```mermaid
flowchart TD
    A[Bun.write呼び出し] --> B{書き込み先種別}
    B -->|ファイルパス| C[パス解析]
    B -->|BunFile| D[Blob.Store取得]
    B -->|FD| E[FD使用]
    C --> F{データ種別}
    D --> F
    E --> F
    F -->|string| G[文字列書き込み]
    F -->|Blob/BunFile| H{コピー最適化可能?}
    F -->|ArrayBuffer| I[バイナリ書き込み]
    F -->|Response| J[ストリーム書き込み]
    H -->|Yes| K[copy_file使用]
    H -->|No| L[通常書き込み]
    G --> M[Promise返却]
    I --> M
    J --> M
    K --> M
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-58-1 | コピー最適化 | BunFile同士の場合はファイルコピー最適化を使用 | Store.File to Store.File |
| BR-58-2 | ストリーミング | Responseの場合はストリームから書き込み | Response指定時 |
| BR-58-3 | 自動エンコード | 文字列はUTF-8でエンコード | string指定時 |
| BR-58-4 | 親ディレクトリ作成 | 存在しない親ディレクトリは自動作成しない | 常時 |

### 最適化パス

```
BunFile → BunFile: copy_file.copyFilePosix/copyFileNonBlocking
string → File: WriteFile.create
ArrayBuffer → File: WriteFile.create
Response → File: ストリーム書き込み
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ENOENT | FileNotFound | 親ディレクトリが存在しない | ディレクトリを作成 |
| EACCES | PermissionDenied | 書き込み権限なし | 権限を確認 |
| ENOSPC | NoSpace | ディスク容量不足 | 空き容量確保 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし（ファイルシステムの原子性に依存）。

## パフォーマンス要件

- io_uring（Linux）による非同期I/O
- copy_file_range（Linux）によるゼロコピー最適化
- バッファリングによるシステムコール削減

## セキュリティ考慮事項

- ファイルシステムのパーミッションチェック
- パストラバーサル対策

## 備考

- Bun.writeはBlob.writeFileで実装されている
- ファイルコピー最適化はcopy_file.zigで実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

まず、Bun.writeの登録箇所を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BunObject.zig | `src/bun.js/api/BunObject.zig` | write関数の登録 |

**読解のコツ**: `pub const write = toJSCallback(jsc.WebCore.Blob.writeFile);`を確認する。

**主要処理フロー**:
- **42行目**: writeはBlob.writeFileにマッピング

#### Step 2: 書き込み処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Blob.zig | `src/bun.js/webcore/Blob.zig` | writeFile関数 |
| 2-2 | write_file.zig | `src/bun.js/webcore/blob/write_file.zig` | WriteFile.create |

**主要処理フロー**:
- Blob.zigでデータ型の判定と分岐
- write_file.zigで実際の書き込み処理

#### Step 3: コピー最適化を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | copy_file.zig | `src/bun.js/webcore/blob/copy_file.zig` | copyFilePosix、copyFileNonBlocking |

**主要処理フロー**:
- Store.File to Store.Fileの場合にコピー最適化を適用
- copy_file_range（Linux）やsendfile等のシステムコールを使用

### プログラム呼び出し階層図

```
Bun.write(dest, data) (BunObject.zig)
    │
    └─ WebCore.Blob.writeFile
           │
           ├─ 書き込み先判定
           │      └─ パス/BunFile/FD
           │
           ├─ データ種別判定
           │      └─ string/Blob/ArrayBuffer/Response
           │
           └─ 書き込み実行
                  ├─ copy_file (BunFile to BunFile)
                  │      └─ copyFilePosix/copyFileNonBlocking
                  │
                  └─ write_file (その他)
                         └─ WriteFile.create → Promise
```

### データフロー図

```
[入力]                    [処理]                           [出力]

書き込み先パス ────────▶ パス解析 ──────────────▶ ファイルハンドル
書き込みデータ ────────▶ 型判定 ────────────────┐
                              │               │
                              ▼               │
                        最適化パス選択 ◀───────┘
                              │
                              ├─ copy_file (BunFile to BunFile)
                              │       └─ copy_file_range
                              │
                              └─ write_file (その他)
                                      └─ WriteFile.create
                                             │
                                             ▼
                                      Promise<number>
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| BunObject.zig | `src/bun.js/api/BunObject.zig` | ソース | Bun.write登録 |
| Blob.zig | `src/bun.js/webcore/Blob.zig` | ソース | writeFile実装 |
| write_file.zig | `src/bun.js/webcore/blob/write_file.zig` | ソース | 書き込み処理 |
| copy_file.zig | `src/bun.js/webcore/blob/copy_file.zig` | ソース | コピー最適化 |
