# 機能設計書 60-Bun.Transpiler

## 概要

本ドキュメントは、BunのトランスパイラAPI（Bun.Transpiler）の設計仕様を記述する。

### 本機能の処理概要

Bun.Transpilerは、JavaScript/TypeScript/JSXのトランスパイルを行うAPIである。単一ファイルのトランスパイルに特化しており、コードの変換・最適化・分析を提供する。

**業務上の目的・背景**：ビルドツールやエディタ拡張において、個別ファイルのトランスパイルが必要となるケースに対応する。Bun.buildと異なり、バンドリングを行わず単一ファイルの変換に集中する。

**機能の利用シーン**：ランタイムでのコード変換、REPLでの即時実行、マクロの展開、コードの静的解析。

**主要な処理内容**：
1. new Bun.Transpiler(config)でインスタンス生成
2. transform/transformSyncでコード変換
3. scan/scanImportsでコード解析
4. minify/define/externalの各種最適化

**関連システム・外部連携**：Bunのjs_parser、js_printerモジュール。

**権限による制御**：特になし（メモリ上での変換処理）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | プログラマティックなコード変換 |

## 機能種別

コード変換

## 入力仕様

### コンストラクタオプション

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| loader | string | No | デフォルトローダー（"jsx", "tsx", "ts", "js"） | JavaScript系のみ |
| target | string | No | 出力ターゲット（"browser", "bun", "node"） | - |
| define | object | No | コンパイル時定数定義 | - |
| external | string[] | No | 外部モジュール指定 | - |
| tsconfig | string/object | No | TypeScript設定 | - |
| macro | boolean/object | No | マクロ設定 | - |
| minify | boolean/object | No | minify設定 | - |
| sourcemap | boolean/string | No | ソースマップ設定 | - |
| treeShaking | boolean | No | Tree Shaking有効化 | - |
| trimUnusedImports | boolean | No | 未使用インポート削除 | - |
| deadCodeElimination | boolean | No | デッドコード削除 | - |
| exports | object | No | エクスポート置換設定 | - |

### minifyオプション詳細

```typescript
interface MinifyOptions {
  whitespace?: boolean;   // 空白削除
  syntax?: boolean;       // 構文最適化
  identifiers?: boolean;  // 識別子短縮
}
```

### exportsオプション詳細

```typescript
interface ExportsOptions {
  eliminate?: string[];   // 削除するエクスポート
  replace?: Record<string, string | number | boolean | null | undefined>;
}
```

## 出力仕様

### transform/transformSyncの出力

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | Promise\<string\> / string | トランスパイル結果 |

### scanの出力

| 項目名 | 型 | 説明 |
|--------|-----|------|
| imports | array | インポート情報 |
| exports | array | エクスポート情報 |

### scanImportsの出力

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | array | インポート情報のみ |

## 処理フロー

### 処理シーケンス

```
1. new Bun.Transpiler(config)
   └─ Config.fromJS でオプション解析
2. transformSync(code, loader)呼び出し
   └─ getParseResult でパース
   └─ JSPrinter.BufferPrinter で出力
3. transform(code, loader)呼び出し（非同期）
   └─ TransformTask.create でタスク生成
   └─ ConcurrentPromiseTask でスケジュール
4. scan(code, loader)呼び出し
   └─ getParseResult でパース
   └─ namedImportsToJS, namedExportsToJS で結果変換
```

### フローチャート

```mermaid
flowchart TD
    A[new Bun.Transpiler] --> B[Config解析]
    B --> C{オプション検証}
    C -->|成功| D[Transpiler生成]
    C -->|失敗| E[エラーthrow]
    D --> F{メソッド呼び出し}
    F -->|transformSync| G[同期トランスパイル]
    F -->|transform| H[非同期トランスパイル]
    F -->|scan| I[コード解析]
    F -->|scanImports| J[インポート解析]
    G --> K[パース]
    H --> L[TransformTask生成]
    L --> M[スケジュール]
    M --> N[Promise返却]
    K --> O[プリント]
    O --> P[結果返却]
    I --> Q[パース]
    Q --> R[exports/imports抽出]
    R --> S[結果オブジェクト返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-60-1 | デフォルトローダー | デフォルトはjsxローダー | loader未指定時 |
| BR-60-2 | JavaScript系限定 | JavaScript/TypeScript系ファイルのみ対応 | 常時 |
| BR-60-3 | デフォルトターゲット | デフォルトはbrowserターゲット | target未指定時 |
| BR-60-4 | REPLモード | replModeでDCE無効化 | replMode=true時 |
| BR-60-5 | オブジェクトリテラル | REPLモードで{}を()で囲む | replMode=true かつ {}で始まる時 |
| BR-60-6 | TreeShaking連動 | exports指定時はtreeShaking自動有効 | exports指定時 |

### 対応ローダー

```
jsx  - JSX (デフォルト)
tsx  - TypeScript + JSX
ts   - TypeScript
js   - JavaScript
```

### ターゲット

```
browser - ブラウザ向け（デフォルト）
bun     - Bun向け
node    - Node.js向け
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ParseError | 構文エラー | コードのパースに失敗 | コードを修正 |
| InvalidArgument | 引数エラー | 無効なオプション指定 | オプションを確認 |
| UnsupportedLoader | ローダーエラー | JavaScript系以外のローダー指定 | ローダーを変更 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- RefCountによる参照カウント管理
- MimallocArenaによるメモリプール
- BufferWriterの再利用による最適化
- ConcurrentPromiseTaskによる非同期処理

## セキュリティ考慮事項

- マクロ展開時のコード実行に注意
- defineによる値インジェクションの検証

## 備考

- transformは非同期版、transformSyncは同期版
- scanはimports/exports両方を返却
- scanImportsはimportsのみを高速に取得

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、JSTranspiler構造体とConfigを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | JSTranspiler構造体、Config構造体 |

**読解のコツ**: JSTranspilerはbun.transpiler.Transpilerをラップし、Configで設定を保持する。RefCountで参照カウント管理される。

**主要処理フロー**:
- **1-19行目**: JSTranspiler構造体定義。transpiler、config、scan_pass_result等のフィールド
- **27-436行目**: Config構造体。transform options、loader、minify設定等
- **47-435行目**: Config.fromJS関数。JSオブジェクトからConfig生成

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | constructor関数 |
| 2-2 | BunObject.zig | `src/bun.js/api/BunObject.zig` | Transpiler登録 |

**主要処理フロー**:
- **651-732行目**: constructor関数。JSTranspiler生成、Config解析、Transpiler初期化
- **BunObject.zig 69行目**: Bun.Transpilerの登録
- **BunObject.zig 1255-1256行目**: getTranspilerConstructor

#### Step 3: transform処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | transform、transformSync関数 |
| 3-2 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | TransformTask構造体 |

**主要処理フロー**:
- **886-919行目**: transform関数（非同期）。TransformTask.createでタスク生成
- **921-1038行目**: transformSync関数（同期）。getParseResultでパース、BufferPrinterで出力
- **445-593行目**: TransformTask構造体。run、then、deinitメソッド

#### Step 4: scan処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | scan、scanImports関数 |

**主要処理フロー**:
- **815-884行目**: scan関数。パース後にimports/exportsを抽出
- **1082-1170行目**: scanImports関数。高速なインポート解析
- **1040-1061行目**: namedExportsToJS関数
- **1063-1080行目**: namedImportsToJS関数

#### Step 5: ヘルパー関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | getParseResult、isLikelyObjectLiteral |

**主要処理フロー**:
- **780-813行目**: getParseResult関数。パースオプション設定とパース実行
- **754-778行目**: isLikelyObjectLiteral関数。REPLモードでのオブジェクトリテラル判定

### プログラム呼び出し階層図

```
Bun.Transpiler (BunObject.zig)
    │
    └─ getTranspilerConstructor
           │
           └─ JSTranspiler.js.getConstructor
                  │
                  └─ JSTranspiler.constructor
                         │
                         ├─ Config.fromJS
                         │      ├─ define解析
                         │      ├─ external解析
                         │      ├─ loader解析
                         │      ├─ target解析
                         │      ├─ tsconfig解析
                         │      ├─ macro解析
                         │      ├─ minify解析
                         │      ├─ sourcemap解析
                         │      └─ exports解析
                         │
                         └─ Transpiler.init
                                ├─ configureLinkerWithAutoJSX
                                └─ configureDefines

transpiler.transform(code, loader)
    │
    └─ TransformTask.create
           │
           ├─ AsyncTransformTask.createOnJSThread
           │
           └─ TransformTask.run
                  │
                  ├─ JSParser.parse
                  │
                  └─ JSPrinter.print
                         │
                         └─ Promise.resolve(output)

transpiler.transformSync(code, loader)
    │
    └─ getParseResult
           │
           ├─ JSParser.parse
           │
           └─ JSPrinter.BufferPrinter.print
                  │
                  └─ return string

transpiler.scan(code, loader)
    │
    └─ getParseResult
           │
           └─ namedImportsToJS / namedExportsToJS
                  │
                  └─ return { imports, exports }
```

### データフロー図

```
[入力]                    [処理]                           [出力]

コード文字列 ──────────▶ ローダー判定 ─────────────────────┐
                              │                          │
オプション ────────────▶ Config.fromJS ───────────────────┤
                              │                          │
                              ▼                          │
                        Transpiler.init                  │
                              │                          │
                              ├─ define設定              │
                              ├─ external設定            │
                              ├─ minify設定              │
                              └─ target設定              │
                                    │                    │
                                    ▼                    │
                              パース処理 ◀───────────────┘
                                    │
                                    ├─ AST生成
                                    │
                                    ▼
                              [transformSync]
                                    │
                                    └─ JSPrinter.print
                                          │
                                          ▼
                                    変換コード (string)

                              [transform]
                                    │
                                    └─ TransformTask
                                          │
                                          ▼
                                    Promise<string>

                              [scan]
                                    │
                                    ├─ import_records抽出
                                    ├─ named_exports抽出
                                    │
                                    ▼
                                    { imports, exports }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| JSTranspiler.zig | `src/bun.js/api/JSTranspiler.zig` | ソース | Transpiler API実装 |
| BunObject.zig | `src/bun.js/api/BunObject.zig` | ソース | Bun.Transpiler登録 |
| transpiler.zig | `src/transpiler.zig` | ソース | 内部Transpiler |
| js_parser.zig | `src/js_parser.zig` | ソース | パーサー |
| js_printer.zig | `src/js_printer.zig` | ソース | プリンター |
| options.zig | `src/options.zig` | ソース | Loader、Target定義 |
