# 機能設計書 62-Bun.sql

## 概要

本ドキュメントは、BunランタイムのPostgreSQLクライアント機能（`Bun.sql`）の設計仕様を記載する。

### 本機能の処理概要

`Bun.sql`は、PostgreSQLデータベースへの高性能な接続と操作を提供するBun組み込みクライアントである。ネイティブZig実装により、外部依存なしに高速なデータベース接続、コネクションプーリング、自動パイプライニングを実現する。

**業務上の目的・背景**：Webアプリケーションやバックエンドサービスでは、PostgreSQLは最も広く使用されるリレーショナルデータベースの1つである。従来のNode.js用PostgreSQLドライバ（pg、postgres.js等）はJavaScript実装のため、パフォーマンス面での制約があった。Bunはネイティブ実装により、これらの課題を解決し、より高速なデータベースアクセスを提供する。

**機能の利用シーン**：
- Webアプリケーションのバックエンドでのデータ永続化
- RESTful APIやGraphQL APIのデータ取得
- バッチ処理でのデータ操作
- リアルタイムアプリケーションでの頻繁なDB操作
- マイクロサービス間のデータ共有

**主要な処理内容**：
1. PostgreSQLサーバーへの接続確立（TCP/TLS/Unix Socket）
2. コネクションプールの管理（自動プーリング）
3. SQLクエリの実行（タグ付きテンプレートリテラル）
4. パラメータ化クエリによる安全なデータ操作
5. トランザクション管理（BEGIN、COMMIT、ROLLBACK、SAVEPOINT）
6. 自動パイプライニングによる性能最適化

**関連システム・外部連携**：PostgreSQLサーバー（外部）、TLS/SSL証明書（任意）

**権限による制御**：PostgreSQLのユーザー認証およびデータベース権限に依存

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI専用であり、直接関連する画面はない |

## 機能種別

データ連携 / CRUD操作 / データベースアクセス

## 入力仕様

### 入力パラメータ

#### Bun.sql() 接続オプション

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| hostname | string | No | ホスト名（デフォルト: localhost） | - |
| port | number | No | ポート番号（デフォルト: 5432） | 正の整数 |
| username | string | No | ユーザー名 | - |
| password | string \| () => string | No | パスワードまたはパスワード取得関数 | - |
| database | string | No | データベース名 | - |
| tls | boolean \| TLSOptions | No | TLS設定 | - |
| sslMode | 'disable' \| 'prefer' \| 'require' \| 'verify-ca' \| 'verify-full' | No | SSLモード | - |
| idleTimeout | number | No | アイドルタイムアウト（ミリ秒） | 正の整数 |
| connectionTimeout | number | No | 接続タイムアウト（ミリ秒、デフォルト: 30000） | 正の整数 |
| maxLifetime | number | No | 接続の最大ライフタイム（ミリ秒） | 正の整数 |
| prepare | boolean | No | プリペアドステートメントの使用（デフォルト: true） | - |
| path | string | No | Unix Socketパス | - |

#### タグ付きテンプレートリテラル

```javascript
const result = await sql`SELECT * FROM users WHERE id = ${userId}`;
```

### 入力データソース

- PostgreSQLサーバー（TCP、TLS、Unix Socket経由）
- 環境変数（DATABASE_URL等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| SQLResultArray | Array | クエリ結果の配列（拡張プロパティ付き） |
| result.command | string | 実行されたコマンド（SELECT, INSERT等） |
| result.count | number | 影響を受けた行数 |

### 出力先

- JavaScript オブジェクト/配列（クエリ結果）

## 処理フロー

### 処理シーケンス

```
1. 接続プールの初期化
   └─ 接続オプションの解析、プールマネージャーの作成
2. 接続の確立
   └─ PostgreSQLサーバーへの接続、認証、ハンドシェイク
3. クエリの準備
   └─ タグ付きテンプレートのパース、パラメータのバインド
4. クエリの送信
   └─ プリペアドステートメントの作成、パイプライニング
5. 結果の受信
   └─ PostgreSQLプロトコルの解析、型変換
6. 結果の返却
   └─ SQLResultArrayとしてPromiseを解決
7. 接続の管理
   └─ プールへの返却、アイドルタイムアウト管理
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{接続済み?}
    B -->|No| C[コネクションプールから接続取得]
    B -->|Yes| D[既存接続を使用]
    C --> E{接続成功?}
    E -->|No| F[エラーを返却]
    E -->|Yes| D
    D --> G[SQLをパース]
    G --> H[パラメータをバインド]
    H --> I{プリペアド?}
    I -->|Yes| J[プリペアドステートメント作成]
    I -->|No| K[シンプルクエリ送信]
    J --> L[パイプラインでExecute]
    K --> L
    L --> M[結果を受信]
    M --> N[型変換]
    N --> O[SQLResultArrayを生成]
    O --> P[Promiseを解決]
    P --> Q[接続をプールに返却]
    Q --> R[終了]
    F --> R
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 自動パイプライニング | 複数クエリを自動的にパイプラインで送信 | enableAutoPipelining: true（デフォルト） |
| BR-002 | コネクションプーリング | 接続を再利用してオーバーヘッドを削減 | 常時 |
| BR-003 | パラメータ化クエリ | テンプレートリテラルの値を自動的にパラメータ化 | sql\`...\`使用時 |
| BR-004 | 型自動変換 | PostgreSQLの型をJavaScript型に自動変換 | 結果取得時 |

### 計算ロジック

- `count`: 影響を受けた行数（INSERT/UPDATE/DELETE）
- コネクションプールサイズ: 動的に管理

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| sql\`...\` | 任意 | SELECT/INSERT/UPDATE/DELETE | クエリを実行 |
| sql.begin() | なし | BEGIN | トランザクション開始 |
| sql.commit() | なし | COMMIT | トランザクションコミット |
| sql.rollback() | なし | ROLLBACK | トランザクションロールバック |

### テーブル別操作詳細

本機能は汎用データベースアクセスのため、特定テーブルに依存しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| PostgresError | データベースエラー | SQL構文エラー、制約違反など | エラーメッセージを確認し、SQL修正 |
| CONNECTION_REFUSED | 接続エラー | サーバーに接続できない | ホスト名/ポートを確認 |
| AUTHENTICATION_FAILED | 認証エラー | 認証情報が不正 | ユーザー名/パスワードを確認 |
| TIMEOUT | タイムアウト | 接続/クエリタイムアウト | タイムアウト設定を調整 |

### リトライ仕様

- 接続失敗時は自動リトライなし（アプリケーション側で実装）
- コネクションプールが枯渇した場合は待機後リトライ

## トランザクション仕様

- `sql.begin()`: トランザクション開始
- `sql.begin('read only')`: 読み取り専用トランザクション
- `sql.savepoint(name)`: セーブポイント作成
- `sql.rollback()`: ロールバック
- `sql.commit()`: コミット
- ネストされたトランザクションはSAVEPOINTで実現

## パフォーマンス要件

- ネイティブZig実装による高速処理
- 自動パイプライニングによるラウンドトリップ削減
- コネクションプーリングによる接続オーバーヘッド削減
- プリペアドステートメントのキャッシュ

## セキュリティ考慮事項

- パラメータ化クエリによるSQLインジェクション防止
- TLS/SSL暗号化接続のサポート
- パスワードのセキュアな管理（関数での遅延評価対応）
- 証明書検証オプション（verify-ca, verify-full）

## 備考

- postgres.js互換のAPI設計を目指している
- SQLiteアダプターと統一インターフェースを提供
- 分散トランザクションはサポートしていない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、PostgreSQL接続とクエリに関わるデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | postgres.ts | `src/js/internal/sql/postgres.ts` | SQLResultArray、SQLCommand、PostgresAdapter構造 |
| 1-2 | postgres.zig | `src/sql/postgres.zig` | Zigバインディングの構造 |

**読解のコツ**: TypeScriptの型定義とZigの構造体を対比しながら、JS-Zig間のデータ受け渡しを理解する。

#### Step 2: エントリーポイントを理解する

PostgreSQLへの接続処理の流れを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | postgres.ts | `src/js/internal/sql/postgres.ts` | createConnection、PooledPostgresConnection |
| 2-2 | PostgresSQLConnection.zig | `src/sql/postgres/PostgresSQLConnection.zig` | 接続状態管理、TLS設定 |

**主要処理フロー**:
1. **473-499行目** (postgres.ts): PooledPostgresConnection.createConnectionで接続オプション解析
2. **1-61行目** (PostgresSQLConnection.zig): 接続の状態管理とタイマー設定

#### Step 3: クエリ実行処理を理解する

SQLクエリの実行フローを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | postgres.ts | `src/js/internal/sql/postgres.ts` | detectCommand、onResolvePostgresQuery |

**主要処理フロー**:
- **383-451行目**: detectCommand()でSQLコマンドを解析
- **236-316行目**: initPostgres()でクエリ解決/拒否コールバックを設定

#### Step 4: コネクションプーリングを理解する

接続プールの管理ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | postgres.ts | `src/js/internal/sql/postgres.ts` | PooledConnectionState、PooledConnectionFlags |
| 4-2 | PostgresSQLConnection.zig | `src/sql/postgres/PostgresSQLConnection.zig` | タイムアウト管理 |

**主要処理フロー**:
- **453-466行目** (postgres.ts): プール接続の状態とフラグ定義
- **108-134行目** (PostgresSQLConnection.zig): タイムアウト処理

### プログラム呼び出し階層図

```
Bun.sql (JavaScript API)
    │
    ├─ PostgresAdapter (postgres.ts)
    │      ├─ PooledPostgresConnection
    │      │      ├─ createConnection()
    │      │      └─ queries管理
    │      └─ normalizeQuery()
    │
    ├─ postgres.zig (Zigバインディング)
    │      ├─ createBinding()
    │      ├─ PostgresSQLConnection
    │      ├─ PostgresSQLContext
    │      └─ PostgresSQLQuery
    │
    └─ PostgresSQLConnection.zig
           ├─ Socket管理
           ├─ TLS設定
           ├─ タイムアウト管理
           └─ バッファ管理
```

### データフロー図

```
[入力]                    [処理]                         [出力]

SQL文字列 + パラメータ ──▶ normalizeQuery() ─────────▶ パラメータ化SQL
                                │
接続オプション ────────────▶ createConnection() ──────▶ Connection
                                │
                                ▼
                         PostgresSQLQuery ──────────▶ サーバー送信
                                │
                                ▼
                         PostgresProtocol ──────────▶ 結果パース
                                │
                                ▼
                         onResolvePostgresQuery() ──▶ SQLResultArray
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| postgres.ts | `src/js/internal/sql/postgres.ts` | ソース | TypeScriptアダプター |
| postgres.zig | `src/sql/postgres.zig` | ソース | Zigバインディングエントリ |
| PostgresSQLConnection.zig | `src/sql/postgres/PostgresSQLConnection.zig` | ソース | 接続管理 |
| PostgresSQLContext.zig | `src/sql/postgres/PostgresSQLContext.zig` | ソース | コンテキスト管理 |
| PostgresSQLQuery.zig | `src/sql/postgres/PostgresSQLQuery.zig` | ソース | クエリ管理 |
| PostgresProtocol.zig | `src/sql/postgres/PostgresProtocol.zig` | ソース | プロトコル解析 |
| PostgresTypes.zig | `src/sql/postgres/PostgresTypes.zig` | ソース | 型変換 |
| query.ts | `src/js/internal/sql/query.ts` | ソース | クエリ基底クラス |
| shared.ts | `src/js/internal/sql/shared.ts` | ソース | 共通ユーティリティ |
| errors.ts | `src/js/internal/sql/errors.ts` | ソース | エラークラス定義 |
