# 機能設計書 71-Bun.hash

## 概要

本ドキュメントは、Bun.hash APIの機能設計を記述する。Bun.hashは、文字列やバイナリデータに対して高速な非暗号学的ハッシュ関数を提供するAPIである。

### 本機能の処理概要

Bun.hashは、入力データに対して様々なハッシュアルゴリズムを適用し、数値またはBigInt形式のハッシュ値を返す機能である。デフォルトではwyhashアルゴリズムを使用するが、複数のハッシュアルゴリズムがサブ関数として提供される。

**業務上の目的・背景**：データの一意性識別、キャッシュキーの生成、ハッシュテーブル実装、データ整合性の簡易チェックなど、高速なハッシュ計算が必要な場面で使用される。暗号学的安全性は必要ないが、高速な処理が求められるユースケースに最適化されている。

**機能の利用シーン**：
- ファイルやデータの変更検知（キャッシュ無効化判定）
- オブジェクトのハッシュ値によるルックアップ
- 分散システムでのデータシャーディング
- デバッグ時のデータ識別

**主要な処理内容**：
1. 入力データ（文字列、Blob、ArrayBuffer、TypedArray等）を受け取る
2. 指定されたハッシュアルゴリズム（デフォルトはwyhash）を適用
3. オプションでシード値を指定可能
4. 計算結果をnumber（32bit）またはBigInt（64bit）で返却

**関連システム・外部連携**：Zigの標準ライブラリ（std.hash）およびzlib（CRC32用）を内部で使用。JavaScriptCoreのバインディングを通じてJavaScriptから利用可能。

**権限による制御**：特に権限による制御は行われない。すべてのコードから利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドライン/API経由での利用（画面なし） |

## 機能種別

計算処理（ハッシュ計算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | string / Blob / ArrayBuffer / TypedArray | Yes | ハッシュ対象のデータ | 空データも許容 |
| seed | number / bigint | No | ハッシュ計算のシード値 | 64bit整数範囲 |

### 入力データソース

API呼び出し時の引数として直接指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| hashValue | number / bigint | ハッシュ値（32bit出力の場合はnumber、64bit出力の場合はbigint） |

### 出力先

API戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. 入力データの型判定
   └─ Blob、ArrayBuffer、TypedArray、文字列を判別
2. データのバイト配列への変換
   └─ 文字列の場合はUTF-8エンコード
3. シード値の取得（オプション）
   └─ 指定がない場合は0を使用
4. ハッシュ関数の実行
   └─ Zigの標準ライブラリまたはzlibを呼び出し
5. 結果の返却
   └─ 32bit結果はnumber、64bit結果はbigintで返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{入力データ型判定}
    B -->|Blob| C[sharedView取得]
    B -->|ArrayBuffer/TypedArray| D[byteSlice取得]
    B -->|その他| E[文字列変換]
    C --> F{シード値あり?}
    D --> F
    E --> F
    F -->|Yes| G[シード値を64bit整数に変換]
    F -->|No| H[シード=0を設定]
    G --> I[ハッシュ関数実行]
    H --> I
    I --> J{32bit出力?}
    J -->|Yes| K[numberで返却]
    J -->|No| L[bigintで返却]
    K --> M[終了]
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-71-01 | デフォルトアルゴリズム | Bun.hash()直接呼び出し時はwyhashを使用 | 常時 |
| BR-71-02 | 32bit出力 | adler32, crc32, cityHash32, xxHash32, murmur32v2, murmur32v3はnumberを返却 | 該当アルゴリズム使用時 |
| BR-71-03 | 64bit出力 | wyhash, cityHash64, xxHash64, xxHash3, murmur64v2, rapidhashはbigintを返却 | 該当アルゴリズム使用時 |

### 計算ロジック

各ハッシュアルゴリズムの実装はZigの標準ライブラリに委譲。CRC32のみzlibのハードウェアアクセラレーション版を使用し、大きなデータは32bit長制限を回避するためチャンク分割して処理。

## データベース操作仕様

### 操作別データベース影響一覧

なし（データベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArguments | TypeError | ArrayBuffer変換エラー | 有効なバッファを指定 |

### リトライ仕様

なし（同期処理のため）

## トランザクション仕様

なし（データベース操作を行わない）

## パフォーマンス要件

- wyhashは非常に高速な非暗号学的ハッシュ関数
- CRC32はzlibのハードウェアアクセラレーションを使用
- 大規模データでもメモリコピーを最小化

## セキュリティ考慮事項

- 非暗号学的ハッシュのため、パスワードハッシュや署名には使用不可
- 暗号学的用途にはBun.CryptoHasherを使用すること

## 備考

サポートされるハッシュアルゴリズム一覧：
- `wyhash` - 64bit、高速汎用ハッシュ（デフォルト）
- `adler32` - 32bit、チェックサム用
- `crc32` - 32bit、CRCチェック（ハードウェア加速）
- `cityHash32` - 32bit、Google製高速ハッシュ
- `cityHash64` - 64bit、Google製高速ハッシュ
- `xxHash32` - 32bit、高速ハッシュ
- `xxHash64` - 64bit、高速ハッシュ
- `xxHash3` - 64bit、xxHashの最新版
- `murmur32v2` - 32bit、MurmurHash2
- `murmur32v3` - 32bit、MurmurHash3
- `murmur64v2` - 64bit、MurmurHash2
- `rapidhash` - 64bit、高速ハッシュ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Bun.hashのAPIインターフェースはJavaScriptからZigへの呼び出しで構成される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | HashObject.zig | `src/bun.js/api/HashObject.zig` | サポートされるハッシュ関数の定義（3-47行目） |

**読解のコツ**: Zigの`hashWrap`関数がラッパーを生成する仕組みを理解する。各ハッシュ関数はZig標準ライブラリの実装をラップしている。

#### Step 2: エントリーポイントを理解する

BunObjectからHashObjectへの参照経路を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BunObject.zig | `src/bun.js/api/BunObject.zig` | hashプロパティの登録（74行目） |
| 2-2 | BunObject.zig | `src/bun.js/api/BunObject.zig` | getHashObject関数（1263-1265行目） |

**主要処理フロー**:
1. **74行目**: `pub const hash = toJSLazyPropertyCallback(Bun.getHashObject)` でhashプロパティを登録
2. **1263-1265行目**: `HashObject.create(globalThis)` でHashObjectを生成して返却

#### Step 3: ハッシュ関数の実装を理解する

HashObject内のハッシュ関数ラッパーの実装詳細。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | HashObject.zig | `src/bun.js/api/HashObject.zig` | create関数（49-71行目）- JSへの関数公開 |
| 3-2 | HashObject.zig | `src/bun.js/api/HashObject.zig` | hashWrap関数（73-148行目）- ハッシュ計算の共通処理 |

**主要処理フロー**:
- **49-50行目**: デフォルトのhash関数としてwyhashを設定
- **51-64行目**: サブ関数名の配列定義
- **65-68行目**: 各サブ関数をJSオブジェクトに追加
- **76-146行目**: 入力データの型判定、シード値取得、ハッシュ計算、結果返却

#### Step 4: CRC32の特殊処理を理解する

zlibを使用したハードウェアアクセラレーション版CRC32の実装。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | HashObject.zig | `src/bun.js/api/HashObject.zig` | crc32定義（6-20行目）- チャンク処理による大規模データ対応 |

**主要処理フロー**:
- **7-18行目**: zlibのCRC32関数を使用、32bit長制限を回避するためチャンク分割処理

### プログラム呼び出し階層図

```
Bun.hash() / Bun.hash.wyhash() etc.
    │
    ├─ BunObject.getHashObject()
    │      └─ HashObject.create()
    │             └─ JSFunction.create() (各ハッシュ関数)
    │
    └─ HashObject.hashWrap().hash()
           ├─ 入力データ解析
           │      ├─ Blob.sharedView()
           │      ├─ ArrayBuffer.byteSlice()
           │      └─ arg.toSlice()
           │
           └─ std.hash.Xxx.hash() / bun.zlib.crc32()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

string/Blob/            HashObject.hashWrap()
ArrayBuffer/    ───▶    ├─ 入力バイト列取得     ───▶    number (32bit)
TypedArray              ├─ シード値解析                 または
    │                   └─ ハッシュ計算                 bigint (64bit)
    │
seed (optional) ───▶    シード値として使用
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| HashObject.zig | `src/bun.js/api/HashObject.zig` | ソース | ハッシュ関数のメイン実装 |
| BunObject.zig | `src/bun.js/api/BunObject.zig` | ソース | Bun.hashプロパティの登録 |
| api.zig | `src/bun.js/api.zig` | ソース | HashObjectのモジュールエクスポート |
| hashing.ts | `test/integration/bun-types/fixture/hashing.ts` | テスト | 型定義テスト |
| 09555.test.ts | `test/regression/issue/09555.test.ts` | テスト | ハッシュ機能の回帰テスト |
