# 機能設計書 75-UDPソケット

## 概要

本ドキュメントは、BunのUDPソケット機能（Bun.udpSocket）の機能設計を記述する。UDPソケットは、コネクションレスのデータグラム通信を行うためのAPIを提供する。

### 本機能の処理概要

Bun.udpSocketは、UDPプロトコルによる送受信機能を提供する。コネクションレスの特性を活かした高速な通信が可能で、ブロードキャストやマルチキャストにも対応する。

**業務上の目的・背景**：リアルタイム性が重要で、信頼性よりも速度を優先する通信に使用される。ゲームサーバー、ストリーミング、DNS、IoTセンサーデータ収集など、低遅延が求められるユースケースに適している。

**機能の利用シーン**：
- リアルタイムゲームサーバー
- 音声/動画ストリーミング
- DNSクライアント/サーバー
- IoTデバイスからのデータ収集
- ネットワーク監視ツール
- ブロードキャストメッセージ配信

**主要な処理内容**：
1. Bun.udpSocket()でUDPソケットを作成・バインド
2. connectオプションで特定の宛先に接続（オプション）
3. onDataコールバックでデータ受信
4. send()メソッドでデータ送信
5. close()でソケットを閉じる

**関連システム・外部連携**：uWebSockets（uws）のUDPモジュールを内部で使用。OSのネットワークスタック、IPv4/IPv6プロトコルスタックと連携。

**権限による制御**：ポート1024未満はroot/管理者権限が必要。ブロードキャストはソケットフラグで有効化が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドライン/API経由での利用（画面なし） |

## 機能種別

ネットワーク通信 / 非同期I/O

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| hostname | string | No | バインドするホスト名 | デフォルト: 0.0.0.0 |
| port | number | No | バインドするポート番号 | 0-65535、デフォルト: 0（自動割り当て） |
| flags | number | No | ソケットフラグ | ブロードキャスト等 |
| socket | object | No | イベントハンドラ | data, drain, error |
| binaryType | string | No | 受信データ形式 | arraybuffer, uint8array, buffer |
| connect | object | No | 接続先情報 | hostname, port |

### ソケットイベントハンドラ

| ハンドラ名 | 説明 | 引数 |
|-----------|------|------|
| data | データ受信時 | (socket, data, port, address) |
| drain | バッファが空になった時 | (socket) |
| error | エラー発生時 | (socket, error) |

### 入力データソース

API呼び出し時の引数として直接指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| UDPSocket | Promise<UDPSocket> | UDPソケットオブジェクト |

### UDPSocketオブジェクトメソッド

| メソッド | 説明 |
|---------|------|
| send(data, port?, address?) | データを送信 |
| close() | ソケットを閉じる |
| ref() | イベントループ参照を保持 |
| unref() | イベントループ参照を解除 |

### 出力先

Promiseとして返却。resolveでUDPSocketオブジェクトを取得。

## 処理フロー

### 処理シーケンス

```
1. Bun.udpSocket()呼び出し
   └─ UDPSocketConfigをパース
2. ソケット作成
   └─ uws.udp.Socket.createでバインド
3. 接続（オプション）
   └─ connectオプション指定時は特定宛先に接続
4. データ送受信
   └─ onDataでデータ受信、send()で送信
5. 終了
   └─ close()でソケット解放
```

### フローチャート

```mermaid
flowchart TD
    A[Bun.udpSocket] --> B[UDPSocketConfig解析]
    B --> C[UDPSocket作成]
    C --> D[uws.udp.Socket.create]
    D --> E{バインド成功?}
    E -->|No| F[SystemError投げる]
    E -->|Yes| G{connect指定?}
    G -->|Yes| H[socket.connect]
    G -->|No| I[Promise resolve]
    H --> J{connect成功?}
    J -->|No| K[エラー]
    J -->|Yes| I
    I --> L[データ送受信ループ]
    L --> M{データ受信?}
    M -->|Yes| N[onData callback]
    N --> L
    L --> O{send呼び出し?}
    O -->|Yes| P[データ送信]
    P --> L
    L --> Q[close呼び出し]
    Q --> R[onClose]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-75-01 | コネクションレス | UDPは接続状態を持たない | 常時 |
| BR-75-02 | connect済み送信 | connect後はsend()で宛先省略可 | connect指定時 |
| BR-75-03 | ポート0 | ポート0指定でOS自動割り当て | 常時 |
| BR-75-04 | IPv4/IPv6 | 両プロトコルをサポート | 常時 |

### 計算ロジック

なし

## データベース操作仕様

### 操作別データベース影響一覧

なし（データベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| EADDRINUSE | SystemError | ポートが使用中 | 別ポートを使用 |
| EACCES | SystemError | 権限不足 | 管理者権限で実行 |
| InvalidArguments | TypeError | 不正な引数 | 引数を修正 |
| EAI_ERROR | Error | アドレス解決エラー | ホスト名を確認 |

### リトライ仕様

なし（呼び出し側で制御）

## トランザクション仕様

なし

## パフォーマンス要件

- コネクションレスで低遅延
- uWebSocketsベースの高効率実装
- 非同期IOでメインスレッド非ブロック

## セキュリティ考慮事項

- UDPは暗号化なし（DTLSは別途実装が必要）
- 送信元偽装が可能（プロトコル特性）
- ファイアウォール設定に注意

## 備考

- IPv6スコープIDに対応（リンクローカルアドレス等）
- パケットバッファで複数パケットを効率的に処理
- binaryTypeでデータ形式を指定可能（Buffer, Uint8Array, ArrayBuffer）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

UDPソケット関連の型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | UDPSocketConfig構造体（108-229行目） |
| 1-2 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | UDPSocket構造体（231-257行目） |

**読解のコツ**: UDPSocketConfigはJSオプションからの設定をパースして保持。handlersタプルでコールバック名を定義。

**主要処理フロー**:
1. **110-114行目**: handlersでdata, drain, errorの3つのコールバックを定義
2. **116-119行目**: ConnectConfigで接続先情報を保持
3. **121-125行目**: 設定フィールド（hostname, port, flags, binary_type）

#### Step 2: エントリーポイントを理解する

Bun.udpSocket関数の実装。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | udpSocket関数（260-341行目） |

**主要処理フロー**:
1. **264-270行目**: UDPSocket構造体の初期化
2. **284行目**: UDPSocketConfig.fromJSでオプションパース
3. **293-319行目**: uws.udp.Socket.createでソケット作成
4. **321-337行目**: connect指定時の接続処理
5. **340行目**: Promise.resolvedPromiseValueで返却

#### Step 3: イベントコールバックを理解する

データ受信等のコールバック処理。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | onData関数（38-106行目） |
| 3-2 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | onDrain関数（21-36行目） |
| 3-3 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | onClose関数（11-19行目） |

**主要処理フロー**:
- **38-106行目**: onDataでパケット受信、IPv4/IPv6アドレス解析、JSコールバック呼び出し
- **50-72行目**: IPv4/IPv6アドレスの変換（inet_ntop使用）
- **78-105行目**: 各パケットについてコールバック呼び出し

#### Step 4: ネットワーク関数を理解する

低レベルネットワーク関数。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | 外部関数宣言（5-9行目） |

**主要処理フロー**:
- **5-9行目**: ntohs, htonl, htons, inet_ntop, inet_ptonのextern宣言

### プログラム呼び出し階層図

```
Bun.udpSocket(opts)
    │
    ├─ UDPSocket.new()
    │      └─ UDPSocketConfig.fromJS()
    │
    ├─ uws.udp.Socket.create()
    │      ├─ onData callback設定
    │      ├─ onDrain callback設定
    │      └─ onClose callback設定
    │
    └─ socket.connect() [オプション]

データ受信時:
uws.udp.Socket
    │
    └─ onData(socket, buf, packets)
           ├─ inet_ntop() [アドレス変換]
           └─ callback.call() [JSコールバック]

送信時:
UDPSocket.send(data, port?, address?)
    │
    └─ socket.send() [uws]
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Bun.udpSocket(opts) ───▶  UDPSocketConfig.fromJS()
                          ├─ hostname解析
                          ├─ port解析
                          └─ connect設定
                                 │
                                 ▼
                          uws.udp.Socket.create()
                          └─ ポートバインド
                                 │
                                 ▼
                          Promise<UDPSocket>        ───▶   UDPSocket object

パケット受信         ───▶  onData callback
                          ├─ アドレス変換
                          └─ binaryType変換
                                 │
                                 ▼
                          JS callback              ───▶   (socket, data, port, addr)

socket.send(data)   ───▶  uws送信処理            ───▶   ネットワーク送信
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| udp_socket.zig | `src/bun.js/api/bun/udp_socket.zig` | ソース | UDPソケット実装 |
| uws/udp.zig | `src/deps/uws/` | ソース | uwsのUDPモジュール（参照先） |
| dgram.ts | `src/js/node/dgram.ts` | ソース | Node.js dgram互換レイヤー |
