# 機能設計書 81-crypto

## 概要

本ドキュメントは、BunランタイムにおけるWeb Crypto API機能の設計を記述するものである。Web Crypto APIはWebブラウザ標準の暗号化インターフェースであり、Bunはこれをサーバーサイドで利用可能にしている。

### 本機能の処理概要

Web Crypto APIは、暗号化、復号、署名、検証、ハッシュ計算、鍵生成などのセキュリティ関連操作をJavaScriptから利用可能にする標準APIである。

**業務上の目的・背景**：現代のWebアプリケーションでは、データの暗号化、デジタル署名、セキュアな乱数生成など、セキュリティ機能が不可欠である。Web Crypto APIを提供することで、ブラウザと同じインターフェースでサーバーサイドでもセキュアなアプリケーションを構築できる。これにより、既存のWebアプリケーションコードをBunランタイムに移植する際の互換性を確保し、開発者の学習コストを削減する。

**機能の利用シーン**：
- パスワードのハッシュ化と検証
- APIリクエストの署名と検証
- データの暗号化と復号
- セキュアなトークン生成
- TLS/SSL通信のサポート
- ファイルの整合性チェック（ハッシュ検証）

**主要な処理内容**：
1. 暗号学的に安全な乱数の生成（`crypto.getRandomValues()`）
2. ハッシュ計算（SHA-1、SHA-256、SHA-384、SHA-512等）
3. 対称暗号化/復号（AES-CBC、AES-GCM、AES-CTR等）
4. 非対称暗号化/復号（RSA-OAEP）
5. デジタル署名と検証（RSA-PSS、ECDSA、Ed25519）
6. 鍵導出（PBKDF2、HKDF）
7. 鍵ペアの生成とインポート/エクスポート

**関連システム・外部連携**：BoringSSL（Googleが管理するOpenSSLフォーク）を暗号プリミティブの実装に使用。JavaScriptCoreエンジンとの連携によりJavaScript APIを提供。

**権限による制御**：特になし。全てのユーザーコードから利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドラインまたはAPIとして利用 |

## 機能種別

計算処理 / セキュリティAPI

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| algorithm | string/object | Yes | 暗号アルゴリズム名または設定オブジェクト | サポート対象アルゴリズムのみ |
| key | CryptoKey | Yes | 暗号鍵オブジェクト | 有効なCryptoKeyであること |
| data | ArrayBuffer/TypedArray | Yes | 処理対象データ | バッファ形式であること |

### 入力データソース

JavaScript API呼び出しを通じて提供されるバイナリデータおよびパラメータ。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | ArrayBuffer/CryptoKey | 処理結果（暗号化データ、ハッシュ値、鍵等） |
| Promise | Promise<ArrayBuffer/CryptoKey> | 非同期操作の場合 |

### 出力先

JavaScript呼び出し元への戻り値。

## 処理フロー

### 処理シーケンス

```
1. JavaScript APIの呼び出し
   └─ globalThis.crypto.subtle.* または crypto.getRandomValues() 等
2. パラメータのバリデーション
   └─ アルゴリズム、鍵、データ形式の検証
3. BoringSSL関数の呼び出し
   └─ 暗号プリミティブ処理の実行
4. 結果の変換とJavaScriptへの返却
   └─ ArrayBuffer/CryptoKeyオブジェクトとして返却
```

### フローチャート

```mermaid
flowchart TD
    A[JavaScript API呼び出し] --> B{パラメータ検証}
    B -->|有効| C[アルゴリズム選択]
    B -->|無効| D[TypeError例外]
    C --> E[BoringSSL関数呼び出し]
    E --> F{処理成功?}
    F -->|Yes| G[結果をJSに変換]
    F -->|No| H[CryptoError例外]
    G --> I[Promiseの解決]
    H --> J[Promiseの拒否]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Web標準準拠 | W3C Web Crypto API仕様に準拠する | 全ての暗号操作 |
| BR-02 | 同期/非同期 | getRandomValuesは同期、subtle.*は非同期 | API呼び出し時 |
| BR-03 | 鍵の不変性 | CryptoKeyオブジェクトは不変 | 鍵操作時 |

### 計算ロジック

PBKDF2、HKDF等の鍵導出関数は反復回数やソルトを指定して派生鍵を計算。

## データベース操作仕様

該当なし（メモリ内での暗号操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | パラメータエラー | 不正なパラメータ型 | パラメータの型を確認 |
| NotSupportedError | 非サポート | 未サポートアルゴリズム | 対応アルゴリズムを使用 |
| OperationError | 操作エラー | 暗号操作の失敗 | 鍵やデータを確認 |
| DataError | データエラー | 不正なデータ形式 | 入力データを確認 |

### リトライ仕様

暗号操作は決定論的であり、リトライは不要。

## トランザクション仕様

該当なし（ステートレスな暗号操作）

## パフォーマンス要件

- ハッシュ計算：数百MB/秒以上
- 暗号化/復号：ハードウェア支援時は高速処理
- 鍵生成：RSA-2048で数秒以内

## セキュリティ考慮事項

- BoringSSLによる検証済み暗号実装の使用
- 暗号学的に安全な乱数生成器（CSPRNG）の使用
- 鍵データのメモリ保護
- タイミング攻撃への耐性

## 備考

BunはNode.js互換の`crypto`モジュール（機能No.90）も提供しており、両方のAPIを利用可能。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Web Crypto APIの中心となるデータ構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | crypto.zig | `src/bun.js/api/crypto.zig` | CryptoHasher、MD5、SHA256等の型定義とエクスポート |
| 1-2 | CryptoHasher.zig | `src/bun.js/api/crypto/CryptoHasher.zig` | ハッシュ計算の中核実装 |

**読解のコツ**: Zigでは`pub const`でエクスポートされる型を確認し、`@import`で依存関係を追跡する。

#### Step 2: エントリーポイントを理解する

JavaScriptからの呼び出しがどのように処理されるかを把握。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | crypto.zig | `src/bun.js/api/crypto.zig` | `createCryptoError`関数がエラー生成のエントリーポイント |

**主要処理フロー**:
1. **1-3行目**: `createCryptoError`関数でBoringSSLエラーをJSエラーに変換
2. **5-18行目**: 各種ハッシュアルゴリズム（MD4, MD5, SHA1等）のエクスポート

#### Step 3: BoringSSLとの連携を理解する

実際の暗号処理はBoringSSLライブラリで実行される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | EVP.zig | `src/bun.js/api/crypto/EVP.zig` | OpenSSL/BoringSSL EVP APIのラッパー |
| 3-2 | HMAC.zig | `src/bun.js/api/crypto/HMAC.zig` | HMAC実装 |

### プログラム呼び出し階層図

```
globalThis.crypto (JavaScript)
    │
    ├─ crypto.getRandomValues()
    │      └─ BoringSSL RAND_bytes
    │
    └─ crypto.subtle.*
           ├─ digest() → CryptoHasher → BoringSSL EVP_Digest*
           ├─ encrypt/decrypt() → EVP.zig → BoringSSL EVP_Cipher*
           ├─ sign/verify() → HMAC.zig / EVP.zig → BoringSSL
           └─ generateKey() → BoringSSL key generation
```

### データフロー図

```
[入力]               [処理]                    [出力]

ArrayBuffer    ───▶ crypto.zig             ───▶ ArrayBuffer
(平文/暗号文)         │                          (暗号文/平文)
                     ├─ CryptoHasher.zig
                     ├─ EVP.zig
                     └─ BoringSSL (C)

CryptoKey      ───▶ 鍵管理               ───▶ CryptoKey
(鍵オブジェクト)                              (派生鍵/新規鍵)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| crypto.zig | `src/bun.js/api/crypto.zig` | ソース | Web Crypto APIのエントリーポイント |
| CryptoHasher.zig | `src/bun.js/api/crypto/CryptoHasher.zig` | ソース | ハッシュ計算実装 |
| EVP.zig | `src/bun.js/api/crypto/EVP.zig` | ソース | EVP暗号APIラッパー |
| HMAC.zig | `src/bun.js/api/crypto/HMAC.zig` | ソース | HMAC実装 |
| PasswordObject.zig | `src/bun.js/api/crypto/PasswordObject.zig` | ソース | パスワードハッシュ |
| crypto.classes.ts | `src/bun.js/api/crypto.classes.ts` | コード生成 | JSクラス定義 |
