# 機能設計書 82-Console

## 概要

本ドキュメントは、Bunランタイムにおけるコンソール出力機能（console.log等）の設計を記述するものである。ConsoleObjectはWeb標準のConsole APIを実装し、デバッグやログ出力に使用される。

### 本機能の処理概要

Console機能は、JavaScriptアプリケーションからのログ出力、デバッグ情報、エラーメッセージをターミナルに出力するためのAPIを提供する。

**業務上の目的・背景**：アプリケーション開発において、実行時の状態把握やデバッグは不可欠である。Console APIはブラウザで広く使用されており、同じインターフェースをサーバーサイドでも提供することで、開発者が馴染みのある方法でデバッグやログ出力を行える。また、本番環境でのログ監視やトラブルシューティングにも重要な役割を果たす。

**機能の利用シーン**：
- アプリケーションのデバッグ中の値の確認
- エラー発生時のスタックトレース出力
- パフォーマンス計測（console.time/timeEnd）
- テーブル形式でのデータ表示（console.table）
- アサーション検証（console.assert）
- グループ化されたログ出力

**主要な処理内容**：
1. 各種ログレベル（log, warn, error, debug, info）でのメッセージ出力
2. オブジェクトの整形表示（JSON形式、プロパティ展開）
3. ANSIカラーコードによる色付き出力
4. タイマー機能（time, timeEnd, timeLog）
5. テーブル形式表示
6. スタックトレース出力（trace）
7. カウンター機能（count, countReset）
8. グループ化（group, groupEnd）

**関連システム・外部連携**：標準出力（stdout）および標準エラー出力（stderr）への書き込み。テストランナー（Jest互換）との連携。

**権限による制御**：特になし。全てのユーザーコードから利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | test | 補助機能 | テスト実行中のログ出力 |

## 機能種別

デバッグ/ログ出力

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ...args | any[] | No | 出力する値（可変長） | 任意の型を許容 |
| label | string | No | タイマー/カウンターのラベル | 文字列 |
| tabularData | object/array | Yes (table) | テーブル表示用データ | オブジェクトまたは配列 |
| properties | string[] | No | 表示するプロパティ指定 | 文字列配列 |

### 入力データソース

JavaScript API呼び出しを通じて提供される任意の値。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 文字列出力 | string | フォーマット済みテキスト |
| undefined | undefined | 全てのconsoleメソッドの戻り値 |

### 出力先

- stdout（log, info, debug, table, dir, time系）
- stderr（warn, error, assert, trace）

## 処理フロー

### 処理シーケンス

```
1. console.*メソッドの呼び出し
   └─ JavaScript VMからの呼び出し
2. メッセージタイプとレベルの決定
   └─ Log/Warning/Error/Debug等
3. フォーマット処理
   └─ オブジェクト展開、色付け、インデント
4. Mutexによる排他制御
   └─ 複数スレッドからの同時出力対応
5. バッファへの書き込みとフラッシュ
   └─ stdout/stderrへの出力
```

### フローチャート

```mermaid
flowchart TD
    A[console.*呼び出し] --> B{メッセージタイプ判定}
    B -->|Log/Info/Debug| C[stdoutバッファ取得]
    B -->|Warn/Error| D[stderrバッファ取得]
    C --> E[Mutex取得]
    D --> E
    E --> F{テーブル出力?}
    F -->|Yes| G[TablePrinter処理]
    F -->|No| H[format2関数でフォーマット]
    G --> I[バッファ書き込み]
    H --> I
    I --> J[フラッシュ]
    J --> K[Mutex解放]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Web標準準拠 | WHATWG Console Standard準拠 | 全メソッド |
| BR-02 | 出力先分離 | warn/errorはstderr、他はstdout | メッセージレベルに応じて |
| BR-03 | スレッドセーフ | Mutex使用で排他制御 | 複数スレッドからの出力時 |
| BR-04 | 深度制限 | オブジェクト展開の深度を制限 | --console-depthオプション |

### 計算ロジック

console.timeEnd: 経過時間 = 現在時刻 - console.time記録時刻（ミリ秒単位）

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | TypeError | 不正な引数型（一部メソッド） | 適切な型に変換 |
| - | 出力エラー | stdout/stderrへの書き込み失敗 | エラーを無視して継続 |

### リトライ仕様

出力エラー時は無視して継続（アプリケーションの停止を防ぐ）。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 4096バイトのバッファを使用した効率的な出力
- Mutexによる最小限のロック時間
- 色付け出力のオプション化

## セキュリティ考慮事項

- センシティブ情報のログ出力に注意が必要（アプリケーション側の責任）
- 大量出力によるDoS的状況の可能性

## 備考

`--console-depth`フラグまたはbunfig.tomlの`console.depth`設定でオブジェクト展開深度を変更可能。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Console機能の中心となるデータ構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ConsoleObject.zig | `src/bun.js/ConsoleObject.zig` | ConsoleObject構造体の定義、バッファサイズ（4096バイト） |

**読解のコツ**:
- **9-21行目**: ConsoleObjectの主要フィールド（stderr_buffer, stdout_buffer, error_writer, writer等）
- **44-69行目**: MessageLevel（Log/Warning/Error/Debug/Info）とMessageType（Log/Dir/Table/Trace等）のenum定義

#### Step 2: エントリーポイントを理解する

JavaScriptからの呼び出しがどのように処理されるかを把握。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ConsoleObject.zig | `src/bun.js/ConsoleObject.zig` | `messageWithTypeAndLevel`関数がメインエントリーポイント |

**主要処理フロー**:
1. **78-88行目**: `messageWithTypeAndLevel`がcallconv(.c)でエクスポートされ、JSから呼び出される
2. **114-142行目**: Mutex取得/解放のロジック（スレッドセーフ対応）
3. **192-211行目**: console.tableの特別処理（TablePrinter使用）

#### Step 3: フォーマット処理を理解する

出力フォーマットの詳細処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ConsoleObject.zig | `src/bun.js/ConsoleObject.zig` | `format2`関数、TablePrinter構造体 |

**主要処理フロー**:
- **251-306行目**: TablePrinter構造体の定義（console.table用）
- **359-424行目**: updateColumnsForRow関数でカラム幅を計算

### プログラム呼び出し階層図

```
console.log() (JavaScript)
    │
    └─ ConsoleObject.messageWithTypeAndLevel() (Zig)
           │
           ├─ Mutex取得 (stdout_mutex/stderr_mutex)
           │
           ├─ メッセージタイプ判定
           │      ├─ Table → TablePrinter.printTable()
           │      ├─ Dir → format2() with depth options
           │      ├─ Trace → writeTrace()
           │      └─ その他 → format2()
           │
           ├─ バッファ書き込み (writer/error_writer)
           │
           ├─ フラッシュ
           │
           └─ Mutex解放
```

### データフロー図

```
[入力]               [処理]                    [出力]

JavaScript値    ───▶ ConsoleObject.zig       ───▶ stdout
(any型)              │                            │
                     ├─ MessageLevel判定          └─ stderr
                     ├─ フォーマット処理
                     │   ├─ 文字列変換
                     │   ├─ オブジェクト展開
                     │   └─ ANSI色付け
                     └─ バッファリング
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ConsoleObject.zig | `src/bun.js/ConsoleObject.zig` | ソース | Console APIのメイン実装 |
| ConsoleObject.cpp | `src/bun.js/bindings/ConsoleObject.cpp` | ソース | C++側のJSCバインディング |
| ConsoleObject.h | `src/bun.js/bindings/ConsoleObject.h` | ヘッダー | C++ヘッダー定義 |
| ConsoleObject.ts | `src/js/builtins/ConsoleObject.ts` | ソース | JavaScript側の補助実装 |
| console.ts | `src/js/node/console.ts` | ソース | Node.js互換のconsoleモジュール |
