# 機能設計書 83-Workers

## 概要

本ドキュメントは、BunランタイムにおけるWeb Workers機能の設計を記述するものである。Web Workersは、メインスレッドとは独立したバックグラウンドスレッドでJavaScriptを実行する仕組みを提供する。

### 本機能の処理概要

Web Workers機能は、マルチスレッドJavaScript実行を可能にし、CPUバウンドな処理をメインスレッドから分離してアプリケーションの応答性を維持する。

**業務上の目的・背景**：JavaScriptはシングルスレッドで実行されるため、重い計算処理があるとUIやI/O処理がブロックされる。Web Workersを提供することで、CPU集約的な処理を別スレッドで実行し、メインスレッドの応答性を維持できる。これはブラウザ標準のWeb Workers APIと互換性があり、既存のコードを移植しやすい。また、Node.jsのworker_threadsモジュールとも互換性を持つ。

**機能の利用シーン**：
- 画像処理や動画エンコード等の重い計算処理
- 大量データの解析やソート
- バックグラウンドでのファイル処理
- 並列タスク実行によるパフォーマンス向上
- 暗号化処理の並列実行
- サーバーサイドでのリクエスト処理の分散

**主要な処理内容**：
1. Workerスレッドの生成と初期化
2. メッセージパッシング（postMessage/onmessage）
3. エントリーポイントスクリプトの解決と実行
4. Worker終了処理（terminate）
5. エラーハンドリングと親スレッドへの伝播
6. ref/unrefによるイベントループ制御

**関連システム・外部連携**：JavaScriptCore VirtualMachine、Node.js worker_threads互換、標準スレッドライブラリ。

**権限による制御**：特になし。全てのユーザーコードからWorkerを生成可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドラインまたはAPIとして利用 |

## 機能種別

並列処理 / スレッド管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| specifier | string | Yes | Workerスクリプトのパス/URL | ファイルが存在すること |
| options.name | string | No | Workerの名前（デバッグ用） | 文字列 |
| options.type | string | No | モジュールタイプ（module/classic） | 許可値のみ |
| options.eval | boolean | No | specifierをコードとして評価 | - |
| options.argv | string[] | No | Worker内のprocess.argv | 文字列配列 |
| options.execArgv | string[] | No | 実行引数 | 文字列配列 |

### 入力データソース

コンストラクタ引数として提供されるスクリプトパスおよびオプション。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Worker | Worker | Workerオブジェクトインスタンス |
| message event | Event | postMessageで送信されたデータ |
| error event | Event | Worker内で発生したエラー |
| exit event | Event | Worker終了時のイベント |

### 出力先

イベントリスナーへのコールバック呼び出し。

## 処理フロー

### 処理シーケンス

```
1. new Worker(specifier, options)の呼び出し
   └─ WebWorker.create()でWorkerオブジェクト生成
2. スレッドの生成
   └─ std.Thread.spawn()で新規スレッド起動
3. エントリーポイントの解決
   └─ resolveEntryPointSpecifier()でパス解決
4. VirtualMachineの初期化
   └─ jsc.VirtualMachine.initWorker()で新VMを作成
5. スクリプトのロードと実行
   └─ loadEntryPointForWebWorker()でスクリプト実行
6. イベントループの開始
   └─ spin()でWorkerのイベントループを維持
7. メッセージング
   └─ postMessage/onmessageでデータ送受信
8. 終了処理
   └─ terminate()またはスクリプト終了でクリーンアップ
```

### フローチャート

```mermaid
flowchart TD
    A[new Worker] --> B[WebWorker.create]
    B --> C[スレッド生成]
    C --> D[エントリーポイント解決]
    D --> E{解決成功?}
    E -->|Yes| F[VM初期化]
    E -->|No| G[エラーディスパッチ]
    F --> H[スクリプト実行]
    H --> I[dispatchOnline]
    I --> J[イベントループ]
    J --> K{終了要求?}
    K -->|No| L[tick処理]
    L --> J
    K -->|Yes| M[exitAndDeinit]
    G --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Web Workers標準 | Web Workers APIに準拠 | Worker生成・操作時 |
| BR-02 | worker_threads互換 | Node.js worker_threadsと互換 | require('worker_threads')時 |
| BR-03 | スレッド独立性 | 各Workerは独立したVM | 全Worker |
| BR-04 | メッセージシリアライズ | 構造化クローンでデータ転送 | postMessage時 |

### 計算ロジック

特になし。

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | エントリーポイントエラー | スクリプトパスが不正 | パスを確認 |
| - | BlobURLエラー | Blob URLが存在しない | URLを確認 |
| - | 未処理拒否 | Workerでの未処理Promise | errorイベントで捕捉 |
| - | スレッド生成失敗 | システムリソース不足 | リソース確認 |

### リトライ仕様

Worker生成失敗時は自動リトライなし。アプリケーション側でリトライロジックを実装。

## トランザクション仕様

該当なし

## パフォーマンス要件

- Worker生成：数ミリ秒以内
- メッセージパッシング：構造化クローンのオーバーヘッドあり
- スレッドスタックサイズ：bun.default_thread_stack_size使用

## セキュリティ考慮事項

- Workerは親スレッドとメモリを共有しない（SharedArrayBufferを除く）
- メッセージは構造化クローンでシリアライズされる
- Workerスクリプトは信頼できるソースから取得すること

## 備考

`--no-addons`オプションでネイティブアドオンの使用を禁止可能。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

WebWorker構造体の主要フィールドを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | web_worker.zig | `src/bun.js/web_worker.zig` | WebWorker構造体とStatusのenum定義 |

**読解のコツ**:
- **7-28行目**: WebWorker構造体のフィールド（vm, status, parent, name, cpp_worker等）
- **43-48行目**: Status enum（start, starting, running, terminated）

#### Step 2: エントリーポイントを理解する

Worker生成の流れを把握。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | web_worker.zig | `src/bun.js/web_worker.zig` | `create`関数と`WebWorker__updatePtr` |

**主要処理フロー**:
1. **188-262行目**: `create`関数でWebWorkerインスタンス生成
2. **68-81行目**: `WebWorker__updatePtr`でスレッド生成（std.Thread.spawn）
3. **83-186行目**: `resolveEntryPointSpecifier`でスクリプトパス解決

#### Step 3: Workerライフサイクルを理解する

Workerの開始から終了までの流れを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | web_worker.zig | `src/bun.js/web_worker.zig` | `start`, `spin`, `exitAndDeinit`関数 |

**主要処理フロー**:
- **273-364行目**: `start`関数でVM初期化と環境設定
- **452-540行目**: `spin`関数でイベントループ実行
- **587-635行目**: `exitAndDeinit`関数で終了処理とリソース解放

### プログラム呼び出し階層図

```
new Worker() (JavaScript)
    │
    └─ WebWorker.create() (Zig, lines 188-262)
           │
           ├─ resolveEntryPointSpecifier() (lines 83-186)
           │
           └─ WebWorker__updatePtr() (lines 68-81)
                  │
                  └─ std.Thread.spawn() → startWithErrorHandling()
                         │
                         └─ start() (lines 273-364)
                                │
                                ├─ jsc.VirtualMachine.initWorker()
                                │
                                └─ spin() (lines 452-540)
                                       │
                                       ├─ loadEntryPointForWebWorker()
                                       ├─ WebWorker__dispatchOnline()
                                       └─ イベントループ (vm.tick())
```

### データフロー図

```
[入力]               [処理]                    [出力]

Worker specifier ───▶ WebWorker.create      ───▶ Worker object
options              │
                     ├─ スレッド生成
                     ├─ VM初期化
                     └─ スクリプト実行

postMessage     ───▶ メッセージキュー      ───▶ onmessage event
(parent→worker)      (構造化クローン)

postMessage     ───▶ メッセージキュー      ───▶ onmessage event
(worker→parent)      (構造化クローン)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| web_worker.zig | `src/bun.js/web_worker.zig` | ソース | Web Workersのメイン実装 |
| worker_threads.ts | `src/js/node/worker_threads.ts` | ソース | Node.js worker_threads互換API |
| VirtualMachine | `src/bun.js/` | ソース | JavaScriptCore VM管理 |
