# 機能設計書 85-JSON5

## 概要

本ドキュメントは、BunランタイムにおけるJSON5パーサー機能の設計を記述するものである。JSON5はJSONの拡張形式であり、コメント、末尾カンマ、シングルクォート文字列などを許容する。

### 本機能の処理概要

JSON5機能は、標準JSONよりも柔軟な構文を持つJSON5形式のデータをパースおよびシリアライズする機能を提供する。

**業務上の目的・背景**：設定ファイルや人間が編集するデータファイルでは、コメントや末尾カンマなど、標準JSONでは許容されない構文が望まれることが多い。JSON5はこれらの要求に応えるフォーマットであり、特にTypeScript/JavaScript開発においてtsconfig.jsonやpackage.jsonの記述で広く使われている。Bunがネイティブでサポートすることで、外部ライブラリなしでこれらのファイルを処理できる。

**機能の利用シーン**：
- 設定ファイル（tsconfig.json、bun.config.js等）の読み込み
- 人間が編集するデータファイルの解析
- コメント付きJSONファイルの処理
- 緩やかなJSON構文を必要とするアプリケーション

**主要な処理内容**：
1. JSON5文字列のパース（parse）
2. JavaScriptオブジェクトのJSON5シリアライズ（stringify）
3. 各種リテラル（Infinity, NaN, 16進数）の処理
4. コメント（// および /* */）のスキップ
5. 末尾カンマの許容

**関連システム・外部連携**：Bunの内部パーサー（js_lexer）を使用。bun.interchange.json5モジュール。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | APIとして利用 |

## 機能種別

パーサー / データ変換

## 入力仕様

### 入力パラメータ（JSON5.parse）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| text | string/Buffer/TypedArray | Yes | パース対象テキスト | JSON5形式であること |

### 入力パラメータ（JSON5.stringify）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| value | any | Yes | シリアライズ対象値 | - |
| replacer | null | No | 未サポート | nullのみ |
| space | number/string | No | インデント設定 | 0-10または文字列 |

### 入力データソース

JavaScript API呼び出しを通じて提供される文字列またはバッファ。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| value | any | パース結果（parseの場合） |
| string | string | シリアライズ結果（stringifyの場合） |

### 出力先

JavaScript呼び出し元への戻り値。

## 処理フロー

### 処理シーケンス

```
1. JSON5.parse(text)の呼び出し
   └─ 入力文字列の取得
2. ASTメモリアロケータの初期化
   └─ bun.ArenaAllocatorの使用
3. JSON5Parserでパース
   └─ ast.Exprツリーの生成
4. exprToJSでJavaScript値に変換
   └─ 再帰的な値変換
5. 結果の返却
```

### フローチャート

```mermaid
flowchart TD
    A[JSON5.parse呼び出し] --> B[入力文字列取得]
    B --> C[ArenaAllocator初期化]
    C --> D[JSON5Parser.parse]
    D --> E{パース成功?}
    E -->|Yes| F[exprToJS変換]
    E -->|No| G[SyntaxError]
    F --> H[JSValue返却]
    G --> I[エラー返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | JSON5仕様準拠 | JSON5仕様に準拠したパース | parse時 |
| BR-02 | 特殊値対応 | Infinity, NaN, 16進数リテラル対応 | stringify時 |
| BR-03 | replacer未サポート | stringify時replacerは未サポート | stringify時 |
| BR-04 | 末尾カンマ出力 | 整形出力時は末尾カンマを追加 | space指定時 |

### 計算ロジック

space引数が10より大きい場合は10にクランプ。

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SyntaxError | 構文エラー | 不正なJSON5構文 | 構文を確認 |
| TypeError | 型エラー | 無効な入力型 | 入力を確認 |
| RangeError | 範囲エラー | 循環参照検出 | データ構造を確認 |

### リトライ仕様

パースエラー時は自動リトライなし。

## トランザクション仕様

該当なし

## パフォーマンス要件

- ArenaAllocatorによる効率的なメモリ管理
- スタックチェックによるスタックオーバーフロー防止

## セキュリティ考慮事項

- 巨大ファイルのパース時のリソース消費に注意
- 循環参照の検出でエラー発生

## 備考

Bunはbun:json5としてモジュールをエクスポート。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

JSON5処理の中心となるデータ構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | JSON5Object.zig | `src/bun.js/api/JSON5Object.zig` | create関数、Stringifier構造体 |

**読解のコツ**:
- **1-14行目**: create関数でparseとstringify関数をエクスポート
- **95-136行目**: Stringifier構造体とSpace union定義

#### Step 2: parse関数を理解する

JSON5文字列のパース処理を把握。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | JSON5Object.zig | `src/bun.js/api/JSON5Object.zig` | `parse`関数 |

**主要処理フロー**:
1. **43-93行目**: parse関数
2. **47-53行目**: ArenaAllocator初期化とASTメモリアロケータ設定
3. **61-66行目**: 入力データの取得（BlobOrStringOrBuffer）
4. **74行目**: JSON5Parser.parseでASTを生成
5. **92行目**: exprToJSでJavaScript値に変換

#### Step 3: stringify関数を理解する

JavaScript値のJSON5シリアライズ処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | JSON5Object.zig | `src/bun.js/api/JSON5Object.zig` | `stringify`関数、Stringifier構造体 |

**主要処理フロー**:
- **16-41行目**: stringify関数
- **155-213行目**: stringifyValue関数（再帰的な値変換）
- **215-259行目**: stringifyArray関数（配列処理）
- **261-313行目**: stringifyObject関数（オブジェクト処理）

### プログラム呼び出し階層図

```
JSON5.parse() (JavaScript)
    │
    └─ parse() (Zig, lines 43-93)
           │
           ├─ ArenaAllocator.init()
           ├─ BlobOrStringOrBuffer.fromJS() - 入力取得
           ├─ JSON5Parser.parse() - AST生成
           └─ exprToJS() - JS値変換 (lines 388-418)
                  │
                  ├─ e_null → .null
                  ├─ e_boolean → .jsBoolean()
                  ├─ e_number → .jsNumber()
                  ├─ e_string → str.toJS()
                  ├─ e_array → JSValue.createEmptyArray()
                  └─ e_object → JSValue.createEmptyObject()

JSON5.stringify() (JavaScript)
    │
    └─ stringify() (Zig, lines 16-41)
           │
           └─ Stringifier.stringifyValue() (lines 155-213)
                  │
                  ├─ stringifyArray() (lines 215-259)
                  └─ stringifyObject() (lines 261-313)
```

### データフロー図

```
[入力]               [処理]                    [出力]

JSON5 text      ───▶ JSON5Parser.parse     ───▶ ast.Expr
                     (bun.interchange.json5)

ast.Expr        ───▶ exprToJS              ───▶ JSValue
                     (再帰変換)

JSValue         ───▶ Stringifier           ───▶ JSON5 string
(stringify)          │
                     ├─ 循環参照チェック
                     └─ 整形/圧縮切り替え
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| JSON5Object.zig | `src/bun.js/api/JSON5Object.zig` | ソース | JSON5 APIのメイン実装 |
| json5.zig | `src/bun/interchange/json5/` | ソース | JSON5パーサー本体 |
| js_lexer.zig | `src/js_lexer.zig` | ソース | 字句解析器 |
