# 機能設計書 89-node:os

## 概要

本ドキュメントは、Bunランタイムにおけるnode:os（OS情報取得）モジュールの設計を記述するものである。Node.jsのosモジュールと互換性のあるOS情報取得APIを提供する。

### 本機能の処理概要

node:os機能は、実行環境のオペレーティングシステムに関する情報（CPU、メモリ、ネットワーク、ユーザー情報等）を取得するNode.js互換モジュールである。

**業務上の目的・背景**：サーバーサイドアプリケーションでは、実行環境のシステム情報を取得してリソース管理、ログ出力、診断情報の収集などを行う必要がある。Node.jsのosモジュールは広く使われており、Bunでもこれと互換性のあるAPIを提供することで、既存コードの移植性を確保する。

**機能の利用シーン**：
- システムリソースの監視（CPU使用率、メモリ使用量）
- 診断情報の収集とログ出力
- プラットフォーム判定による処理分岐
- 一時ディレクトリやホームディレクトリの取得
- ネットワークインターフェース情報の取得

**主要な処理内容**：
1. CPU情報取得（cpus）
2. メモリ情報取得（freemem, totalmem）
3. ホスト情報取得（hostname, platform, type, release）
4. ユーザー情報取得（userInfo, homedir）
5. ネットワーク情報取得（networkInterfaces）
6. プロセス優先度操作（getPriority, setPriority）
7. 負荷平均取得（loadavg）
8. システム稼働時間取得（uptime）

**関連システム・外部連携**：OS固有のシステムコール、sysctlbyname（macOS）、/proc（Linux）、libuv（Windows）。

**権限による制御**：一部操作（setPriority）は権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | APIとして利用 |

## 機能種別

システム情報取得 / Node.js互換

## 入力仕様

### 主要入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pid | number | No (getPriority) | プロセスID | 整数 |
| priority | number | Yes (setPriority) | 優先度 | -20〜19 |
| options | object | No (userInfo) | エンコーディング等 | - |

### 入力データソース

JavaScript API呼び出し、OSシステムコール。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cpus | object[] | CPU情報配列 |
| freemem | number | 空きメモリ（バイト） |
| totalmem | number | 総メモリ（バイト） |
| hostname | string | ホスト名 |
| platform | string | プラットフォーム名 |
| networkInterfaces | object | ネットワークIF情報 |

### 出力先

JavaScript呼び出し元への戻り値。

## 処理フロー

### 処理シーケンス

```
1. os.*メソッドの呼び出し
   └─ JavaScript引数の取得
2. プラットフォーム判定
   └─ Linux/macOS/Windowsで分岐
3. システムコール実行
   └─ プラットフォーム固有の情報取得
4. JSオブジェクトへの変換
   └─ 結果をJSValue形式に
5. 結果の返却
```

### フローチャート

```mermaid
flowchart TD
    A[os.* 呼び出し] --> B{プラットフォーム?}
    B -->|Linux| C[/proc読み取り等]
    B -->|macOS| D[sysctlbyname等]
    B -->|Windows| E[libuv経由]
    C --> F[JSValue変換]
    D --> F
    E --> F
    F --> G[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Node.js互換 | Node.js osモジュールのAPIと互換 | 全API |
| BR-02 | プラットフォーム固有 | OS毎に異なる実装 | Linux/macOS/Windows |
| BR-03 | 単位統一 | メモリはバイト単位 | freemem/totalmem |
| BR-04 | loadavg Windows | Windowsでは[0,0,0]を返す | loadavg |

### 計算ロジック

- CPUタイマー: tick単位からミリ秒への変換（multiplier = 1000 / SC_CLK_TCK）

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ESRCH | プロセス不在 | 存在しないPID指定 | PIDを確認 |
| EPERM | 権限エラー | setPriorityの権限不足 | 権限を確認 |
| ERR_SYSTEM_ERROR | システムエラー | OS情報取得失敗 | 環境を確認 |

### リトライ仕様

システムエラー時は自動リトライなし。

## トランザクション仕様

該当なし

## パフォーマンス要件

- スタックフォールバックでメモリ効率化
- /proc読み取りはファイルI/Oを伴う（Linux）

## セキュリティ考慮事項

- システム情報の漏洩に注意（外部公開時）
- setPriorityは権限昇格に注意

## 備考

os.EOL、os.constants等の定数も提供。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

node:os処理の中心となるデータ構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node_os.zig | `src/bun.js/node/node_os.zig` | CPUTimes構造体、createNodeOsBinding関数 |

**読解のコツ**:
- **1-17行目**: createNodeOsBinding関数でOS API全体をJSオブジェクトとしてエクスポート
- **19-34行目**: CPUTimes構造体（user, nice, sys, idle, irq）

#### Step 2: プラットフォーム固有実装を理解する

各OSでの情報取得方法を把握。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node_os.zig | `src/bun.js/node/node_os.zig` | cpusImplLinux、cpusImplDarwin、cpusImplWindows |

**主要処理フロー**:
1. **53-173行目**: cpusImplLinux関数（/proc/statと/proc/cpuinfoを読み取り）
2. **176-246行目**: cpusImplDarwin関数（host_processor_info使用）
3. **248-277行目**: cpusImplWindows関数（libuv使用）

#### Step 3: 各種API実装を理解する

主要なAPIの実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node_os.zig | `src/bun.js/node/node_os.zig` | freemem、homedir、hostname、loadavg等 |

**主要処理フロー**:
- **279-284行目**: freemem関数（C++側Bun__Os__getFreeMemory呼び出し）
- **304-379行目**: homedir関数（環境変数HOMEまたはgetpwuid_r）
- **382-405行目**: hostname関数（POSIX/Windows分岐）
- **407-450行目**: loadavg関数（macOS/Linux/Windows分岐）

### プログラム呼び出し階層図

```
os.cpus() (JavaScript)
    │
    └─ cpus() (Zig, lines 36-51)
           │
           └─ cpusImpl (プラットフォーム分岐)
                  │
                  ├─ Linux: cpusImplLinux() (lines 53-173)
                  │         ├─ /proc/stat読み取り
                  │         ├─ /proc/cpuinfo読み取り
                  │         └─ /sys/devices/system/cpu/*/cpufreq
                  │
                  ├─ macOS: cpusImplDarwin() (lines 176-246)
                  │         ├─ host_processor_info()
                  │         ├─ sysctlbyname("machdep.cpu.brand_string")
                  │         └─ sysctlbyname("hw.cpufrequency")
                  │
                  └─ Windows: cpusImplWindows() (lines 248-277)
                              └─ libuv.uv_cpu_info()
```

### データフロー図

```
[入力]               [処理]                    [出力]

(なし)          ───▶ /proc/stat           ───▶ CPU times
                     /proc/cpuinfo              CPU model
                     (Linux)

(なし)          ───▶ host_processor_info  ───▶ CPU times
                     sysctlbyname               CPU model/speed
                     (macOS)

(なし)          ───▶ uv_cpu_info          ───▶ CPU times/model/speed
                     (Windows/libuv)

CPU情報         ───▶ JSValue配列構築      ───▶ JSValue[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node_os.zig | `src/bun.js/node/node_os.zig` | ソース | メイン実装 |
| OsBinding.cpp | `src/bun.js/bindings/OsBinding.cpp` | ソース | C++側実装（freemem等） |
| os.ts | `src/js/node/os.ts` | ソース | JavaScript側実装 |
