# 機能設計書: node:http/https - HTTPクライアント/サーバー

## 1. 機能概要

### 1.1 機能名
node:http/https - HTTPクライアント/サーバー

### 1.2 機能ID
91

### 1.3 概要説明
BunにおけるNode.js互換のHTTP/HTTPSモジュール実装。HTTPサーバーの作成、HTTPクライアントリクエストの送信、IncomingMessage/ServerResponseオブジェクトの管理など、Node.jsのhttpモジュールと同等のAPIを提供する。内部ではBun独自の高性能HTTPサーバー実装と連携し、Node.jsアプリケーションとの互換性を維持しながら高速な処理を実現する。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでHTTPサーバーアプリケーションを実行
- API: `require("node:http")`, `require("node:https")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-91-1: HTTPサーバーの作成と起動
- **アクター**: 開発者
- **事前条件**: Bunランタイムがインストールされている
- **基本フロー**:
  1. `http.createServer()`でサーバーインスタンスを作成
  2. リクエストハンドラを設定
  3. `server.listen()`でサーバーを起動
  4. クライアントからのリクエストを受信・処理
- **事後条件**: HTTPサーバーが指定ポートでリッスン

#### UC-91-2: HTTPクライアントリクエストの送信
- **アクター**: 開発者
- **事前条件**: 対象サーバーがアクセス可能
- **基本フロー**:
  1. `http.request()`または`http.get()`でリクエストを作成
  2. リクエストボディを書き込み（POST等の場合）
  3. レスポンスを受信
  4. レスポンスデータを処理
- **事後条件**: レスポンスデータが取得される

### 2.2 機能仕様

#### 2.2.1 HTTPサーバー機能
| 項目 | 仕様 |
|------|------|
| サーバー作成 | `http.createServer([options][, requestListener])` |
| リッスン開始 | `server.listen(port[, host][, backlog][, callback])` |
| 接続終了 | `server.close([callback])` |
| タイムアウト設定 | `server.setTimeout([msecs][, callback])` |
| Keep-Alive | `server.keepAliveTimeout`, `server.headersTimeout` |

#### 2.2.2 HTTPクライアント機能
| 項目 | 仕様 |
|------|------|
| リクエスト作成 | `http.request(options[, callback])` |
| GETリクエスト | `http.get(options[, callback])` |
| エージェント | `http.Agent`クラスによる接続プール管理 |
| グローバルエージェント | `http.globalAgent` |

#### 2.2.3 IncomingMessage
| プロパティ/メソッド | 説明 |
|---------------------|------|
| `headers` | リクエスト/レスポンスヘッダーオブジェクト |
| `httpVersion` | HTTPプロトコルバージョン |
| `method` | HTTPメソッド（リクエスト時） |
| `url` | リクエストURL |
| `statusCode` | ステータスコード（レスポンス時） |

#### 2.2.4 ServerResponse
| プロパティ/メソッド | 説明 |
|---------------------|------|
| `writeHead(statusCode[, headers])` | レスポンスヘッダー送信 |
| `write(chunk)` | レスポンスボディ書き込み |
| `end([data])` | レスポンス終了 |
| `setHeader(name, value)` | ヘッダー設定 |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:http / node:https (src/js/node/http.ts)           ││
│  │  ┌───────────────┐ ┌───────────────┐ ┌───────────────┐  ││
│  │  │ createServer  │ │   request     │ │    Agent      │  ││
│  │  └───────────────┘ └───────────────┘ └───────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
│                            │                                 │
│  ┌─────────────────────────▼─────────────────────────────┐  │
│  │  Internal HTTP Implementation                          │  │
│  │  - IncomingMessage                                     │  │
│  │  - ServerResponse                                      │  │
│  │  - ClientRequest                                       │  │
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  Bun HTTP Server (src/bun.js/api/server.zig)            ││
│  │  - High-performance event loop                          ││
│  │  - Zero-copy buffer handling                            ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  HTTP Client (src/http/)                                ││
│  │  - Connection pooling                                   ││
│  │  - TLS support                                          ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
http
├── Server (extends net.Server)
│   ├── createServer() - サーバーインスタンス作成
│   ├── listen() - ポートバインド
│   └── close() - サーバー停止
├── ClientRequest (extends OutgoingMessage)
│   ├── request() - リクエスト作成
│   └── get() - GETリクエスト
├── IncomingMessage (extends Readable)
│   ├── headers - ヘッダーオブジェクト
│   ├── method - HTTPメソッド
│   └── url - リクエストURL
├── ServerResponse (extends Writable)
│   ├── writeHead() - ヘッダー送信
│   ├── write() - ボディ書き込み
│   └── end() - レスポンス終了
└── Agent
    ├── globalAgent - グローバルエージェント
    └── createConnection() - 接続作成
```

### 3.3 処理フロー

#### サーバー処理フロー
```
1. createServer()
   └── Server インスタンス作成
       └── イベントハンドラ登録

2. listen(port)
   └── Bun native server にバインド
       └── イベントループ開始

3. リクエスト受信
   ├── IncomingMessage 作成
   ├── ServerResponse 作成
   └── 'request' イベント発火

4. レスポンス送信
   ├── writeHead() - ヘッダー送信
   ├── write() - ボディ送信
   └── end() - 完了
```

#### クライアント処理フロー
```
1. request(options)
   └── ClientRequest インスタンス作成
       └── Agent から接続取得

2. write(data)
   └── リクエストボディ書き込み

3. end()
   └── リクエスト送信完了
       └── サーバーへ送信

4. レスポンス受信
   └── IncomingMessage 作成
       └── 'response' イベント発火
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **型定義・インターフェース** (`src/js/node/http.ts`)
   - **1-50行目**: モジュールインポートと定数定義
   - STATUS_CODESの定義とエクスポート

2. **エントリーポイント** (`src/js/node/http.ts`)
   - **createServer関数**: Serverインスタンスの作成
   - **request/get関数**: ClientRequestの作成

3. **Serverクラス** (`src/js/node/http.ts`)
   - listen()メソッド: ポートバインドとBunサーバー連携
   - close()メソッド: サーバー停止処理

4. **リクエスト/レスポンス処理**
   - IncomingMessage: リクエストデータのストリーム化
   - ServerResponse: レスポンス書き込み処理

### 4.2 プログラム呼び出し階層図

```
http.createServer(callback)
├── new Server()
│   ├── net.Server.call(this)
│   └── this.on('request', callback)
└── server.listen(port)
    └── [native] Bun.serve() 連携
        └── リクエスト受信時
            ├── new IncomingMessage(socket, request)
            ├── new ServerResponse(request)
            └── this.emit('request', req, res)

http.request(options, callback)
├── new ClientRequest(options)
│   ├── OutgoingMessage.call(this)
│   ├── Agent.createConnection()
│   └── this.on('response', callback)
└── req.end()
    └── [native] HTTP client 送信
        └── レスポンス受信時
            ├── new IncomingMessage(socket, response)
            └── this.emit('response', res)
```

### 4.3 データフロー図

```
[Client] ──HTTP Request──> [Server]
                              │
                              ▼
                    ┌─────────────────┐
                    │ IncomingMessage │
                    │ - headers       │
                    │ - method        │
                    │ - url           │
                    └────────┬────────┘
                              │
                    ┌─────────▼────────┐
                    │ Request Handler  │
                    │ (User Code)      │
                    └────────┬─────────┘
                              │
                    ┌─────────▼────────┐
                    │ ServerResponse   │
                    │ - writeHead()    │
                    │ - write()        │
                    │ - end()          │
                    └────────┬─────────┘
                              │
                              ▼
[Client] <──HTTP Response── [Server]
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/http.ts` | TypeScript | node:httpモジュールのメイン実装 |
| `src/js/node/https.ts` | TypeScript | node:httpsモジュール（httpを拡張） |
| `src/js/internal/http.ts` | TypeScript | 内部HTTP実装ユーティリティ |
| `src/bun.js/api/server.zig` | Zig | Bunネイティブサーバー実装 |
| `src/http/` | Zig/C++ | HTTPクライアント実装 |
| `src/js/node/net.ts` | TypeScript | TCP/ソケット基盤（Serverの親クラス） |
| `src/js/node/tls.ts` | TypeScript | TLS/SSL実装（httpsで使用） |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのhttp/httpsモジュールAPIと完全互換を目指す
- EventEmitterベースのイベントモデルを維持
- Streamインターフェースの互換性確保

### 5.2 パフォーマンス最適化
- Bunネイティブサーバーとの効率的な連携
- ゼロコピーバッファ処理の活用
- 接続プーリングによるリソース効率化

### 5.3 セキュリティ
- HTTPSでのTLS/SSL証明書検証
- ヘッダーインジェクション対策
- タイムアウト設定によるDoS対策

## 6. テスト方針

### 6.1 単体テスト
- Server.listen()/close()のライフサイクルテスト
- IncomingMessage/ServerResponseの各メソッドテスト
- エラーハンドリングテスト

### 6.2 統合テスト
- HTTPサーバー・クライアント間の通信テスト
- Keep-Alive接続のテスト
- 大容量データ転送テスト

### 6.3 互換性テスト
- Node.jsの既存テストスイートの実行
- 主要なNode.js HTTPフレームワーク（Express等）の動作確認

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
