# 機能設計書: node:http2 - HTTP/2プロトコル

## 1. 機能概要

### 1.1 機能名
node:http2 - HTTP/2プロトコル

### 1.2 機能ID
92

### 1.3 概要説明
BunにおけるNode.js互換のHTTP/2モジュール実装。HTTP/2プロトコルの多重化ストリーム、サーバープッシュ、ヘッダー圧縮（HPACK）などの機能を提供する。クライアントとサーバーの両方をサポートし、TLS/SSLによるセキュア通信（ALPN）にも対応。内部ではZigで実装されたH2FrameParserによる高性能なフレーム解析を行う。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでHTTP/2サーバーアプリケーションを実行
- API: `require("node:http2")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-92-1: HTTP/2セキュアサーバーの作成
- **アクター**: 開発者
- **事前条件**: TLS証明書と秘密鍵が用意されている
- **基本フロー**:
  1. `http2.createSecureServer(options)`でサーバー作成
  2. 'stream'イベントまたは'request'イベントでリクエスト処理
  3. `server.listen()`でサーバー起動
- **事後条件**: HTTP/2サーバーがALPNでリッスン

#### UC-92-2: HTTP/2クライアント接続
- **アクター**: 開発者
- **事前条件**: HTTP/2対応サーバーがアクセス可能
- **基本フロー**:
  1. `http2.connect(authority)`でセッション作成
  2. `session.request(headers)`でストリーム作成
  3. レスポンスストリームを処理
- **事後条件**: HTTP/2セッションが確立

#### UC-92-3: サーバープッシュの利用
- **アクター**: 開発者
- **事前条件**: HTTP/2サーバーが稼働中
- **基本フロー**:
  1. リクエストストリームで`pushStream()`呼び出し
  2. プッシュされたリソースを送信
- **事後条件**: クライアントにリソースがプッシュされる

### 2.2 機能仕様

#### 2.2.1 サーバー機能
| 項目 | 仕様 |
|------|------|
| セキュアサーバー作成 | `http2.createSecureServer([options][, onRequestHandler])` |
| 非セキュアサーバー | `http2.createServer([options][, onRequestHandler])` |
| 設定取得 | `http2.getDefaultSettings()`, `http2.getPackedSettings()` |
| 設定解凍 | `http2.getUnpackedSettings(buf)` |

#### 2.2.2 セッション管理
| 項目 | 仕様 |
|------|------|
| クライアント接続 | `http2.connect(authority[, options][, listener])` |
| 設定変更 | `session.settings(settings)` |
| セッション終了 | `session.close([callback])`, `session.destroy()` |
| GOAWAY送信 | `session.goaway([code][, lastStreamID][, opaqueData])` |

#### 2.2.3 ストリーム操作
| 項目 | 仕様 |
|------|------|
| リクエスト作成 | `session.request(headers[, options])` |
| レスポンス送信 | `stream.respond(headers[, options])` |
| サーバープッシュ | `stream.pushStream(headers, callback)` |
| 追加ヘッダー | `stream.additionalHeaders(headers)` |

#### 2.2.4 HTTP/2定数
| カテゴリ | 定数例 |
|---------|--------|
| 擬似ヘッダー | `:status`, `:method`, `:path`, `:authority`, `:scheme` |
| エラーコード | `NGHTTP2_NO_ERROR`, `NGHTTP2_PROTOCOL_ERROR`, etc. |
| フレームタイプ | DATA, HEADERS, PRIORITY, RST_STREAM, etc. |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:http2 (src/js/node/http2.ts)                      ││
│  │  ┌───────────────┐ ┌───────────────┐ ┌───────────────┐  ││
│  │  │ Http2Session  │ │ Http2Stream   │ │ Http2Server   │  ││
│  │  └───────────────┘ └───────────────┘ └───────────────┘  ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │ Http2ServerRequest / Http2ServerResponse          │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  H2FrameParser (src/bun.js/h2_frame_parser.zig)         ││
│  │  - フレーム解析                                          ││
│  │  - HPACK圧縮/解凍                                        ││
│  │  - 設定管理                                              ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  TLS/Socket Layer (node:tls, node:net)                  ││
│  │  - ALPN negotiation                                      ││
│  │  - Secure connection                                     ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
http2
├── Http2Session (extends EventEmitter)
│   ├── ClientHttp2Session
│   │   ├── request(headers) - ストリーム作成
│   │   └── connect() - 接続
│   └── ServerHttp2Session
│       └── origin() - オリジン設定
├── Http2Stream (extends Duplex)
│   ├── ClientHttp2Stream
│   │   └── レスポンス受信
│   └── ServerHttp2Stream
│       ├── respond(headers) - レスポンス送信
│       ├── pushStream() - サーバープッシュ
│       └── additionalHeaders() - 追加ヘッダー
├── Http2Server (extends net.Server)
│   └── createSecureServer()
├── Http2SecureServer (extends tls.Server)
│   └── createSecureServer()
├── Http2ServerRequest (extends Readable)
│   └── HTTP/1互換リクエストオブジェクト
└── Http2ServerResponse (extends Stream)
    └── HTTP/1互換レスポンスオブジェクト
```

### 3.3 処理フロー

#### サーバーセッション確立フロー
```
1. createSecureServer()
   └── TLSサーバー作成（ALPN: h2）

2. クライアント接続
   ├── TLSハンドシェイク
   ├── ALPN negotiation
   └── Http2Session作成

3. SETTINGS交換
   ├── サーバー SETTINGS送信
   └── クライアント SETTINGS ACK

4. ストリーム処理
   ├── HEADERS受信 → 'stream'イベント
   ├── DATA受信 → ストリームデータ
   └── respond() → HEADERS/DATA送信
```

#### クライアント接続フロー
```
1. http2.connect(authority)
   ├── TLS接続確立
   ├── ALPN negotiation (h2)
   └── ClientHttp2Session作成

2. SETTINGS交換
   ├── クライアント SETTINGS送信
   └── サーバー SETTINGS ACK受信

3. session.request(headers)
   ├── Http2Stream作成
   ├── HEADERS フレーム送信
   └── 'response'イベント待機

4. レスポンス処理
   ├── HEADERS受信 → 'response'イベント
   ├── DATA受信 → ストリームデータ
   └── END_STREAM → ストリーム終了
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **定数・シンボル定義** (`src/js/node/http2.ts`)
   - **29-45行目**: 内部シンボル定義（kInfoHeaders, kProxySocket等）
   - **77-106行目**: 状態管理用シンボル（kState, kStream, kHeaders等）

2. **ネイティブ連携** (`src/js/node/http2.ts`)
   - **72-75行目**: H2FrameParser、getPackedSettings、getUnpackedSettingsのZig関数インポート
   ```typescript
   const H2FrameParser = $zig("h2_frame_parser.zig", "H2FrameParserConstructor");
   const assertSettings = $newZigFunction("h2_frame_parser.zig", "jsAssertSettings", 1);
   ```

3. **Http2ServerRequest クラス** (`src/js/node/http2.ts`)
   - **329-457行目**: HTTP/1互換のリクエストオブジェクト
   - Readableストリームを継承
   - headers, method, url等のプロパティ

4. **Http2ServerResponse クラス** (`src/js/node/http2.ts`)
   - **458-892行目**: HTTP/1互換のレスポンスオブジェクト
   - writeHead(), write(), end()メソッド
   - setHeader(), getHeader()等のヘッダー操作

5. **ユーティリティ関数**
   - **96-135行目**: emitError, emitEventNT等のイベント発火関数
   - **263-313行目**: assertValidHeader, isPseudoHeader等のバリデーション

### 4.2 プログラム呼び出し階層図

```
http2.createSecureServer(options, onRequest)
├── new Http2SecureServer()
│   ├── tls.createServer()
│   └── session 管理設定
└── server.on('stream', handler)
    └── onServerStream()
        ├── new Http2ServerRequest(stream, headers)
        │   ├── Readable.call(this)
        │   └── stream イベントリスナー設定
        ├── new Http2ServerResponse(stream)
        │   ├── Stream.call(this)
        │   └── [kHeaders], [kTrailers] 初期化
        └── server.emit('request', request, response)

http2.connect(authority, options)
├── TLS/TCP 接続確立
├── new ClientHttp2Session()
│   └── H2FrameParser 初期化
└── session.request(headers)
    ├── new ClientHttp2Stream()
    ├── HEADERS フレーム送信
    └── 'response' イベント待機
```

### 4.3 データフロー図

```
[Client]                                [Server]
    │                                       │
    │──── SETTINGS ────────────────────────>│
    │<─── SETTINGS ─────────────────────────│
    │<─── SETTINGS ACK ─────────────────────│
    │──── SETTINGS ACK ────────────────────>│
    │                                       │
    │──── HEADERS (stream 1) ──────────────>│
    │                                       │
    │     ┌─────────────────────────────┐   │
    │     │ Http2ServerRequest          │   │
    │     │ - headers                   │   │
    │     │ - method, url               │   │
    │     └─────────────────────────────┘   │
    │                                       │
    │<─── HEADERS (stream 1, :status) ─────│
    │<─── DATA (stream 1) ─────────────────│
    │<─── DATA (END_STREAM) ───────────────│
    │                                       │
    │     ┌─────────────────────────────┐   │
    │     │ Http2ServerResponse         │   │
    │     │ - respond()                 │   │
    │     │ - write(), end()            │   │
    │     └─────────────────────────────┘   │
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/http2.ts` | TypeScript | node:http2モジュールのメイン実装 |
| `src/bun.js/h2_frame_parser.zig` | Zig | HTTP/2フレーム解析エンジン |
| `src/js/node/tls.ts` | TypeScript | TLS/SSL実装（ALPN対応） |
| `src/js/node/net.ts` | TypeScript | TCP接続基盤 |
| `src/js/internal/http.ts` | TypeScript | STATUS_CODES等の共通定義 |
| `src/js/internal/validators.ts` | TypeScript | 入力検証ユーティリティ |
| `src/js/internal/primordials.ts` | TypeScript | SafeSet, SafeArrayIterator等 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.js http2 APIとの完全互換を目指す
- Http2ServerRequest/Responseでhttp1互換レイヤーを提供
- 擬似ヘッダー（:status, :method等）の適切な処理

### 5.2 パフォーマンス最適化
- Zigで実装されたH2FrameParserによる高速フレーム解析
- HPACKヘッダー圧縮の効率的な実装
- 多重化ストリームによる並列処理

### 5.3 セキュリティ
- TLS 1.2以上でのALPN negotiation
- 不正なフレームシーケンスの検出
- ストリーム数制限によるDoS対策

### 5.4 エラーハンドリング
- GOAWAY フレームによる graceful shutdown
- RST_STREAM によるストリーム単位のエラー通知
- 適切なエラーコードの使用（PROTOCOL_ERROR等）

## 6. テスト方針

### 6.1 単体テスト
- フレーム解析のテスト（HEADERS, DATA, SETTINGS等）
- HPACK圧縮/解凍のテスト
- ストリーム状態遷移のテスト

### 6.2 統合テスト
- クライアント・サーバー間の通信テスト
- サーバープッシュのテスト
- 複数ストリームの並列処理テスト

### 6.3 互換性テスト
- Node.jsのhttp2テストスイートの実行
- 主要なHTTP/2クライアント/サーバーとの相互運用テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
