# 機能設計書: node:net - TCPネットワーキング

## 1. 機能概要

### 1.1 機能名
node:net - TCPネットワーキング

### 1.2 機能ID
93

### 1.3 概要説明
BunにおけるNode.js互換のnetモジュール実装。TCP/IPソケット通信のためのサーバーとクライアントを提供し、低レベルネットワークプログラミングを可能にする。ストリームベースの非同期I/Oを採用し、IPCパイプやUnixドメインソケットもサポートする。内部ではBunのネイティブソケット実装と連携して高性能な通信を実現する。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでTCPサーバー/クライアントアプリケーションを実行
- API: `require("node:net")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-93-1: TCPサーバーの作成と接続受付
- **アクター**: 開発者
- **事前条件**: Bunランタイムがインストールされている
- **基本フロー**:
  1. `net.createServer()`でサーバー作成
  2. 'connection'イベントでクライアント接続を処理
  3. `server.listen()`でポートをバインド
  4. クライアントからのデータを受信・送信
- **事後条件**: TCPサーバーが稼働

#### UC-93-2: TCPクライアント接続
- **アクター**: 開発者
- **事前条件**: 対象サーバーがアクセス可能
- **基本フロー**:
  1. `net.createConnection()`または`net.connect()`で接続
  2. 'connect'イベントで接続確立を確認
  3. `socket.write()`でデータ送信
  4. 'data'イベントでデータ受信
- **事後条件**: TCP接続が確立

#### UC-93-3: Unixドメインソケット通信
- **アクター**: 開発者
- **事前条件**: Unixシステム
- **基本フロー**:
  1. `net.createServer()`でサーバー作成
  2. `server.listen(path)`でソケットファイルにバインド
  3. `net.createConnection(path)`で接続
- **事後条件**: IPC通信が確立

### 2.2 機能仕様

#### 2.2.1 net.Server
| 項目 | 仕様 |
|------|------|
| サーバー作成 | `net.createServer([options][, connectionListener])` |
| リッスン開始 | `server.listen(port[, host][, backlog][, callback])` |
| リッスン終了 | `server.close([callback])` |
| アドレス取得 | `server.address()` |
| 最大接続数 | `server.maxConnections` |

#### 2.2.2 net.Socket
| 項目 | 仕様 |
|------|------|
| 接続作成 | `net.createConnection(options[, connectListener])` |
| データ送信 | `socket.write(data[, encoding][, callback])` |
| 接続終了 | `socket.end([data][, encoding][, callback])` |
| 強制切断 | `socket.destroy([error])` |
| タイムアウト | `socket.setTimeout(timeout[, callback])` |
| Keep-Alive | `socket.setKeepAlive([enable][, initialDelay])` |
| Nagle無効化 | `socket.setNoDelay([noDelay])` |

#### 2.2.3 Socketプロパティ
| プロパティ | 説明 |
|-----------|------|
| `localAddress` | ローカルIPアドレス |
| `localPort` | ローカルポート |
| `remoteAddress` | リモートIPアドレス |
| `remotePort` | リモートポート |
| `remoteFamily` | IPバージョン（IPv4/IPv6） |
| `bytesRead` | 受信バイト数 |
| `bytesWritten` | 送信バイト数 |

#### 2.2.4 ユーティリティ関数
| 関数 | 説明 |
|------|------|
| `net.isIP(input)` | IPアドレス判定（0, 4, 6を返す） |
| `net.isIPv4(input)` | IPv4判定 |
| `net.isIPv6(input)` | IPv6判定 |
| `net.getDefaultAutoSelectFamily()` | デフォルトのAutoSelectFamily取得 |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:net (src/js/node/net.ts)                          ││
│  │  ┌───────────────┐ ┌───────────────┐ ┌───────────────┐  ││
│  │  │   Server      │ │    Socket     │ │  BlockList    │  ││
│  │  └───────────────┘ └───────────────┘ └───────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  Bun Socket Implementation                               ││
│  │  - TCP Socket                                            ││
│  │  - Unix Domain Socket                                    ││
│  │  - IPC Pipe                                              ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  Event Loop Integration                                  ││
│  │  - Async I/O                                             ││
│  │  - Connection pooling                                    ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
net
├── Server (extends EventEmitter)
│   ├── createServer() - サーバーインスタンス作成
│   ├── listen() - ポート/パスバインド
│   ├── close() - サーバー停止
│   ├── address() - バインドアドレス取得
│   └── getConnections() - 接続数取得
├── Socket (extends Duplex)
│   ├── createConnection() - クライアント接続
│   ├── connect() - 接続実行
│   ├── write() - データ送信
│   ├── end() - 接続終了
│   ├── destroy() - 強制切断
│   ├── pause()/resume() - フロー制御
│   └── setEncoding() - エンコーディング設定
├── BlockList
│   ├── addAddress() - IPアドレス追加
│   ├── addRange() - IP範囲追加
│   ├── addSubnet() - サブネット追加
│   └── check() - ブロック判定
└── SocketAddress
    └── IPアドレス/ポート情報
```

### 3.3 処理フロー

#### サーバー処理フロー
```
1. net.createServer(options, connectionListener)
   └── new Server()
       ├── EventEmitter.call(this)
       └── allowHalfOpen, pauseOnConnect設定

2. server.listen(port, host)
   └── [native] ソケットバインド
       └── 'listening' イベント発火

3. クライアント接続時
   ├── new Socket() 作成
   ├── ソケットオプション設定
   └── 'connection' イベント発火

4. データ受信時
   └── socket 'data' イベント発火

5. 接続終了時
   ├── socket 'end' イベント発火
   └── socket 'close' イベント発火
```

#### クライアント処理フロー
```
1. net.createConnection(options)
   └── new Socket()
       └── socket.connect(options)

2. 接続確立時
   └── 'connect' イベント発火

3. データ送受信
   ├── socket.write(data) - 送信
   └── 'data' イベント - 受信

4. 接続終了
   ├── socket.end() - 正常終了
   └── socket.destroy() - 強制切断
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **モジュールインポートと定数** (`src/js/node/net.ts`)
   - EventEmitter, Duplexストリームのインポート
   - bunSocketInternal等のBunネイティブ連携シンボル

2. **Socket クラス** (`src/js/node/net.ts`)
   - Duplexストリームを継承したソケット実装
   - connect(), write(), end()等のメソッド
   - 各種イベント（connect, data, end, error, close）

3. **Server クラス** (`src/js/node/net.ts`)
   - EventEmitterを継承したサーバー実装
   - listen(), close(), address()等のメソッド
   - connection イベントの発火

4. **ユーティリティ関数**
   - isIP(), isIPv4(), isIPv6()
   - BlockListクラスの実装

### 4.2 プログラム呼び出し階層図

```
net.createServer([options], connectionListener)
├── new Server(options)
│   ├── EventEmitter.call(this)
│   └── this.on('connection', connectionListener)
└── server.listen(port, host, callback)
    ├── [native] Bun.listen() 連携
    │   └── ソケットバインド
    └── 接続受付時
        ├── new Socket(connectionOptions)
        │   ├── Duplex.call(this)
        │   └── ネイティブソケット関連付け
        └── this.emit('connection', socket)

net.createConnection(options, callback)
├── new Socket(options)
│   └── Duplex.call(this)
└── socket.connect(options, callback)
    ├── [native] TCP接続開始
    └── 接続完了時
        └── this.emit('connect')
```

### 4.3 データフロー図

```
[TCP Client]                           [TCP Server]
     │                                       │
     │ ─────── SYN ─────────────────────────>│
     │ <────── SYN-ACK ─────────────────────│
     │ ─────── ACK ─────────────────────────>│
     │                                       │
     │        ┌─────────────────────┐        │
     │        │ 'connect' event     │        │
     │        │ 'connection' event  │        │
     │        └─────────────────────┘        │
     │                                       │
     │ ─────── DATA ────────────────────────>│
     │        ┌─────────────────────┐        │
     │        │ socket.write()      │        │
     │        │ 'data' event        │        │
     │        └─────────────────────┘        │
     │                                       │
     │ <────── DATA ─────────────────────────│
     │        ┌─────────────────────┐        │
     │        │ 'data' event        │        │
     │        │ socket.write()      │        │
     │        └─────────────────────┘        │
     │                                       │
     │ ─────── FIN ─────────────────────────>│
     │        ┌─────────────────────┐        │
     │        │ socket.end()        │        │
     │        │ 'end' event         │        │
     │        └─────────────────────┘        │
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/net.ts` | TypeScript | node:netモジュールのメイン実装 |
| `src/js/node/tls.ts` | TypeScript | TLS拡張（net.Serverを継承） |
| `src/js/internal/streams/duplex.ts` | TypeScript | Duplexストリーム基底クラス |
| `src/js/node/events.ts` | TypeScript | EventEmitter基底クラス |
| `src/bun.js/api/` | Zig | Bunネイティブソケット実装 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのnet APIと完全互換を目指す
- EventEmitterベースのイベントモデルを維持
- Duplexストリームインターフェースの互換性確保

### 5.2 パフォーマンス最適化
- Bunネイティブソケットとの効率的な連携
- 非同期I/Oによる高スループット
- TCP_NODELAYオプションによる低遅延

### 5.3 プラットフォーム対応
- TCPソケット（IPv4/IPv6）
- Unixドメインソケット（Unix系OS）
- 名前付きパイプ（Windows）

### 5.4 エラーハンドリング
- 接続エラー（ECONNREFUSED, ETIMEDOUT等）
- 送受信エラー（EPIPE, ECONNRESET等）
- アドレスエラー（EADDRINUSE, EADDRNOTAVAIL等）

## 6. テスト方針

### 6.1 単体テスト
- Socket.connect()/end()/destroy()のテスト
- Server.listen()/close()のテスト
- isIP(), isIPv4(), isIPv6()のテスト

### 6.2 統合テスト
- TCP通信の双方向データ送受信テスト
- 大容量データ転送テスト
- 複数クライアント同時接続テスト

### 6.3 互換性テスト
- Node.jsのnetテストスイートの実行
- 各種ネットワークライブラリとの相互運用テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
